/*
 * Copyright (C) 2003 Klaus Frahm <frahm@irsamc.ups-tlse.fr>
 * Quantware MIPS Center, Laboratoire de Physique Theorique
 * University Paul Sabatier, Toulouse III
 * 118, route de Narbonne, 31062 Toulouse Cedex 4 - FRANCE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public Licens
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-
 *
 */

/* realisation of the modified kicked rotator by quantum-computer 
   operations, version where all elementary operations are perturbed 
   by static error

*/

#include "complex.h"
#include "random.h"
#include "quantum.h"
#include <time.h>

#define GLOBAL_FILE "all_stat_fid.dat"

#define SAVE_SECONDS 60  // number of seconds between saving of quantum states


/***********************************/
/* fill an array of complex numbers with random phases */

void fill(int n,complex *f){
  int i;

  for(i=0;i<n;i++){
      f[i]=complex(random_rectangular_old(1.0),random_rectangular_old(1.0));
  }
}

/***********************************/
/* Quantum code of the unitary operator \exp(-i*p^2*T/2) */

void U_p(double T,qubit_state &a){
    int j,k,L;
    double x;
    complex *phases;

    L=a.L();
    phases=new complex[2*L+5];
    x=-T/2.0;
    for(j=0;j<2*L+2;j++){
	phases[j]=complex(cos(x),sin(x));
	x*=2.0;
    }
    for(j=0;j<L;j++) a.B1_gate(j,phases[2*j]);
    for(k=0;k<L;k++) for(j=0;j<k;j++) a.B2_gate(j,k,phases[j+k+1]);
    delete[] phases;
}

/***********************************/
/* Quantum code of the unitary operator \exp(-i*p^2*T/2) with static errors */

void U_p_stat(double T,qubit_state &a,int n){
    int j,k,L;
    double x;
    complex *phases;

    L=a.L();
    phases=new complex[2*L+5];
    x=-T/2.0;
    for(j=0;j<2*L+2;j++){
	phases[j]=complex(cos(x),sin(x));
	x*=2.0;
    }
    for(j=0;j<L;j++){
	a.B1_gate(j,phases[2*j]);
	a.stat_error_simple(1.0,n);
    }
    for(k=0;k<L;k++) for(j=0;j<k;j++){
	a.B2_gate(j,k,phases[j+k+1]);
	a.stat_error_simple(1.0,n);
    }
    delete[] phases;
}

/***********************************/
/* Quantum code of the unitary operator \exp(-i*p^2*T/2) 
   test version with naive classical calculation */

void U_p_test(double T,qubit_state &a){
    int l,N;
    double x;
    complex *phases;

    N=a.N();
    phases=new complex[N+5];
    for(l=0;l<N;l++){
	x=-T*(double)l*(double)l/2.0;
	phases[l]=complex(cos(x),sin(x));
    }
    a.phase_mult(phases);
    delete[] phases;
}

/***********************************/
/* Quantum code of the unitary operator \exp(-i*V(\theta)) */

void U_theta(double K,qubit_state &a){
    int j,k,L,L1;
    double x;
    complex *phases,e1,e2;

    L=a.L(); L1=L-1;
    phases=new complex[2*L+7];
    x=-K*M_PI*M_PI/2.0;
    for(j=0;j<L;j++) x/=4.0;
    for(j=0;j<2*L+4;j++){
	phases[j]=complex(cos(x),sin(x));
	x*=2.0;
    }
    for(k=0;k<L1;k++) for(j=0;j<k;j++){
	//	a.B3_gate_cheat(j,k,L1,phases[j+k+4]);
       	a.B3_gate(j,k,L1,phases[j+k+3]);
	a.B2_gate(j,k,conj(phases[j+k+3]));
    }
    for(j=0;j<L1;j++){
	a.B2_gate(j,L1,phases[2*j+3]*conj(phases[j+L+2]));
	a.B1_gate(j,phases[j+1+L]*conj(phases[2*j+2]));
    }
    delete[] phases;
}

/***********************************/
/* Quantum code of the unitary operator \exp(-i*V(\theta)) with static errors */

void U_theta_stat(double K,qubit_state &a,int n){
    int j,k,L,L1;
    double x;
    complex *phases,e1,e2;

    L=a.L(); L1=L-1;
    phases=new complex[2*L+7];
    x=-K*M_PI*M_PI/2.0;
    for(j=0;j<L;j++) x/=4.0;
    for(j=0;j<2*L+4;j++){
	phases[j]=complex(cos(x),sin(x));
	x*=2.0;
    }
    for(k=0;k<L1;k++) for(j=0;j<k;j++){
	//     	a.B3_gate(j,k,L1,phases[j+k+3]);
       	a.B3_gate_stat(j,k,L1,phases[j+k+3],n);
	a.B2_gate(j,k,conj(phases[j+k+3]));
	a.stat_error_simple(1.0,n);
    }
    for(j=0;j<L1;j++){
	a.B2_gate(j,L1,phases[2*j+3]*conj(phases[j+L+2]));
	a.stat_error_simple(1.0,n);
	a.B1_gate(j,phases[j+1+L]*conj(phases[2*j+2]));
	a.stat_error_simple(1.0,n);
    }
    delete[] phases;
}

/***********************************/
/* Quantum code of the unitary operator \exp[-i*V(\theta)] 
   test version with naive classical calculation */

void U_theta_test(double K,qubit_state &a){
    int l,N,N2;
    double theta,x,dtheta;
    complex *phases;

    N=a.N(); N2=N/2;
    phases=new complex[N+5];
    dtheta=2.0*M_PI/(double)N;
    for(l=0;l<N2;l++){
	theta=dtheta*(double)l;
	x=-K/2.0*theta*(theta-M_PI);
	phases[l]=complex(cos(x),sin(-x));
    }
    for(l=0;l<N2;l++){
	theta=dtheta*(double)l;
	x=K/2.0*theta*(theta-M_PI);
	phases[l+N2]=complex(cos(x),sin(-x));
    }
    a.phase_mult(phases);
    delete[] phases;
}

/***********************************/
/* quantum code for a full application of the quantum tent map operator */

void kick(double K,double T,qubit_state &a){
    a.QFT(0);
    U_theta(K,a);
    a.QFT(1);
    U_p(T,a);
}

/***********************************/
/* quantum code for a full application of 
   the inverse quantum tent map operator */

void kick_inv(double K,double T,qubit_state &a){
    U_p(-T,a);
    a.QFT(0);
    U_theta(-K,a);
    a.QFT(1);
}

/***********************************/
/* quantum code for a full application of the quantum tent map operator 
   naive test version */

void kick_test(double K,double T,qubit_state &a){
    a.QFT(0);
    U_theta_test(K,a);
    a.QFT(1);
    U_p_test(T,a);
}

/***********************************/
/* quantum code for a full application of the quantum tent map operator 
   with static errors */

void kick_stat(double K,double T,qubit_state &a,int n){
    a.QFT_stat(0,n);
    U_theta_stat(K,a,n);
    a.QFT_stat(1,n);
    U_p_stat(T,a,n);
}

/***********************************/
/* quantum code for a full application of the 
   inverse quantum tent map operator with static erros */

void kick_stat_inv(double K,double T,qubit_state &a,int n){
    U_p_stat(-T,a,n);
    a.QFT_stat(0,n);
    U_theta_stat(-K,a,n);
    a.QFT_stat(1,n);
}

/***********************************/
/* create a state that has circle form in the Husimi representation */

void put_cercle(double r,qubit_state &a){
    double p0,q0,sigma2,x,dx,N2;
    int i,num;

    a.put_to_zero();
    N2=(double)a.N()/2.0;
    sigma2=N2/6.0;
    num=(int)(sqrt((double)a.N())*M_PI*r);
    //    num=EXP2(a.L()/2+3);
    //    num=EXP2(a.L()/2);
    dx=2.0*M_PI/(double)num;
    for(i=0;i<num;i++){
	x=dx*(double)i;
	p0=N2*(1.0+r*sin(x));
	q0=M_PI*(1.0+r*cos(x));
	a.gauss_add(p0,q0,sigma2);
    }
    a.normalize();
}

/***********************************/
/* generate an inital state which is a coherent state at the classical 
   postion q=\pi/2 and p=1 and p-width of \sqrt{N/12} */

void put_point(qubit_state &a){
    double p0,q0,sigma2;
    int i,num;

    a.put_to_zero();
    sigma2=(double)a.N()/12.0;
    p0=1.0;
    q0=M_PI/2.0;
    a.gauss_add(p0,q0,sigma2);
    a.normalize();
}

/***********************************/
/* a logical function that returns true if a certain number of seconds 
   has ellapsed since the last call of this function. This function is 
   used to determine if the calculation data must be saved in files. */

int save_time(int seconds){
  static time_t former=0;
  time_t now,diff;

  now=clock();
  if(former==0){
    former=now;
    return 1;
  }
  diff=now-former;
  if(diff>=seconds*CLOCKS_PER_SEC){
    former=now;
    return 1;
  }else{
    return 0;
  }
}


/***********************************/
/* main function */

int main(int argc,char **argv){
    int L,num,steps1,steps2,i,N,n_default,init_time,Lx;
    double K_cl,K,fsize,T,eps1,eps2;
    double fid,lfid,sat_val;
    char *buff,*filename,*save0,*save1,*save2;
    FILE *fp;
    
    
    if(argc<8) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");
    if(sscanf(argv[1],"%lf",&K_cl)==0) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");
    if(sscanf(argv[2],"%d",&L)==0) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");
    if(sscanf(argv[3],"%lf",&eps1)==0) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");
    if(sscanf(argv[4],"%lf",&eps2)==0) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");
    if(sscanf(argv[5],"%d",&num)==0) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");
    if(sscanf(argv[6],"%d",&steps1)==0) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");
    if(sscanf(argv[7],"%d",&steps2)==0) 
	fehler("syntax: K n_q eps1 eps2 print_step steps1 steps2");

    buff=new char[200];
    filename=new char[200];
    save0=new char[200];
    save1=new char[200];
    save2=new char[200];



    n_default=1;     // default value for n in the static error 
    if(L<=1 || L>MAX_QUBITS) L=DEFAULT_NBQUBITS;
    N=EXP2(L);
    //    sat_val=4/(double)N;
    sat_val=0.5;

    qubit_state a_clean(L);
    a_clean.draw_couplings(eps1,eps2,1.0,Lx);

    //    put_cercle(0.7,a_clean);
    put_point(a_clean);
    //    qubit_state a_stat(a_clean),b(a_clean);
    qubit_state a_stat(a_clean);
    T=2.0*M_PI/(double)EXP2(L);
    K=K_cl/T;
    
    if(num>0){
	sprintf(buff,"quant_clean_%d_%4.2lf_%d_%3.1lf.raw",L,K_cl,0,0.0);
	a_clean.save_state(buff);

	sprintf(buff,"quant_stat_%d_%4.2lf_%d_%7.5lf.raw",L,K_cl,0,eps1);
	a_stat.save_state(buff);
    }

    sprintf(filename,"stat_fid_%d_%lg_%lg_%lg.dat",L,eps1,eps2,K_cl);

    fp=fopen(GLOBAL_FILE,"a");
    fprintf(fp,"### new data:   L=%d   eps1=%lg   eps2=%lg   K_cl=%lg\n",
	    L,eps1,eps2,K_cl);
    fprintf(fp,"%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
	    0,1.0,0.0,L,eps1,eps2,K_cl);
    fclose(fp);

    sprintf(save0,"time_%d_%lg_%lg_%lg.dat",L,eps1,eps2,K_cl);
    sprintf(save1,"vector_clean_%d_%lg_%lg_%lg.raw",L,eps1,eps2,K_cl);
    sprintf(save2,"vector_stat_%d_%lg_%lg_%lg.raw",L,eps1,eps2,K_cl);

    // reading of init-files if they exist!!
    fp=fopen(save0,"r");
    if(fp==NULL){
      init_time=0;
      fp=fopen(filename,"w");
      fprintf(fp,"%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
	      0,1.0,0.0,L,eps1,eps2,K_cl);
      fclose(fp);
      printf("Starting with new calculation at t=0.\n");
      fflush(stdout);
    }else{
      fscanf(fp,"%d",&init_time);
      fclose(fp);
      a_clean.load_state(save1);
      a_stat.load_state(save2);
      printf("Recovering from previous calculation with t=%d.\n",init_time);
      fflush(stdout);
      fid=a_clean.fidelity(a_stat);
      if(fid<sat_val){
	printf("No further calculation due to small value!\n");
	return 0;
      }
    }

    if(num==0) num=steps1+steps2+10;
    for(i=1+init_time;i<=steps1;i++){
	kick(K,T,a_clean);
	kick_stat(K,T,a_stat,n_default);

	fid=a_clean.fidelity(a_stat);

	// saving of quantum states in case of interuption 
	if(save_time(SAVE_SECONDS) || i==steps1 || fid<sat_val){
	  fp=fopen(save0,"w");
	  fprintf(fp,"%5d\n",i);
	  fclose(fp);
	  fp=fopen("save_times.dat","a");
	  fprintf(fp,"%5d\n",i);
	  fclose(fp);

	  a_clean.save_state(save1);
	  a_stat.save_state(save2);
	}

	lfid=log(fid);
	fp=fopen(filename,"a");
	fprintf(fp,"%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
		i,fid,lfid,L,eps1,eps2,K_cl);
	fclose(fp);
	fp=fopen(GLOBAL_FILE,"a");
	fprintf(fp,"%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
		i,fid,lfid,L,eps1,eps2,K_cl);
	fclose(fp);
	printf("%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
		i,fid,lfid,L,eps1,eps2,K_cl);
	fflush(stdout);

	if(i%num==0){
	    sprintf(buff,"quant_clean_%d_%4.2lf_%d_%3.1lf.raw",L,K_cl,i,0.0);
	    a_clean.save_state(buff);
	  
	    sprintf(buff,"quant_stat_%d_%4.2lf_%d_%7.5lf.raw",L,K_cl,i,eps1);
	    a_stat.save_state(buff);
	} 
	if(fid<sat_val){
	    printf("Stop due to small value!\n");
	    steps2=0;
	    break;
	}

    }
    if(steps2){
	printf("\nBack calculation:\n");
	fflush(stdout);
    }
    if(init_time>steps1){
      init_time-=steps1;
    }else{
      init_time=0;
    }
    for(i=1+init_time;i<=steps2;i++){
	kick_inv(K,T,a_clean);
	kick_stat_inv(K,T,a_stat,n_default);

	fid=a_clean.fidelity(a_stat);

	// saving of quantum states in case of interuption 
	if(save_time(SAVE_SECONDS) || i==steps2 || fid<sat_val){
	  fp=fopen(save0,"w");
	  fprintf(fp,"%5d\n",i+steps1);
	  fclose(fp);
	  fp=fopen("save_times.dat","a");
	  fprintf(fp,"%5d\n",i+steps1);
	  fclose(fp);

	  a_clean.save_state(save1);
	  a_stat.save_state(save2);
	}

	lfid=log(fid);
	fp=fopen(filename,"a");
	fprintf(fp,"%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
		i+steps1,fid,lfid,L,eps1,eps2,K_cl);
	fclose(fp);
	fp=fopen(GLOBAL_FILE,"a");
	fprintf(fp,"%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
		i+steps1,fid,lfid,L,eps1,eps2,K_cl);
	fclose(fp);
	printf("%5d  %16.8lf  %16.8lg  %5d %lg %lg %lg\n",
		i+steps1,fid,lfid,L,eps1,eps2,K_cl);
	fflush(stdout);
	if((i+steps1)%num==0){
	    sprintf(buff,"quant_clean_back_%d_%4.2lf_%d_%3.1lf.raw",
		    L,K_cl,steps1-i,0.0);
	    a_clean.save_state(buff);
	  
	    sprintf(buff,"quant_stat_back_%d_%4.2lf_%d_%7.5lf.raw",
		    L,K_cl,steps1-i,eps1);
	    a_stat.save_state(buff);
	} 
	if(fid<sat_val){
	    printf("Stop due to small value!\n");
	    steps2=0;
	    break;
	}

	if(fid<sat_val){
	    printf("Stop due to small value!\n");
	    break;
	}

    }
    fp=fopen(GLOBAL_FILE,"a");
    fprintf(fp,"\n");
    fclose(fp);
    
    delete[] save2;
    delete[] save1;
    delete[] save0;
    delete[] filename;
    delete[] buff;
    return 0;
}
