//
//  Quantum Gates Algebra 
//
//  these objects provide an interface to gate routines, 
//  which are actually implemented in the file 'qubits.h'  
//
//  last modified 19/07/2004
//
//
//

#ifndef _Q_ALGEBRA_H_
#define _Q_ALGEBRA_H_


#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <complex>
#include "qubits.h"
#include "qLattice.h"
using std::complex;

class Gate    // -------- generic gate
{ 
 public:
 int nq1,nq2;
 
 Gate(int j1,int j2) { nq1=j1; nq2=j2;}
 ~Gate(){};
}; 
 

class WH : public Gate    //-------- Walsh-Hadamard gate at qubit(i)
{ public:
  WH(int j1, int j2=-1):Gate(j1,j2) {};
};

class X : public Gate     // Sigma X
{ public:
  X(int j1, int j2=-1):Gate(j1,j2) {};
};

class Y : public Gate     // Sigma Y
{ public:
  Y(int j1, int j2=-1):Gate(j1,j2) {};
};

class Z : public Gate     // Sigma Z
{ public:
  Z(int j1, int j2=-1):Gate(j1,j2) {};
};

class rotQBitBy : public Gate   // rotate a single qubit by angle A
{ public:
  double Angle;
  rotQBitBy(double A,int j1, int j2=-1):Gate(j1,j2){ Angle=A;};
};

class Cnot                      // Cnot gate
{ public:
  int ic,iq;
  Cnot(int iC, int iQ) { ic=iC; iq=iQ;};
};

class CCnot                     // CCnot gate
{ public:
  int ic1,ic2,iq;
  CCnot(int iC1, int iC2, int iQ) { ic1=iC1; ic2=iC2; iq=iQ;};
};

static int PertCount=0;

class Pert                      // Perturbation by static imperfections
{ public:
  Lattice* L;
  Pert(Lattice* Lat) { L=Lat; PertCount++;};  
};

class PertRandMap               // Perturbation by static imperfections with random permutation of qubits
{ public:
  Lattice* L;
  PertRandMap(Lattice* Lat) { L=Lat; PertCount++;};  
};

class PertRand                  // Perturbation by time-dependent imperfections 
{ public:
  Lattice* L;
  PertRand(Lattice* Lat) { L=Lat;  PertCount++;};  
};


QBitsWaveFunction&  operator << (QBitsWaveFunction& wf, WH G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) wf.WH_tr(i);
   return wf; 
   
}

QBitsWaveFunction& operator << (QBitsWaveFunction& wf, X G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) wf.SigmaX(i); return wf; }

QBitsWaveFunction& operator << (QBitsWaveFunction& wf, Y G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) wf.SigmaY(i); return wf; }

QBitsWaveFunction& operator << (QBitsWaveFunction& wf, Z G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) wf.SigmaZ(i); return wf; }

QBitsWaveFunction& operator << (QBitsWaveFunction& wf, rotQBitBy G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) wf.RotateQBit(i,G.Angle); return wf; }

QBitsWaveFunction& operator << (QBitsWaveFunction& wf, Pert G) { G.L->Perturb(wf); return wf;}
QBitsWaveFunction& operator << (QBitsWaveFunction& wf, PertRand G) { G.L->Rand(); G.L->Perturb(wf); return wf;}
QBitsWaveFunction& operator << (QBitsWaveFunction& wf, PertRandMap G) { G.L->RandMap(); G.L->Perturb(wf); return wf;}

QBitsWaveFunction& operator << (QBitsWaveFunction& wf, Cnot G) { wf.Cnot_tr(G.ic,G.iq); return wf; }
QBitsWaveFunction& operator << (QBitsWaveFunction& wf, CCnot G) { wf.CCnot_tr(G.ic1,G.ic2,G.iq); return wf; }



#endif
