#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <complex>
#include <string.h>

using std::complex;

const complex<double> I(0,1);

inline double sqr(double x) { return x*x; };


//========================== set of gates primitives =============
class Gate    // -------- generic gate
{ 
 public:
 int nq1,nq2;
 
 Gate(int j1,int j2) { nq1=j1; nq2=j2;}
 ~Gate(){};
}; 
 

class WH : public Gate    //-------- Walsh-Hadamard gate at qubit(i)
{ public:
  WH(int j1, int j2=-1):Gate(j1,j2) {};
};

class X : public Gate     // Sigma X
{ public:
  X(int j1, int j2=-1):Gate(j1,j2) {};
};

class Y : public Gate     // Sigma Y
{ public:
  Y(int j1, int j2=-1):Gate(j1,j2) {};
};

class Z : public Gate     // Sigma Z
{ public:
  Z(int j1, int j2=-1):Gate(j1,j2) {};
};

class rotQBitBy : public Gate
{ public:
  double Angle;
  rotQBitBy(double A,int j1, int j2=-1):Gate(j1,j2){ Angle=A;};
};

class Cnot 
{ public:
  int ic,iq;
  Cnot(int iC, int iQ) { ic=iC; iq=iQ;};
};

class CCnot 
{ public:
  int ic1,ic2,iq;
  CCnot(int iC1, int iC2, int iQ) { ic1=iC1; ic2=iC2; iq=iQ;};
};

//============================================================

class Lattice
{ public:
  double * a,*b,A,B;
  int Nx,Ny,Nq;
 
  Lattice(int Lx, int Ly, double Alpha, double Beta)
  { Nx=Lx; Ny=Ly; Nq=Nx*Ny;
    A=Alpha; B=Beta;
    a=(double*)calloc(Nq,sizeof(double));
    b=(double*)calloc(Nq*2,sizeof(double));
    for(int i=0; i<Nq; i++)
    { a[i]=2*(drand48()-0.5)*A;
      b[2*i]=2*(drand48()-0.5)*B;
      b[2*i+1]=2*(drand48()-0.5)*B;
    }; 
  };
  
  Lattice(Lattice& L)
  { Nx=L.Nx; Ny=L.Ny; Nq=Nx*Ny;
    a=(double*)calloc(Nq,sizeof(double));
    b=(double*)calloc(Nq*2,sizeof(double));
    for(int i=0; i<Nq; i++)
    { a[i]=L.a[i];
      b[2*i]=L.b[2*i];
      b[2*i+1]=L.b[2*i+1];
    }; 
  };
  
  ~Lattice(){ free(a); free(b); };

// periodically continued  
int iQBit(int x, int y) { while(x>=Nx) x-=Nx; while(y>=Ny) y-=Ny; return x+Nx*y; };

Lattice& operator = (Lattice& L)
  { Nx=L.Nx; Ny=L.Ny; Nq=Nx*Ny;
    a=(double*)calloc(Nq,sizeof(double));
    b=(double*)calloc(Nq*2,sizeof(double));
    for(int i=0; i<Nq; i++)
    { a[i]=L.a[i];
      b[2*i]=L.b[2*i];
      b[2*i+1]=L.b[2*i+1];
    }; 
    return *this;
  };
   
};

static int PertCount=0;

class Pert
{ public:
  Lattice* L;
  Pert(Lattice* Lat) { L=Lat; PertCount++;};  
};

//============================================================

class QBitsWaveFunction
{ 
  public:
  int n;               // number of qubits
  long N;
  complex<double> *wf;  // pointer to Wave Function buffer
  char WFname[16];     // name of Wave Function (optional)
  
    QBitsWaveFunction(int nq,char* name=0) // 
	{ n=nq;
	  N=1<<n;
	  if(name) { strncpy(WFname,name,15); WFname[15]=0;}
	  else strcpy(WFname,"unnamed");
	  wf=(complex<double>*)calloc(N,sizeof(complex<double>));
	  if(wf==0) 
	    { printf("Fail to init wave function %s\n",WFname);
	      abort();
	    };	
	};
	
    QBitsWaveFunction(const QBitsWaveFunction& QBwf, char* name="unnamed") // cloning of wave function
	{ n=QBwf.n; N=QBwf.N;
	  strncpy(WFname,name,15); WFname[15]=0; //truncate name by 15 chars
	  wf=(complex<double>*)calloc(N,sizeof(complex<double>));
	  if(wf==0) 
	    { printf("Fail to init wave function %s\n",WFname);
	      abort();
	    };	
	  memcpy(wf,QBwf.wf,N*sizeof(complex<double>));      
	};
	
    ~QBitsWaveFunction() { free(wf);};

// equating two wave functions
QBitsWaveFunction& operator = (const QBitsWaveFunction& QBwf) 
	{ if(N!=QBwf.N)
          { n=QBwf.n;
	    N=QBwf.N;
	    if(wf) free(wf);
	    wf=(complex<double>*)calloc(N,sizeof(complex<double>));
	    if(wf==0) 
	      { printf("Fail to init wave function %s\n",WFname);
	        abort();
	      };
	  };	
	  strcpy(WFname,"equal_tmp");
	  memcpy(wf,QBwf.wf,N*sizeof(complex<double>));
	  return *this;      
	};


// rescaling of wave function by a scalar factor
QBitsWaveFunction& RescaleBy(complex<double> a) 
	{ if(wf==0) 
	    { printf("error: wave function %s is not inited\n",WFname);
	      abort();
	    };	
          for(long i=0; i<N; i++) wf[i]*=a;
	  return *this;    
	};

// these functions are for efficiency reason only, in order to avoid excessive temporary objects

QBitsWaveFunction& Sum(QBitsWaveFunction& QBwf1,QBitsWaveFunction& QBwf2) 
	{ 
	  if(QBwf1.N!=QBwf2.N) 
	  { printf("error: sum of vectors %s,%s of different lengths\n",QBwf1.WFname,QBwf2.WFname);
	    abort();
	  };
          for(long i=0; i<QBwf1.N; i++) wf[i]=QBwf1.wf[i]+QBwf2.wf[i];
	  return *this;    
	};

QBitsWaveFunction& Product(complex<double> a, QBitsWaveFunction& QBwf) 
	{ *this=QBwf;
          RescaleBy(a);
	  return *this;    
	};
	
QBitsWaveFunction& Allign() 
	{ 
          for(long i=0; i<N; i++) wf[i]=0;
	  wf[N-1]=1;
	  return *this;    
	};
	
QBitsWaveFunction& SetZero() 
	{ 
          for(long i=0; i<N; i++) wf[i]=0;
	  wf[0]=1;
	  return *this;    
	};

//------------------------------ operations on j-th qubit
//   	
QBitsWaveFunction& SigmaX(int j) 
	{ if(j<0 || j>=n) { printf("X: j=%d out of range\n",j); abort();}; 
	  int i1=1<<j;
	  complex<double> tmp;
          for(int i=0; i<N; i++) 
	    if(i1&i) { tmp=wf[i]; wf[i]=wf[i-i1]; wf[i-i1]=tmp;}; //flip the state
	  return *this;    
	};

QBitsWaveFunction& SigmaY(int j) 
	{ if(j<0 || j>=n) { printf("Y: j=%d out of range\n",j); abort();}; 
	  int i1=1<<j;
	  complex<double> tmp;
          for(int i=0; i<N; i++) 
	    if(i1&i) { tmp=I*wf[i]; wf[i]=I*wf[i-i1]; wf[i-i1]=-tmp;}; //flip the state
	  return *this;    
	};


QBitsWaveFunction& SigmaZ(int j) 
	{ if(j<0 || j>=n) { printf("Z: j=%d out of range\n",j); abort();}; 
	  int i1=1<<j;
          for(int i=0; i<N; i++) 
	    if(i1&i) wf[i]=-wf[i];
	  return *this;    
	};

QBitsWaveFunction& WH_tr(int j)   //Walsh-Hadamard transformation
	{ if(j<0 || j>=n) { printf("WH: j=%d out of range\n",j); abort();}; 
	  int i1=1<<j;
	  complex<double> tmp0,tmp1;
	  const double sqrt12=1./sqrt(2);
          for(int i=0; i<N; i++) 
	    if(i1&i) 
	    { tmp0=wf[i-i1];  tmp1=wf[i];
	      wf[i-i1]=(tmp0+tmp1)*sqrt12;   
	      wf[i]=(tmp0-tmp1)*sqrt12;
	    }; 
	  return *this;    
	};

QBitsWaveFunction& RotateQBit(int j, double Angle) 
	{ if(j<0 || j>=n) { printf("rotQBit: j=%d out of range\n",j); abort();}; 
	  int i1=1<<j;
	  complex<double> U0=exp(I*(Angle/2.)), U1=conj(U0);
          for(int i=0; i<N; i++) 
	    if(i1&i) 
	      { wf[i-i1]*=U0;
	        wf[i]*=U1;
	      }; 
	  return *this;    
	};

QBitsWaveFunction& InteractQBits(int j1, int j2, double g) 
	{ if(j1<0 || j1>=n) { printf("InteractBits: j1=%d out of range\n",j1); abort();}; 
	  if(j2<0 || j2>=n) { printf("InteractBits: j2=%d out of range\n",j2); abort();}; 
	  int i1=1<<j1,i2=1<<j2;
	  complex<double> U0=exp(I*g), U1=conj(U0);
          for(int i=0; i<N; i++) 
	    if(i1&i && i2&i) 
	      { wf[i-i1]*=U0;
	        wf[i]*=U1;
		wf[i-i2]*=U0;
		wf[i-i1-i2]*=U1;
	      }; 
	  return *this;    
	};


QBitsWaveFunction& Perturb(Lattice* L) 
	{ 
	  for(int i=0; i<n; i++) RotateQBit(i,2*L->a[i]); // SigmaZ interaction with external field
	  for(int i=0;i<n;i++) WH_tr(i); // Hadamard rotation SigmaX -> Sigma Z
          int iq;
	  for(int ix=0; ix<L->Nx; ix++) 
	     for(int iy=0; iy<L->Ny; iy++) 
	     { iq=L->iQBit(ix,iy);
	       InteractQBits(iq,L->iQBit(ix+1,iy),L->b[2*iq]);
	       InteractQBits(iq,L->iQBit(ix,iy+1),L->b[2*iq+1]);
	     }
	  for(int i=0;i<n;i++) WH_tr(i); // Hadamard rotation SigmaX -> Sigma Z
	  return *this;    
	};

QBitsWaveFunction& Cnot_tr(int j1,int j2) 
	{ if(j1<0 || j1>=n) { printf("Cnot: j1=%d out of range\n",j1); abort();}; 
	  if(j2<0 || j2>=n) { printf("Cnot: j2=%d out of range\n",j2); abort();}; 
	  int i1=1<<j1,i2=1<<j2;
	  complex<double> tmp;
          for(int i=0; i<N; i++) 
	    if(i1&i) 
	      if(i2&i){ tmp=wf[i]; wf[i]=wf[i-i2]; wf[i-i2]=tmp;}; //flip the state
	  return *this;    
	};

QBitsWaveFunction& CCnot_tr(int j1,int j2, int j3) 
	{ if(j1<0 || j1>=n) { printf("Cnot: j1=%d out of range\n",j1); abort();}; 
	  if(j2<0 || j2>=n) { printf("Cnot: j2=%d out of range\n",j2); abort();}; 
	  if(j3<0 || j3>=n) { printf("Cnot: j3=%d out of range\n",j3); abort();}; 
	  int i1=1<<j1,i2=1<<j2,i3=1<<j3;
	  complex<double> tmp;
          for(int i=0; i<N; i++) 
	    if(i1&i) 
	      if(i2&i)
	        if(i3&i){ tmp=wf[i]; wf[i]=wf[i-i3]; wf[i-i3]=tmp;}; //flip the state
	  return *this;    
	};


QBitsWaveFunction& operator << (WH G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) WH_tr(i); return *this; }

QBitsWaveFunction& operator << (X G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) SigmaX(i); return *this; }

QBitsWaveFunction& operator << (Y G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) SigmaY(i); return *this; }

QBitsWaveFunction& operator << (Z G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) SigmaZ(i); return *this; }

QBitsWaveFunction& operator << (rotQBitBy G) 
{ int imax=(G.nq1>G.nq2)?G.nq1:G.nq2; for(int i=G.nq1;i<=imax;i++) RotateQBit(i,G.Angle); return *this; }

QBitsWaveFunction& operator << (Pert G) { Perturb(G.L); return *this;}

QBitsWaveFunction& operator << (Cnot G) { Cnot_tr(G.ic,G.iq); return *this; }
QBitsWaveFunction& operator << (CCnot G) { CCnot_tr(G.ic1,G.ic2,G.iq); return *this; }


//--------------------------
	
void print()
        { printf("%s =\n",WFname);
	  for(int i=0; i<N;i++) printf("i= %d  %f+i*%f\n",i,real(wf[i]),imag(wf[i]));
	};   	
}; //=============== end of class "QBitsWaveFunction" 



// right-side multiplication by scalar 
QBitsWaveFunction operator * (const QBitsWaveFunction& QBwf, complex<double> a)
	{ QBitsWaveFunction QBwf1(QBwf,"mult_tmp");
          QBwf1.RescaleBy(a);
	  return QBwf1;    
	};

// left-side multiplication by scalar 
QBitsWaveFunction operator * (complex<double> a, QBitsWaveFunction& QBwf) 
	{ QBitsWaveFunction QBwf1(QBwf,"mult_tmp");
          QBwf1.RescaleBy(a);
	  return QBwf1;    
	};

// scalar product of two wave functions
complex<double> operator * (QBitsWaveFunction& QBwf1,QBitsWaveFunction& QBwf2) 
	{ 
	  if(QBwf1.N!=QBwf2.N) 
	  { printf("error: scalar product of vectors %s,%s of different lengths\n",QBwf1.WFname,QBwf2.WFname);
	    abort();
	  };
          complex<double> s=0;
          for(long i=0; i<QBwf1.N; i++) s+=conj(QBwf1.wf[i])*QBwf2.wf[i];
	  return s;    
	};

// sum of two wave functions
QBitsWaveFunction operator + (QBitsWaveFunction& QBwf1,QBitsWaveFunction& QBwf2) 
	{ 
	  if(QBwf1.N!=QBwf2.N) 
	  { printf("error: sum of vectors %s,%s of different lengths\n",QBwf1.WFname,QBwf2.WFname);
	    abort();
	  };
          QBitsWaveFunction QBwf(QBwf1.n,"sum_tmp");
          for(long i=0; i<QBwf1.N; i++) QBwf.wf[i]=QBwf1.wf[i]+QBwf2.wf[i];
	  return QBwf;    
	};
