/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <qop_primitives.h>               // for time slice primitives
#include <qop_builders.h>                 // for circuit recipies

/* *********************************************************************
   | Basically this file contains the implementations of controlled    |
   | circuits for primitive time slices. Each control() method builds  |
   | into the supplied container a circuit which implements the con-   |
   | trolled version of the current time slice. Note that:             |
   | 1) a controlled time slice is not in general a single time slice, |
   |    but a circuit (i.e. a sequence of time slices).                |
   | 2) since time slices embed the parallelisation of homogeneous     |
   |    unitary operations, a simply controlled time slice will in     |
   |    general require a cat state of a control register instead of   |
   |    a single control line. These cat state is assumed to be found  |
   |    in the qubit lines immediately following current ancillae.     |
   | 3) our initial choice of primitive time slices, H, R_k and CR_k,  |
   |    is close under control, i.e. their controlled circuits can be  |
   |    built EXACTLY with a finite number of these primitives; more-  |
   |    over, the controlled circuits have bounded depth.              |
   | ----------------------------------------------------------------- |
   | The "control_list" macro returns a qubit list which starts imme-  |
   | diately after the current ancillae and contains as many qubit     |
   | lines as the degree of parallelisation of the current time slice. |
   | It is the control register.                                       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Oct 2001         |
   ********************************************************************* */
#define control_list Qubit_list(cfirst, get_parallelisation())

/* *********************************************************************
   | Controlled Hadamard. Pass the control register and the Hadamard   |
   | register to the constructor of a custom class. The built object   |
   | can act as a slice list to be incorporated into "the_operations". |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Oct 2001         |
   ********************************************************************* */
void Qop_slice_Hadamard::control(class Qop_slicelist *the_operations,
				 size_type cfirst) const {
  the_operations->splice
    (~QCondHadamard_slices(control_list, get_list(0)));
}

/* *********************************************************************
   | Controlled Phase: we are lucky, this is a primitive itself. Thus  |
   | we simply have to push the appropriate slice into the list.       |
   | "get_power_fraction_macro" is a macro which takes care to define  |
   | and appropriately fill the the_power and the_fraction variables.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Oct 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Touloouse,     30 Apr 2002         |
   ********************************************************************* */
void Qop_slice_Phase::control(class Qop_slicelist *the_operations,
			      size_type cfirst) const {
  get_power_fraction_macro;
  the_operations->push_back
    (new Qop_slice_CondPhase(control_list, get_list(0),
			     the_power, the_fraction));
}

/* *********************************************************************
   | Controlled controlled Phase. Pass the control register, the slice |
   | control register and the phase register to the constructor of a   |
   | custom class. The built object can act as a slice list to be      |
   | incorporated into "the_operations". "get_power_fraction_macro" is |
   | a macro which takes care to define and appropriately fill the     |
   | the_power and the_fraction variables.                             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Oct 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Touloouse,     30 Apr 2002         |
   ********************************************************************* */
void Qop_slice_CondPhase::control(class Qop_slicelist *the_operations,
				  size_type cfirst) const {
  get_power_fraction_macro;
  the_operations->splice
    (~QCondCondPhase_slices(control_list, get_list(0), get_list(1),
			    the_power, the_fraction));
}

/* *********************************************************************
   | Controlled swap. Though the swap operation is "classical", its    |
   | controlled version needs to be executed on the quantum device.    |
   | Pass the control register and the two registers to be swapped to  |
   | the constructor of a custom class. The built object can act as a  |
   | slice list to be incorporated into "the_operations".              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 03 Dec 2001         |
   ********************************************************************* */
void Qop_slice_Swap::control(class Qop_slicelist *the_operations,
			     size_type cfirst) const {
  the_operations->splice
    (~QCondSwap_slices(control_list, get_list(0), get_list(1)));
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
