/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iostream>           // C++ Input/Output
#include <qregister.h>        // class declaration
#include <qmanager_base.h>    // abstract interface to the address manager
#include <qinterface_base.h>  // address manager and quantum device (abstract)

/* *********************************************************************
   | This private inline method is a quick link to the address manager |
   | for this register. It is here to hide the actual way of finding   |
   | the manager for each register. Currently the language uses only   |
   | one quantum interface, hence only one address manager, but things |
   | could change in the future ....                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      06 Nov 2002         |
   ********************************************************************* */
inline Qaddress_manager_base *Qreg::get_manager(void) const {
  /* get_quantum_interface() is a global function which returns
     the single instance of a quantum interface, which the quantum
     library takes care to instantiate before main(). */
  return get_quantum_interface().get_manager();
}

/* *********************************************************************
   | This is the copy constructor. It simply initialises the internal  |
   | list by "copying" that of the supplied register. This requires    |
   | some degree of communication with the address manager, because    |
   | the usage count of the copied segments must be incremented.       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 26 Mar 2001         |
   ********************************************************************* */
Qreg::Qreg(const Qreg &a_register) {
  /* create a temporary list for holding the copy. */
  Qubit_list temporary_list = a_register.get_qubit_list();
  /* incorporate this temporary list and checkin all its qubits. */
  incorporate_qubits(temporary_list);
}

/* *********************************************************************
   | The effect of the destructor is to release the allocated qubits   |
   | (this involves contacting the address manager).                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Feb 2001         |
   ********************************************************************* */
Qreg::~Qreg() {
  /* inform the address manager that these resources will be dallocated */
  get_manager()->checkout_qubits(get_qubit_list());
}

/* *********************************************************************
   | This method initialises a qubit list to a given bit string. Once  |
   | some consistency checks have been run (if the number of bits is   |
   | different from the number of qubits referenced in the list, this  |
   | method will throw exception assignment_error()), the method sends |
   | the bit string with the value to be written and the qubit list.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
void Qreg::run_initialisation(const Qbitset &the_bits,
			      const Qubit_list &a_list) const {
  /* Just be sure that we are going to send the correct number of bits. */
  if (the_bits.size() !=  a_list.size()) throw assignment_error();
  /* now send all the data to the quantum interface */
  get_quantum_interface().submit_initialisation(the_bits, a_list);
}

/* *********************************************************************
   | This method is the core part of a general constructor for a       |
   | quantum register, accepting a collection of booleans. The size    |
   | of the vector is the size of the register.                        |
   | ----------------------------------------------------------------- |
   | 1) Check that size of the vector is positive, otherwise throw an  |
   |    exception of type init_error().                                |
   | 2) Contact the address manager and try to get the required number |
   |    of logical qubits (this may throw an exception). Initialise    |
   |    the qubit list with the returned object (by incorporation).    |
   | 3) Perform the physical operation of "writing" the boolean values |
   |    into the register. This is the most similar thing to an ini-   |
   |    tialisation, but remember that quantum registers can hold more |
   |    than integer numbers ...                                       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
void Qreg::run_construction(const Qbitset &the_bits) {
  /* First check that the size is greater than zero (a zero qubit
     register is a nonsense up to now). */
  if (the_bits.empty()) throw init_error();
  /* Contact the address manager (the object which keeps trace of the
     allocation status of all the logical qubits) and try to get an
     appropriately sized list of free qubits. */
  Qubit_list free_list = get_manager()->get_qubits(the_bits.size());
  /* incorporate the returned list, but DON'T checkin the qubits again,
     since this has already been done by get_qubits(). */
  incorporate_qubits(free_list, false);
  /* Perform the register initialisation. This is not a trivial action, it
     involves contacting the quantum device and issuing the appropriate
     number of laser pulses or similar to encode the value into the register,
     i.e. to select the appropriate element of the computational basis. */
  *this = the_bits;
}

/* *********************************************************************
   | This method is an interface to the join() method of the list      |
   | class. It splices the supplied list into the current register's   |
   | list, performs a recollection on the new list and updates the     |
   | usage counters of the address manager in case of joins. This      |
   | method is well suited for augmentation of registers through       |
   | addition of a new list (at the end).                              |
   | ----------------------------------------------------------------- |
   | (06 Sep 2001) S.Bettelli. The update of the usage counts, which   |
   | was once performed through get_manager()->update_border_usage()   |
   | is now implemented by checking in the joined segment and then     |
   | checking out the two parts.                                       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Aug 2001         |
   ********************************************************************* */
void Qreg::run_join(Qubit_list &a_list) {
  /* get two copies of the segments at the end of the current list and
     at the beginning of the supplied list. They will be useful in case
     there is a join in order to update the address manager status. */
  Qubit_segment first_part(get_qubit_list().back());
  Qubit_segment second_part(a_list.front());
  /* splice the two lists (i.e. move all elements from the second one
     into the first, in constant time). The simplification routine is
     automatic, and we can understand if any join took place by looking
     at the return value. The call may generate an exception of type
     Qubit_list::invalid_splice if the lists are not disjoint. */
  bool was_simplified = qubits.join(a_list);
  /* if there was a simplification, we must inform the address manager
     to update some internal structures. This is easily achieved by
     cheking in the joined segment and checking out the two parts. This
     order can't be changed since we don't want recollections to happen
     in the address manager! */
  if (was_simplified) {
    /* calculate the joined segment and check it in. The constructor needs
       the boolean argument because we are suppling first() and last() and
       not lowest() and the size. */
    Qubit_segment whole_segment(first_part.first(), second_part.last(), true);
    get_manager()->checkin_qubits(whole_segment);
    /* checkout the two previous parts. Don't change the order! */
    get_manager()->checkout_qubits(first_part);
    get_manager()->checkout_qubits(second_part);
  }
}

/* *********************************************************************
   | This auxiliary method will call the address manager and ask it to |
   | fragment the main list appropriately. The "target" address is the |
   | absolute address at which the address manager representation is   |
   | to be broken, and it must be inside "the_segment". The default    |
   | behaviour is to fragment the segment so that "target" is found in |
   | the second piece (where second is relative to the segment order). |
   | If the boolean argument is true, the behaviour is reversed (its   |
   | default is to be false).                                          |
   | ----------------------------------------------------------------- |
   | The second argument to fragment_END|BEGIN must be a valid address |
   | in the address manager busy list, close to the summary segment    |
   | containing the target address. Since summary segments are not     |
   | ordered, pass the lowest() address in the passed segment, not     |
   | first(). The fragment type field must be BEGIN or END, and the    |
   | return value of is_reversed() must be checked.                    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Sep 2001         |
   ********************************************************************* */
void Qreg::run_fragmentation(const Qubit_segment &the_segment,
			     address target, bool reverse) const {
  /* select two cases out of four */
  if ((!reverse &&  the_segment.is_reversed()) ||
      ( reverse && !the_segment.is_reversed()))
    /* fragment into [lowest(), target] [target + 1, highest()] */
    get_manager()->fragment_END(target, the_segment.lowest());
  else
    /* fragment into [lowest(), target - 1] [target, highest()] */
    get_manager()->fragment_BEGIN(target, the_segment.lowest());
}

/* *********************************************************************
   | This method will incorporate the supplied list into the register  |
   | address list. Well, what the method actually does is to swap the  |
   | content of the current address list and the supplied list. This   |
   | operation doesn't involve the quantum device, only the address    |
   | manager which must be informed that the usage count of the regi-  |
   | ster qubits must be incremented (this operation is skipped if the |
   | boolean variable is not set).                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 26 Mar 2001         |
   ********************************************************************* */
void Qreg::incorporate_qubits(Qubit_list &a_list, bool run_checkin) {
  /* if register list is not empty, tell the address manager to
     release its parts (because they will not remain in use). */
  if (! get_qubit_list().is_empty())
    get_manager()->checkout_qubits(get_qubit_list());
  /* swap the content of the two lists. The supplied list now contains
     the old address list which was embedded in this register. */
  qubits.swap(a_list);
  /* If the boolean argument "run_checkin" asks for it, contact the
     address manager and inform it that it must increment the usage
     count for all the qubits in the list. */
  if (run_checkin) get_manager()->checkin_qubits(get_qubit_list());
}

/* *********************************************************************
   | This method implements the measurement of a register. We need to  |
   | send the encoding of the register and its size. After this the    |
   | method waits for the result to be returned from the quantum de-   |
   | vice. The return value is a collection of booleans, which is gua- |
   | ranteed to have the correct size unless an exception occurs.      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
Qbitset Qreg::measure(void) const {
  /* send the list which defines this register. */
  get_quantum_interface().submit_measurement(get_qubit_list());
  /* ask the quantum device to return the result. This call is blocking
     and needs the number of qubits referenced in the register. */
  return get_quantum_interface().get_measurement_result(size());
}

/* *********************************************************************
   | This method locates the qubits addressed by the register on the   |
   | quantum device (this is sometimes useful when interacting with    |
   | simulators, where the wavefunction needs additional information   |
   | in order to be interpreted). We need to send the encoding of the  |
   | register and its size. After this the method waits for the result |
   | (i.e. a Qubit_list) to be returned from the quantum device. This  |
   | list is guaranteed to have the correct size, unless an exception  |
   | occurs.                                                           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      16 Feb 2003         |
   ********************************************************************* */
Qubit_list Qreg::locate(void) const {
  /* send the list which defines this register. */
  get_quantum_interface().submit_locate_request(get_qubit_list());
  /* ask the quantum device to return the result. This call is blocking
     and needs the number of qubits referenced in the register. */
  return get_quantum_interface().get_locate_result(size());
}

/* *********************************************************************
   | This method implements the single qubit addressing, i.e. it will  |
   | return a temporary register containing only one qubit of the cur- |
   | rent register (using an_index to select it). This is the basic    |
   | brick for writing qubit addressing expressions like:              |
   |      Qreg my_register = my_reg[k];                                |
   | ----------------------------------------------------------------- |
   | This method must contact the address manager and cause it to      |
   | fragment the appropriate summary segment in its main list. We     |
   | don't want to delve into the details of the address manager here, |
   | but we must send to it the minimum amount of information for      |
   | completing this task: this is the operation type (single qubit    |
   | fragment), the address of the single qubit and an address which   |
   | starts a summary segment in the main list (though it doesn't need |
   | to be the segment which contain the address, which the register   |
   | does not know about; anyway, the closer the better).              |
   | ----------------------------------------------------------------- |
   | (12 Sep 2001) Stefano Bettelli. The list calculation in this me-  |
   | thod has been moved to a method in the list class.                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   ********************************************************************* */
Qreg Qreg::operator[](address an_index) const {
  /* prepare two auxiliary variables to hold the result of get_subrange() */
  address target_address;
  qubits_iterator selected_segment;
  /* calculate a new list of Qubit_segments with a single segment con-
     taining the address corresponding to "an_index". We will use the
     get_subrange() method from the list class for this. A byproduct of
     this call is the qubit address to which this index is translated to
     and the segment in which it was found: they will be both saved in
     the auxiliary variables which we prepared. */
  Qubit_list new_list = get_qubit_list().
    get_subrange(an_index, target_address, selected_segment);
  /* call the address manager and ask it to fragment the main list
     appropriately. The "reference address" must be a valid address in
     the busy list, close to the summary segment containing target_address.
     Since summary segments are not ordered, pass the lowest() address
     in the selected_segment, not first(). The fragment type field must
     be SINGLE. */
  get_manager()->fragment_SINGLE(target_address, selected_segment->lowest());
  /* Now create a temporary register by incorporating the new list and re-
     turn it to the caller (it will probably be used for ctor of a Qreg). */
  return Qreg(new_list);
}

/* *********************************************************************
   | This method implements the qubit range addressing, i.e. it will   |
   | return a temporary register containing "a_size" qubits of the     |
   | current register (using "an_index" to select the start of the     |
   | range). It is the basic brick for writing qubit range addressing  |
   | expressions like:                                                 |
   |      Qreg my_register = my_reg(k, 3);                             |
   | (06 Sep 2001) S.Bettelli; using "a_size" = 0 means "up to the end |
   | of the register. Thus Qreg my_register = my_reg(k, 0) means that  |
   | my_register references all the qubits in my_reg exception made    |
   | for the first three. "a_size" = 0 should be the default value.    |
   | ----------------------------------------------------------------- |
   | This method must contact the address manager and cause it to      |
   | fragment the appropriate summary segments in its main list. We    |
   | don't want to delve into the details of the address manager here, |
   | but we must send to it the minimum amount of information for      |
   | completing this task. See also the operator[] method.             |
   | ----------------------------------------------------------------- |
   | (12 Sep 2001) Stefano Bettelli. The list calculation in this me-  |
   | thod has been moved to a method in the list class.                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   ********************************************************************* */
Qreg Qreg::operator()(address an_index, size_type a_size) const {
  /* "a_size" = 0 conventionally means "up to the end of the register".
     Therefore now we should check for its value, and in case it is
     zero we need to calculate the real size. */
  if (a_size == 0) a_size = (size() - an_index);
  /* prepare auxiliary variables to hold the result of get_subrange() */
  address begin_address, end_address;
  qubits_iterator begin_segment, end_segment;
  /* calculate a new list of Qubit_segments containing the subrange selec-
     ted by "an_index" and "a_size". We will use the get_subrange() method
     from the list class for this. A byproduct of this call are the qubit
     addresses to which this indexes are translated to and the segments
     in which they were found: all these informations will be saved in
     the auxiliary variables which we prepared. */
  Qubit_list temporary_list = get_qubit_list().
    get_subrange(an_index, a_size, begin_address, end_address,
		 begin_segment, end_segment);
  /* fragment the address manager main list appropriately. Reverse the
     behaviour of run_fragmentation() for the end_segment. */
  run_fragmentation(*begin_segment, begin_address); 
  run_fragmentation(*end_segment, end_address, true);
  /* Now create a temporary register by incorporating the new list and re-
     turn it to the caller (it will probably be used for ctor of a Qreg). */
  return Qreg(temporary_list);
}

/* *********************************************************************
   | This method will return a temporary register whose address list   |
   | is the concatenation of the address lists of the current register |
   | and the register passed as an argument. It is the basic brick for |
   | writing concatenation expressions like the following:             |
   |      Qreg my_register = my_reg_1 & my_reg_2;                      |
   | This method can indirectly (i.e. through a called routine) throw  |
   | an exception of type Qubit_list::invalid_splice if the qubit      |
   | lists of the two registers have a non empty overlap. This has to  |
   | be forced because we want a qubit to be referenced at most once   |
   | in every register, otherwise some quantum operations could not be |
   | implemented.                                                      |
   | ----------------------------------------------------------------- |
   | We don't need to contact the address manager here (well, the ctor |
   | of Qreg does this for us, but we don't contact it explicitely in  |
   | the routine) because the "union" assumption for the manager list  |
   | is fulfilled by sure. A simplification routine is run during the  |
   | creation of the register address list, but no communication with  |
   | the address manager is necessary since the list is seeded into    |
   | the new register only at the end alredy in minimal form. The      |
   | simplification routine keeps the segment list as compact as pos-  |
   | sible, by joining neighbouring segments with the same ordering.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   ********************************************************************* */
Qreg Qreg::operator&(const Qreg &second_register) const {
  /* prepare a new qubit list which is the copy of the qubit list of
     the first quantum register (without touching it). This time it
     is faster to create directly the list object. */
  Qubit_list new_list(get_qubit_list());
  /* repeat this operation with the second register, in a different list. */
  Qubit_list second_part(second_register.get_qubit_list());
  /* now splice the second list onto the first, thus concatenating the
     two original ones (this is our definition of concatenation). Also
     run the embedded simplification routine. We don't care whether any
     join occurred or not (i.e. the return value), because the lists
     are not yet part of living registers. The call may generate an
     exception of type Qubit_list::invalid_splice if the lists are
     not disjoint. */
  new_list.join(second_part);
  /* last, create a temporary register object by incorporating the overall
     list and return it to the caller (it will probably be used for ctor
     of another register). */
  return Qreg(new_list);
}

/* *********************************************************************
   | This method will concatenate the address list of the supplied     |
   | register (which is left untouched) with that of the current       |
   | register (which IS modified). It is an efficient version of the   |
   | concatenation operator in case the result must be written back in |
   | the first of the two registers. The return value is the modified  |
   | register itself. It is the basic brick for writing concatenation  |
   | expressions like the following:                                   |
   |      my_register &= my_other_register;                            |
   | This method can indirectly (i.e. through a called routine) throw  |
   | an exception of type Qubit_list::invalid_splice if the qubit      |
   | lists of the two registers have a non empty overlap. This has to  |
   | be forced because we want a qubit to be referenced at most once   |
   | in every register, otherwise some quantum operations could not    |
   | be implemented.                                                   |
   | ----------------------------------------------------------------- |
   | We don't need to fragment the address manager lists here because  |
   | the "union" assumption for the manager list is fulfilled by sure. |
   | A simplification routine is however run after the join, and it    |
   | can contact the address manager. The simplification routine keeps |
   | the segment list as compact as possible, by joining neighbouring  |
   | segments with the same ordering. Segment border usage counts are  |
   | correctly taken into account.                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 23 Apr 2001         |
   ********************************************************************* */
Qreg &Qreg::operator&=(const Qreg &second_register) {
  /* take a temporary copy of the address list of the second register.
     This is no waste of space since we must extend the first register
     anyway (this list will be spliced). It is faster to copy the 
     complete list immediately. */
  Qubit_list list_copy(second_register.qubits);
  /* Now contact the address manager and inform it that it must increment
     the usage count for all the qubits in the copied list (because from
     now on they will be used also by the current register). */
  get_manager()->checkin_qubits(list_copy);
  /* splice the list copy into the current register's list (i.e. move all
     elements from the second one into the first, in constant time), thus
     concatenating the two registers (this is our definition of augmentation).
     The simplification routine is automatic, as well as the update of the
     internal structures of the address manager in case of a join. 
     The call may generate an exception (Qubit_list::invalid_splice)
     if the lists are not disjoint. */
  this->run_join(list_copy);
  /* that's it; return the current register to the caller. */
  return *this;
}

/* *********************************************************************
   | This operator prepends some fresh qubits to the register. The new |
   | qubits come from the free qubit pool of the address manager and   |
   | they must be initialised to |0>. This new list is subsequently    |
   | moved at the beginning of the segment list of the register.       |
   | ----------------------------------------------------------------- |
   | The new qubits are requested from the address manager (which      |
   | automatically cheks them in) as a list object. The initialisation |
   | sequence is then run on this list (though it isn't a real regi-   |
   | ster yet). After this the list is swapped with the register's     |
   | one and then spliced (allowing us to splice it at the beginning.) |
   | (28 Sep 2001) S.Bettelli. run "run_initialisation" BEFORE trans-  |
   | forming the free_list into the temporary_list, otherwise the free |
   | list will be empty and an assignment_error() will be thrown.      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Sep 2001         |
   ********************************************************************* */
Qreg &Qreg::operator+=(size_type the_size) {
  /* if size is zero, this is a joke, ignore it. */
  if (the_size == 0) return *this;
  /* Contact the address manager and try to get an appropriately sized list
     of free qubits. Remember that these qubits are already checked in! */
  Qubit_list free_list = get_manager()->get_qubits(the_size);
  /* the free_list is not yet part of any register, but the address
     manager can't know about this (see before). Therefore it should
     be absolutely safe to initialise them now (initialise to zero). */
  run_initialisation(Qbitset(the_size), free_list);
  /* now we want to splice the new list in, but we need to put it AT THE
     BEGINNING (not at the end as splice_and_recollect would do). The
     easiest way is to swap the two lists before. */
  qubits.swap(free_list);
  /* now the register "contains" only the new list, while the old one is
     stored in the temporary_list. A splice will now finish the work. */
  run_join(free_list);
  /* return the current register after augmentation */
  return *this;
}

/* *********************************************************************
   | This method discards some qubits from the register beginning.     |
   | ----------------------------------------------------------------- |
   | The routine loops until the_size qubits have been discarded. At   |
   | each loop cycle with the_size != 0 only two action can be perfor- |
   | med. Either the current first segment in the qubit list is to be  |
   | erased completely (which is easy) or it has to be erased in part. |
   | All structures in the address manager are correctly kept in sync. |
   | (16 Dec 2001) S.Bettelli, moved some logic to the list class.     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Sep 2001         |
   ********************************************************************* */
Qreg &Qreg::operator-=(size_type the_size) {
  /* loop until we have discarded enough qubits */
  while (the_size != 0) {
    /* get a reference to the first segment in the address list. */
    const Qubit_segment &first_segment = get_qubit_list().front();
    /* save the size of the current first segment */
    size_type size_first = first_segment.size();
    /* if the number of qubits to be discarded is greater than or equal
       to the size of the first segment, we have to eliminate the segment
       as a whole and to decrease the number in "the_size". */
    if (the_size >= size_first) {
      /* checkout the first segment in the register's qubit list. */
      get_manager()->checkout_qubits(first_segment);
      /* we have done with the address manager status. Now actually
	 erase the first segment from the register's qubit list. */
      qubits.pop_front();
      /* decrease the_size using the first segment's size. */
      the_size -= size_first;
    }
    /* if the previous condition is not met, we have to break the first
       segment into two appropriate pieces, and discard the first. */
    else {
      /* calculate the address at which the break occurs. */
      address break_here = first_segment[the_size];
      /* fragment the address manager internal representation. This step
	 is needed in preparation for the checkin of the keep_me segment. */
      run_fragmentation(first_segment, break_here);
      /* create a dummy segment representing the part which is to be kept.
	 The boolean argument in the constructor is needed because we are
	 specifying the first and last address, not the lowest and the size. */
      Qubit_segment keep_me(break_here, first_segment.last(), true);
      /* Now update all the usage counts in the address manager for the
	 part and the whole. This can be simply done by first checking in
	 the dummy segment, THEN checking out the whole segment. You can't
	 checkout the whole segment first, since this could lead to a
	 recollection in the address manager which we don't want! */
      get_manager()->checkin_qubits(keep_me);
      get_manager()->checkout_qubits(first_segment);
      /* Now remove "the_size" qubits from the first segment. */
      qubits.pop_front(the_size);
      /* there is nothing left to remove. */
      the_size = 0;
    }
    /* if the list is empty, we have a real problem: a quantum register
       cannot contain zero qubits. Throw an exception. */
    if (get_qubit_list().is_empty()) throw invalid_discard();
  }
  /* return the current register after augmentation */
  return *this;
}

/* *********************************************************************
   | This is the overloading of operator<< for a Qreg object which is  |
   | the interface for a quantum register. The goal is to show the     |
   | "status" of the register: since we are dealing with quantum       |
   | registers, this means showing the list of linked logical qubits   |
   | of course, not the value!                                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qreg &a_register) {
  os << "Qreg " << (const void *)&a_register           // register identifier
     << " and its segment(s):\n"
     << a_register.get_qubit_list();                   // register content
  return os;                                           // return stream
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
