/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <qinterface_base.h>                  // base class definition

/* *********************************************************************
   | Constructor for the basic quantum interface:                      |
   | 1) save the passed pointer to a specific address manager in a     |
   |    private field. The address manager must be already informed    |
   |    about the quantum memory it must handle.                       |
   | 2) create a permutation list and save its pointer in a private    |
   |    field. The list is initialised with a single segment spanning  |
   |    the whole quantum memory from the first address (this corre-   |
   |    sponds to the identity permutation).                           |
   | 3) set "permutation_is_trivial" to true.                          |
   | ----------------------------------------------------------------- |
   | Both the passed address manager and the permutation list will be  |
   | deleted in the destructor of Qinterface_base, hence the two       |
   | pointers must be inteded as owned by this class.                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      18 Jul 2002         |
   ********************************************************************* */
Qinterface_base::Qinterface_base(Qaddress_manager_base *the_address_manager) {
  /* save the pointer to a specific address manager. */
  address_manager = the_address_manager;
  /* initialise the permutation list with the single segment [0, +infty[. */
  permutation_list = new Qubit_list(0, get_manager()->get_number_of_qubits());
  /* state the the permutation list is currently trivial. */
  permutation_is_trivial = true;
}

/* *********************************************************************
   | Destructor for the basic quantum interface.                       |
   | 1) shut down the quantum address manager. It boils down to delete |
   |    the private pointer "address_manager". Be careful! The address |
   |    manager can not be shut down when any register is still alive, |
   |    this would cause an exception (try to recover it here).        |
   | 2) delete the permutation list (this is always possible)          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      18 Jul 2002         |
   ********************************************************************* */
Qinterface_base::~Qinterface_base() {
  /* try to delete the actual address manager and nullify the pointer. */
  try { delete get_manager(); address_manager = NULL; }
  /* issue an error message if things are messy */
  catch (Qaddress_manager_base::still_busy)
    { std::cerr << "\nAddress manager could not be shut down!\n"; }
  /* delete the permutation list and nullify the pointer. */
  delete get_permutation();
  permutation_list = NULL;
}

/* *********************************************************************
   | This method encodes a generic request for the quantum device,     |
   | which can be blocking (like a measurement or a status requests to |
   | the device) or non blocking (like unitary operations, i.e. time   |
   | slices). During the execution of this method, the command channel |
   | is considered busy. At the end, the "filled" or "blocking" condi- |
   | tion is signalled. The arguments to be specified are:             |
   |   -) the operation code (an easy-to-decode integer value),        |
   |   -) a parameter value (this is optional: use a value of zero     |
   |      if the operation doesn't have a parameter),                  |
   |   -) a boolean variable stating whether it is blocking or not,    |
   |   -) and a few pointers to constant objects (lists, list collec-  |
   |      tions or bitsets) which are to be passed as additional data. |
   |      Null pointers mean that the associated data are not perti-   |
   |      nent to the current request, and will not be processed.      |
   | ----------------------------------------------------------------- |
   | (15/02/2003) S.Bettelli. Renamed submit_timeslices into this new  |
   |   and more generic submit_request. This can be used also for      |
   |   blocking operations (like measurements and status requests)     |
   |   and for initialisation. In principle, each interaction with the |
   |   quantum device should pass through this method. Remember that   |
   |   the protocol needs a specific order for the additional data,    |
   |   so write the bitset first, then the addresses.                  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      15 Jul 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      15 Feb 2003         |
   ********************************************************************* */
void Qinterface_base::submit_request(message_type a_code,
				     message_type a_parameter, bool blocking,
				     const Qubit_list_collection *lists,
				     const Qubit_list *a_list,
				     const Qbitset *a_bitset) {
  /* set the command channel to the busy state */
  channel_busy();
  /* send the operation code */
  submit_operation(a_code);
  /* send an optional parameter for the operation */
  if (a_parameter) submit_parameter(a_parameter);
  /* send the vector of boolean values, if present */
  if (a_bitset)    submit_booleans(*a_bitset);
  /* send the translated addresses, if present (multiple lists) */
  if (lists)       submit_segments(*lists);
  /* send the translated addresses, if present (one list) */
  if (a_list)      submit_segments(*a_list);
  /* set the command channel to the free state */
  channel_free();
  /* to whom it may concern: the channel is not empty
     (and there might be blocking commands on the queue too). */
  if (blocking) channel_blocking(); else channel_filled();
}

/* *********************************************************************
   | This method forwards a vector of booleans to the quantum device.  |
   | ----------------------------------------------------------------- |
   | According to the protocol (defined in the BYTECODE file) the fol- |
   | lowing values are to be sent:                                     |
   | 1) a single quantum_message_type number specifying the number of  |
   |    qubits to be initialised (the number of boolean values in the  |
   |    supplied bit set, say BIT_NUM).                                |
   | 2) ceil(BIT_NUM/S) quantum_message_type numbers, which are to be  |
   |    interpreted as bitmasks, S being the number of bits which can  |
   |    be stored in a quantum_message_type number. Indeed this is     |
   |    very simple since bits are already stored that way in a_bitset |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Sep 2001         |
   ********************************************************************* */
void Qinterface_base::submit_booleans(const Qbitset &a_bitset) {
  /* first, send the number of meaningful bits. */
  write_bytecode(a_bitset.size());
  /* then, copy all the elements of a_bitset. */
  for (Qbitset::size_type i = 0; i < a_bitset.elements(); ++i)
    write_bytecode(a_bitset.get_element(i));
}

/* *********************************************************************
   | This method forwards a collection of lists of addresses to the    |
   | quantum device. The permutation list is used in this routine for  |
   | the last translation, if it is not trivial.                       |
   | ----------------------------------------------------------------- |
   | Let us summarise the address translation stages (for more details |
   | read the reference paper cited in the README file).               |
   |                                                                   |
   | STAGE 1) Executing a quantum operator on a quantum register means |
   |    executing all its time slices in sequence on that register.    |
   |    Each time slice is defined by one or more "index lists" (the   |
   |    formal addresses). Each index list must be matched against the |
   |    "address list" (real addresses) in the register, returning the |
   |    list of the addresses where the elementary operation associa-  |
   |    ted to the slice must be executed. This stage is performed by  |
   |    each time slice and the results are passed to submit_segments()|
   |    in the Qinterface_base class (this routine).                   |
   | STAGE 2) Before being sent to the quantum device, the addresses   |
   |    are modified another time with a permutation (this is done in  |
   |    the current routine). This permutation is necessary for the    |
   |    implementation of qubit swaps without interacting with the     |
   |    quantum device; credits for this idea to Luciano Serafini!     |
   |    Indeed, the algorithm for the permutation is just the previous |
   |    algorithm for matching, with the elements of "the_lists"       |
   |    playing the role of index lists and get_permutation() playing  |
   |    the role of an address list. Since this routine is not cheap,  |
   |    we try to skip it whenever possible (that is when the variable |
   |    permutation_is_trivial is true).                               |
   | STAGE 3) The address lists which have been calculated during      |
   |    stage one and two are sent to the command queue; the protocol  |
   |    requires that they are mixed before being sent.                |
   | ----------------------------------------------------------------- |
   | (17 Jul 2002) S.Bettelli, test the size of the segment list first |
   |    If it is one, call the optimised submit_segment routine which  |
   |    does not need any list mixing (which would be just a copy in   |
   |    this case). This accounts for maybe 10% savings in time!       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 27 Nov 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      17 Jul 2002         |
   ********************************************************************* */
void Qinterface_base::submit_segments(const Qubit_list_collection &the_lists) {
  /* count the number of lists in the collection. */
  typedef Qubit_list_collection::size_type size_type;
  size_type number_of_lists = the_lists.size();
  /* if there is only one list, use the optimised routine. */
  if (number_of_lists == 1) return submit_segments(the_lists[0]);
  /* prepare another list collection which will holds the lists after
     they have undergone the final permutation. */
  Qubit_list_collection permuted_lists;
  /* We have to run the translation (permutation) for each list 
     separately (the optimisation for the case when the permutation
     is trivial is automatic in the "translated" macro). */
  for (size_type index = 0; index < number_of_lists; ++index)
    /* Use the_lists[index] as an index list for the permutation.
       If the permutation is trivial, simply submit the passed list. */
    permuted_lists.push_back(permutation_is_trivial ? the_lists[index] :
			     the_lists[index].translate(*get_permutation()));
  /* now mix all this lists into a single one. The static mix_lists()
     method in the Qubit_list class destroys the passed lists for
     efficiency (they are emtpy when the routine returns). Then send
     the translated and mixed list to the queue. */
  submit_segment_spec(Qubit_list::mix_lists(permuted_lists));
}

/* *********************************************************************
   | This method forwards a collection of lists of addresses to the    |
   | quantum device.According to the protocol (defined in the BYTECODE |
   | file) the following values are to be sent:                        |
   |                                                                   |
   | 1) a single quantum_message_type number specifying the number of  |
   |    segments in the list, say REG_SIZE.                            |
   | 2) REG_SIZE pairs of the form [FIRST, LAST], one for each segment |
   |    in the mixed list. Hence a list with k segments is encoded by  |
   |    2k+1 numbers.                                                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
void Qinterface_base::submit_segment_spec(const Qubit_list &a_list) {
  /* first, send the number of segments in the list. */
  write_bytecode(a_list.segments());
  /* then, copy the [first, last] pair for each segment to the queue. */
  for (Qubit_list::const_iterator a_segment = a_list.begin();
       a_segment != a_list.end(); ++a_segment) {
    write_bytecode(a_segment->first());
    write_bytecode(a_segment->last());
  }
}

/* *********************************************************************
   | This method modifies the current permutation list so that the     |
   | following quantum operations will behave as if a quantum swap had |
   | really been executed. However, it is a simple reorganisation of   |
   | the addresses. It can issue an exception (invalid_swapgate()).    |
   | ----------------------------------------------------------------- |
   | In simpler words, this method takes two index lists and, for each |
   | pair (i,j) of corresponding indexes in the lists it exchanges the |
   | elements p_i and p_j of the permutation list p. This reorganiza-  |
   | tion is done using the same translate() algorithm that is used    |
   | for the index to address translation in submit_segments; the in-  |
   | dex list used as input for translate() is prepared by a separate  |
   | routine in the Qubit_list class.                                  |
   | ----------------------------------------------------------------- |
   | (22 Jul 2002) S.Bettelli, at the end of the routine, update the   |
   |  value of the "permutation_is_trivial" variable.                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 03 Dec 2001         |
   ********************************************************************* */
void Qinterface_base::submit_permutation(const Qubit_list &swap_list_1,
					 const Qubit_list &swap_list_2) {
  /* translate any exception */
  try {
    /* we need non constant copies of the swap lists. */
    Qubit_list swap_list_copy_1(swap_list_1);
    Qubit_list swap_list_copy_2(swap_list_2);
    /* also get the total number of qubits in the permutation list, i.e.
       the number of qubits managed by the quantum device. */
    Qaddress_manager_base::size_type total_size =
      get_manager()->get_number_of_qubits();
    /* ask the list class to calculate an index list corresponding to the
       passed lists (it is a static method). The previously calculated
       total_size will be the size of the index list. */
    Qubit_list index_list = Qubit_list::create_permutation
      (swap_list_copy_1, swap_list_copy_2, total_size);
    /* use this list to "translate" the interface's permutation list. */
    Qubit_list new_permutation = 
      index_list.translate(*get_permutation());
    /* set the new permutation as the interface's permutation. */
    get_permutation()->swap(new_permutation);
    /* run a minimisation on the permutation list. */
    get_permutation()->make_minimal();
  }
  /* translate any exception to invalid_swapgate() */
  catch (Qexception) { throw invalid_swapgate(); }
  /* now, test if the resulting permutation list is trivial or not.
     The permutation is trivial if there is just one segment and it
     is direct (in case we didn't mess up the list before ... ) */
  if ((get_permutation()->is_single()) && 
      (get_permutation()->front().is_direct()))
    permutation_is_trivial = true;
  else
    permutation_is_trivial = false;
}

/* *********************************************************************
   | This output friend function exposes the status of the quantum     |
   | interface to the outside word (by calling more specific functions |
   | from the embedded objects).                                       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 23 Jul 2001         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os,
			 const Qinterface_base &an_interface) {
  /* output the address manager (no need for a title, because it already
     has one. Use a special form of operator<< which is able to print
     out the details of the most derived class through a pointer to the
     base class. */
  os << an_interface.get_manager() << std::endl;
  /* output the permutation list */
  os << "This is the permutation list speaking!" << std::endl;
  os << *an_interface.get_permutation() << std::endl;
  /* return the output stream */
  return os;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
