/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <qlist.h>                             // base class declaration
#include <set>                  // for STL sorted associative containers
#include <map>                  // for STL sorted associative containers

/* a shortcut for the list constant iterator. */
typedef Qubit_list::const_iterator list_iterator;

/* *********************************************************************
   | This inline function can compare two segments through their ite-  |
   | rators: it is "operator less than" for iterators. The reason for  |
   | the existence of this function is that sometimes the "natural"    |
   | ordering is not the most efficient way for iterating through a    |
   | Qubit_list object. The "absolute address" order iterates through  |
   | the segments in ascending absolute address order: this is useful  |
   | when testing for range overlaps, but not only. If (A,B) is a pair |
   | of iterators to the segments (a,b), then a is less than b if and  |
   | only if (A < B) is true.                                          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
inline bool operator<(const list_iterator &first, const list_iterator &second)
{ return (*first < *second); }

/* *********************************************************************
   | This class is a STL unique sorted associative container which can |
   | "remember" the second ordering; the second ordering can be calcu- |
   | lated each time it is needed and stored in an object from this    |
   | class through constant iterators to the main list.                |
   | (29 Nov 2001) S.Bettelli: an attempt was made in the past to      |
   | store this structure permanently in the Qubit_list class and keep |
   | it updated, but the cons soon became more than the pros. Actually,|
   | recalculating the "absolute address" ordering structure can be    |
   | done with a complexity which is k log k, where k is the number of |
   | segmets, which is OK for our pourposes!                           |
   | ----------------------------------------------------------------- |
   | (30 Jul 2001), Stefano Bettelli: VERY IMPORTANT NOTE!             |
   | The key to the unique sorted set is an iterator; in fact this is  |
   | necessary since we have to dereference the iterator and compare   |
   | the pointed objects in some way. This is also dangerous, since    |
   | the order in the set is supposed to be a set invariant, but we    |
   | can simply screw up this order by changing the segments (which    |
   | are not part of the set, unlike their iterators). In practice,    |
   | this means that a const_set_type object is not necessarily valid  |
   | if any change is made to the corresponding Qubit_list object.     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
typedef std::set<list_iterator> const_set_type;

/* *********************************************************************
   | This is another typedef for a structure which is currently used   |
   | only by the index to address translation routine. It is a unique  |
   | sorted associative container, but the key (an iterator) is diffe- |
   | rent from the value (a pointer to a list). See the translate      |
   | method for further details.                                       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
typedef std::map<list_iterator, Qubit_list *> translation_map_type;

/* *********************************************************************
   | This is another typedef for a structure which is currently used   |
   | only by the create_permutation routine. It is a unique sorted     |
   | associative container, with both the key and the value of type    |
   | iterator. See the create_permutation method for further details.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Dec 2001         |
   ********************************************************************* */
typedef std::map<list_iterator, list_iterator> permutation_map_type;

/* *********************************************************************
   | This function gets all the constant iterators to the elements of  |
   | an address list and saves them into the const_set_type object.    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 Nov 2001         |
   ********************************************************************* */
inline void fill_const_set(const_set_type &a_map,
			   const Qubit_list &a_list) {
  for (list_iterator a_segment = a_list.begin();
       a_segment != a_list.end(); ++a_segment)
    a_map.insert(a_segment);
}

/* *********************************************************************
   | This function gets all the constant iterators to the elements of  |
   | an address list and saves them into the translation_map_type      |
   | object, initialising the pointers to NULL.                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
inline void fill_translation_map(translation_map_type &a_map,
				 const Qubit_list &a_list) {
  for (list_iterator a_segment = a_list.begin();
       a_segment != a_list.end(); ++a_segment)
    a_map[a_segment] = NULL;
}

/* *********************************************************************
   | This method checks whether there is any overlap between the       |
   | current list and the list passed as argument. It returns false    |
   | if no element is common to the two lists.                         |
   | ----------------------------------------------------------------- |
   | (11 Jul 2001) S.Bettelli, instead of working by checking overlaps |
   |   between any pair of segments (since segment to segment overlap  |
   |   check is constant time, the complexity would be proportional to |
   |   the product of the number of segments inside the two lists),    |
   |   this method uses the "absolute address" ordering, which is      |
   |   calculated at the beginning: this guarantees linear complexity  |
   |   in the sum of the number of segments in the two objects.        |
   | (29 Nov 2001) S.Bettelli, added an optimisation for the typical   |
   |   case when both lists have only one segment: don't use maps!     |
   | (11 Nov 2002) S.Bettelli, extended the optimisation to the case   |
   |   when at least one list has less than two segments. In this case |
   |   it is possible to use the test between a list and a segment.    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 Mar 2001         |
   | Stefano Bettelli, INFN and Trento University, 11 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      11 Nov 2002         |
   ********************************************************************* */
bool Qubit_list::is_overlapping(const Qubit_list &second_list) const {
  /* return immediately if any of the two qubit lists is empty */
  if (this->is_empty() || second_list.is_empty()) return false;
  /* if the second list contains only one segment, call is_overlapping
     between the current list and this segment. */
  if (second_list.is_single())
    return this->is_overlapping(second_list.front());
  /* if the first list contains only one segment, do the same thing
     as before between the second list and this segment. */
  if (this->is_single())
    return second_list.is_overlapping(this->front());
  /* the following two maps will be used for iterating through the
     lists using the absolute address ordering. The segments will not
     be modified, hence we can use constant iterators. */
  const_set_type first_set, second_set;
  /* fill the two maps using the helper function. */
  fill_const_set(first_set, *this);
  fill_const_set(second_set, second_list);
  /* have two iterators to iterators and let them point to the
     beginning of the qubit_sets of the two lists. */
  const_set_type::iterator first_iterator = first_set.begin();
  const_set_type::iterator second_iterator = second_set.begin();
  /* loop forever; well, there is a test inside the loop that exits the
     loop as soon as one of the two iterators is equal to the end()
     iterator of the corresponding map. */
  while (true) {
    /* test overlap; if it does not fail, return true. */
    if ((**first_iterator).is_overlapping(**second_iterator)) return true;
    /* if we are still here, we have to increment the iterator corresponding
       to the segment with the lowest absolute address between the two that
       we are currently dealing with. If the increased iterator reaches the
       end of the corresponding map, it's time to leave. */
    if (**first_iterator < **second_iterator)
      { if ((++first_iterator) == first_set.end()) break; }
    else
      { if ((++second_iterator) == second_set.end()) break; }
  }
  /* if all tests failed (i.e. found no overlap), return false. */
  return false;
}

/* *********************************************************************
   | This method accepts a list object (the "address list") and builds |
   | a copy of the current list (the "index list") replacing each      |
   | index i with the i-th location in the address list. This method   |
   | will be used at "execution time" by operators which need to tran- |
   | slate their index list into a real location list by fitting it    |
   | onto a quantum register.                                          |
   | ----------------------------------------------------------------- |
   | In order to reduce the runtime for this routine, we will use the  |
   | absolute value ordering for the current list and the assumption   |
   | that each list is overlap free: this makes it possible to search  |
   | in the address list with a one pass algorithm.                    |
   | ----------------------------------------------------------------- |
   | (13May2002) S.Bettelli, this method is now used also to reorga-   |
   | nise the index lists of an operator (the "remap" transformation,  |
   | which is a sort of generalisation of "split", "swap" & "offset"). |
   | In this case, the "address_list" works like a permutation.        |
   | ----------------------------------------------------------------- |
   | (22Jul2002) S.Bettelli, the name of this routine has been changed |
   | from "translate" to "translate_map"; translate is now a small     |
   | inline method which can call either this routine or a simpler     |
   | version without maps (the simplified version is used when the two |
   | lists do not contain too many segments; in this case using maps   |
   | indeed is too expensive).                                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 26 Jul 2001         |
   ********************************************************************* */
Qubit_list Qubit_list::translate_map(const Qubit_list &address_list) const {
  /* This is the list to be returned at the end of the routine,
     containing the translated addresses. Put it here to facilitate
     the named return value optimisation. */
  Qubit_list translated_list;
  /* create a translation map which will make the translation algorithm
     more efficient, then fill it. This structure will contain pairs
     (iterator, list pointer) where iterator points to a segment in
     the index list (the current list) and the list pointer links to
     a translation of that segment. This map is essential both during
     the initial part of the algorithm, when the translation lists are
     calculated, and during the second part, when the lists are joined. */
  translation_map_type translation_map;
  fill_translation_map(translation_map, *this);
  /* "index_iterator" is an iterator in the translation map. It will
     allow us to run through the address list using the absolute (index)
     address ordering, so that the address list can be analised in one
     pass only (using the natural ordering). It sounds a bit complicated
     but it implements a simple idea indeed. Initialise it. */
  translation_map_type::iterator index_iterator = translation_map.begin();
  /* loop through the current list, which is the index list, using the
     absolute address ordering. */
  while (index_iterator != translation_map.end()) {
    /* get the corresponding iterator to a segment in the current list. */
    const const_iterator index_segment = (*index_iterator).first;
    /* translate the current index segment (this is now done by a more
       specialised method). Reset the search only if this is the first
       index segment in the translation map. The translation list is
       returned as a dynamically allocated object: we have thus to update
       the element in the translation map pointed to by index_iterator,
       storing the pointer in the data field (the second). */
    (*index_iterator).second =
      translate_segment(address_list, index_segment,
			index_iterator == translation_map.begin());
    /* increase the index_iterator and repeat. */
    ++index_iterator;
  }
  /* Now join all the translation lists together in the correct order,
     i.e. following the natural order in the current (index) list. The
     pointer to the correct translation list is retrived through the
     translation_map in average logarithmic time. */
  for (const_iterator a_segment = begin(); a_segment != end(); ++a_segment) {
    /* get the iterator to the corresponding element in the translation map. */
    translation_map_type::iterator entry = translation_map.find(a_segment);
    /* get a pointer to the correct translation list. */
    Qubit_list *translation_list = entry->second;
    /* splice the content of this list at the end of the overall list. */
    translated_list.splice(translated_list.end(), *translation_list);
    /* release the dinamically allocated memory. */
    delete translation_list;
    /* erase this entry from the translation map; this will
       speed up the following searches. */
    translation_map.erase(entry);
  }
  /* consistency check, if the translation_map is not empty we have a
     problem. Issue the generic exception for this method. */
  if (! translation_map.empty()) throw invalid_translation();
  /* also run a minimisation routine to reduce the list to minimality
     (it is not guaranteed that it is minimal now). */
  translated_list.make_minimal();
  /* return the translated list. */
  return translated_list;
}

/* *********************************************************************
   | This static method takes two address lists by reference and       |
   | calculates a suitable index list which can implement the corres-  |
   | ponding permutation. In practice, for each pair (a,b) of indexes  |
   | in the lists, the returned list p satisfies p_a = b and p_b = a,  |
   | while it satisfies p_k = k for all other indexes. An exception of |
   | type invalid_permutation() is thrown if the input lists overlap.  |
   | The arguments are modified by this routine, don't reuse them!     |
   | (16 Dec 2001) S.Bettelli, there is now a second argument, the to- |
   |   tal size of the list to be returned. A segment is pushed into   |
   |   the list at the end in order to fulfill the request.            |
   | (07 Jan 2002) S.Bettelli, the list collection as an argument is   |
   |   out, now there are explicitely two lists.                       |
   | ----------------------------------------------------------------- |
   | The algorithm has three stages:                                   |
   | 1) the two lists are fragmented in order to have the same number  |
   |    of segments, with corresponding segments of the same size.     |
   |    For each pair (*a,*b) of corresponding segments, *b gets re-   |
   |    versed if *a is reversed and viceversa.                        |
   | 2) each pair (*a,*b) of corresponding segments causes the pair    |
   |    (a,b) and the pair (b,a) to be inserted into an STL map, which |
   |    is ordered with respect to the absolute address ordering of    |
   |    the segment pointed to by the first iterator of the pair.      |
   | 3) pairs are retrived from the map following its order and the    |
   |    second segment of each pair is copied into the permutation     |
   |    list. Gaps in the list are filled with "identical" segments,   |
   |    including a final segment in order to reach total_size.        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 03 Dec 2001         |
   ********************************************************************* */
Qubit_list Qubit_list::create_permutation
(Qubit_list &first_list, Qubit_list &second_list, size_type total_size){
  /* prepare the list to be returned here, in order to facilitate
     the name return value optimisation. */
  Qubit_list permutation_list;
  /* check that the two lists are not overlapping. */
  if (first_list.is_overlapping(second_list)) throw invalid_permutation();
  /* now fragment the two lists so that they have exactly the same
     number of segments and that corresponding elements have the same
     size. First, take two iterators to the beginning of the lists. */
  iterator first_iterator  = first_list.begin();
  iterator second_iterator = second_list.begin();
  /* loop until the iterators reach the end of the lists. */
  while (first_iterator != first_list.end() &&
	 second_iterator != second_list.end()) {
    /* get the size of the current segment of each list. */
    size_type first_size  = first_iterator->size();
    size_type second_size = second_iterator->size();
    /* if they have different sizes, split the longest segment. */
    if (first_size > second_size) 
      first_list.split(first_iterator, second_size);
    if (first_size < second_size)
      second_list.split(second_iterator, first_size);
    /* now check if these segments are direct or not */
    bool first_reversed  = first_iterator->is_reversed();
    bool second_reversed = second_iterator->is_reversed();
    /* if the first is reversed, reverse the second and viceversa
       (independently). This will not change the ordering in the
       following map, just the actual segments which will be copied
       in the permutation list. */
    if (first_reversed) second_iterator->reverse();
    if (second_reversed) first_iterator->reverse();
    /* advance the two iterators. */
    ++first_iterator;
    ++second_iterator;
  }
  /* check that the lists had the same length. */
  if (first_iterator != first_list.end() ||
      second_iterator != second_list.end()) throw invalid_permutation();
  /* create a permutation map and fill it. The map will contain pairs
     of iterators. For each pair (*a,*b) of segments in the two lists,
     the pair (a,b) and the pair (b,a) are entered into the map. */
  permutation_map_type permutation_map;
  first_iterator  = first_list.begin();
  second_iterator = second_list.begin();
  /* lists are OK, we don't need to check both iterators here. */
  while (first_iterator != first_list.end()) {
    /* insert both pairs */
    permutation_map.insert(make_pair(first_iterator, second_iterator));
    permutation_map.insert(make_pair(second_iterator, first_iterator));
    /* advance both iterators. */
    ++first_iterator;
    ++second_iterator;
  }
  /* build the permutation list, iterating through the permutation
     map (which is ordered with respect to the absolute address order):
     A) for each element in the map, copy the value in the list
     B) push a segment in the list to fill the gap before the next value. */
  size_type covered_indexes = 0;
  for (permutation_map_type::iterator an_element = permutation_map.begin();
       an_element != permutation_map.end(); ++an_element) {
    /* calculate the gap before next segment */
    size_type gap = an_element->first->lowest() - covered_indexes;
    /* if the gap is positive, create a filling segment. */
    if (gap) permutation_list.push_back(Qubit_segment(covered_indexes, gap));
    /* copy the permuted segment. */
    permutation_list.push_back(*(an_element->second));
    /* update the count of covered indexes. */
    covered_indexes += (gap + an_element->second->size());
  }
  /* if the size is already larger than required, we have a problem. */
  if (covered_indexes > total_size) throw invalid_permutation();
  /* see if the size is not ok yet. */
  size_type gap = total_size - covered_indexes;
  /* push a final segment to reach total_size if necessary. */
  if (gap) permutation_list.push_back(Qubit_segment(covered_indexes, gap));
  /* update the count of covered indexes. */
  covered_indexes += gap;
  /* run a minimisation on the output list. */
  permutation_list.make_minimal();
  /* that's it, return the mixed list. */
  return permutation_list;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
