/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_REGISTER_ADDRESS_MANAGER_
#define __QUANTUM_REGISTER_ADDRESS_MANAGER_

#include <iostream>          // C++ standard input/output
#include <map>               // for standard STL sorted maps
#include <qmanager_base.h>   // for the abstract base class interface
#include <qsummary.h>        // extended version of qubit segments

/* *********************************************************************
   | This is the class for a specific quantum register address manager.|
   | Each time a qubit must be allocated/deallocated the request must  |
   | pass through this object which knows how to translate the place-  |
   | holders inside quantum registers (logical qubits) into suitable   |
   | addresses for the quantum device (at least it knows who knows it  |
   | ...). Communication between the address manager and the quantum   |
   | registers is mediated by a list of qubit segments.                |
   | ----------------------------------------------------------------- |
   | 1) The main structure of the address manager is a linked list of  |
   |    Qubit_summary objects, whose class inherits from the segment   |
   |    class Qubit_segment. Each of them is an address segment with   |
   |    the same usage count for all qubits. Since this segments do    |
   |    not need any ordering, we decide that the design will never    |
   |    allow is_reversed() to return true.                            |
   | 2) The main qubit list will never contain two consecutive free,   |
   |    i.e. with zero usage count, segments (automatic recollection   |
   |    is provided). At costruction and destruction time it must      |
   |    contain only one free segment and nothing else. On the other   |
   |    hand, two consecutive segments with the same non-zero usage    |
   |    count may exits.                                               |
   | 3) Every Qubit_segment inside a register will always be a union   |
   |    of Qubit_summaries in the main list (this simplifies deallo-   |
   |    cation a lot); for this reason it is possible for two segments |
   |    with the same (non-zero) usage count to be consecutive in the  |
   |    main list (and joined inside the register!). This is known as  |
   |    the "union assumption".                                        |
   | 4) Each Qubit_summary in the main list will have a corresponding  |
   |    entry in the busy or free list (see later). The cookie field   |
   |    of the Qubit_summary will be kept in sync with this entry,     |
   |    using the information inside their virtual table.              |
   |    The cookie will speed up the erase operations of the busy and  |
   |    free lists (see later).                                        |
   | 5) A map named "busy list" is provided for fast translation of    |
   |    the lowest address of a segment into the corresponding segment |
   |    summary inside the main list. This is to speed up the deallo-  |
   |    cation stage, since finding the element in the map is fast and |
   |    finding the counters union from there is also fast.            |
   | 6) A multimap named "free list" is provided for fast retrival of  |
   |    free segments of the appropriate size. The retrival algorithm  |
   |    should take care to produce the least possible fragmentation   |
   |    (in conjunction with deallocation).                            |
   | 7) An additional structure which keeps trace of which sizes are   |
   |    available in the free list is provided for a both fast and     |
   |    efficient quantum memory allocation tecnique. Basically, this  |
   |    means that we can know efficiently which is the smallest size  |
   |    greater than a given treshold for which at least one free      |
   |    segment exists (this keeps fragmentation low, I hope ...).     |
   | 8) (1 May 2001) a Qubit_summary object now contains two additio-  |
   |    nal counters for holding border usage counts. These counters   |
   |    allow the address manager to remember how many registers are   |
   |    actively using a given border of a segment in the main list.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
class Qaddress_manager : public Qaddress_manager_base {
private:
  // the linked list type for the main list (rename it)
  typedef std::list<Qubit_summary> main_list_type;
  // an abbreviation for the following type
  typedef main_list_type::iterator main_iterator;
  // the container type for pointers to generic busy segments
  typedef std::map<address, main_iterator> busy_list_type;
  // the container type for pointers to generic free segments
  typedef std::multimap<size_type, main_iterator> free_list_type;
  // the container type for the map of available sizes
  typedef std::map<size_type, Qubit_summary::counter_type> size_map_type;
  // a private template class for specialised Qubit_cookies
  // basically it keeps an iterator to the busy or free list
  template <class T>
  struct Qubit_cookie_of : public Qubit_cookie, public T::iterator {
    /* specialised constructor, copying the iterator part. */
    Qubit_cookie_of(const typename T::iterator &it) : T::iterator(it) {}
  };
private:
  // no copy constructors please!
  Qaddress_manager(const Qaddress_manager &another_manager);
  // no assignement operator please!
  Qaddress_manager &operator=(const Qaddress_manager &another_manager);
public:
  // default constructor with amount of quantum memory
  Qaddress_manager(size_type the_area_size);
  // the virtual destructor
  virtual ~Qaddress_manager();
  // this method returns the number of managed qubits
  size_type get_number_of_qubits(void) const { return area_size; }
  // this method allocates the logical qubits
  Qubit_list get_qubits(size_type qubit_number);
  // an output function for this class
  std::ostream &output_details(std::ostream &os) const;
  // this method increments the usage counts for a single Qubit_segment
  void checkin_qubits(const Qubit_segment &a_segment);
  // this method deallocates a single Qubit_segment
  void checkout_qubits(const Qubit_segment &a_segment);
private:
  // this method fragments a summary appropriately
  void fragment(address target, address reference, fragment_type type);
private:
  // the size of the quantum memory area to be managed
  size_type area_size;
  // the main linked list for keeping usage counts
  main_list_type main_list;
  // the map of pointers to busy segments
  busy_list_type busy_list;
  // the map of pointers to free segments
  free_list_type free_list;
  // a sorted associative map for holding counts of available sizes
  size_map_type size_map;
private:
  // this method finds a free segment suitable for allocation
  main_iterator find_free_segment(size_type a_size);
  // this method finds a segment given an address and a hint
  main_iterator search_segment(address target, address reference);
  // this method generates two segments from one in the main list
  main_iterator split_direct(main_iterator first_element,
			     size_type new_size_first);
  // this method tries to recollect a range of segments
  void recollect_segments(main_iterator &first_segment,
			  main_iterator &last_segment);
  // this method builds the address translation table.
  void allocate_resources(void);
  // this method cleans up the internal structures
  void deallocate_resources(void);
  // this method checks whether there are allocated qubits or not
  bool is_free(void) const;
  // a helper function to insert an element into the busy list
  void busy_insert(const main_iterator a_segment);
  // a helper function to insert an element into the free list
  void free_insert(const main_iterator a_segment);
  // a helper function to automatically insert an element into free/busy list
  void insert(const main_iterator a_segment);
  // a helper function to remove an element from the busy list
  void busy_erase(const main_iterator a_segment);
  // a helper function to remove an element from the free list
  void free_erase(const main_iterator a_segment);
  // a helper function to automatically remove an element from free/busy list
  void erase(const main_iterator a_segment);
};

/* this output function will be precious during debugging */
std::ostream &operator<<(std::ostream &os,const Qaddress_manager &the_manager);

/* this class has a lot of small inline methods with a huge amount of
   interspersed comments. The code has been moved to a different file. */
#include <qmanager_inlines.h>

#endif  // __QUANTUM_REGISTER_ADDRESS_MANAGER_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
