/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __ADDRESS_MANAGER_BASE_
#define __ADDRESS_MANAGER_BASE_

#include <qexception.h>               // for "quantum" exceptions
#include <qsegment.h>                 // for basic qubit segments
#include <qlist.h>                    // for the qubit list class

// this define specialises the fragment() with different names (see later)
#define _fragment_(type)                                       \
void fragment_ ## type(address target, address reference)      \
{ return fragment(target, reference, type); }

/* *********************************************************************
   | This class is an abstract interface to a quantum address manager. |
   | It specifies the minimum capabilities that an address manager     |
   | must comply with to be correctly used by quantum registers. This  |
   | class is abstract.                                                |
   | ----------------------------------------------------------------- |
   | Besides defining some typedefs, exception classes and enums, this |
   | interface forces the following virtual capabilities:              |
   | ====>> get_number_of_qubits()                                     |
   |        This method will return the number of qubits managed by    |
   |        the address manager (not less than the size of the device) |
   | ====>> get_qubits(qubit_number)                                   |
   |        This method causes the address manager to find the speci-  |
   |        fied number of qubits inside the free pool, modify its     |
   |        internal structures and return a qubit list object which   |
   |        contains a list of addresses to the caller register.       |
   |        It embeds a call to checkin_qubits (don't do it twice!)    |
   |        The get_qubits method must be used by completely new regi- |
   |        sters, that is which have no overlap with other ones.      |
   | ====>> checkin_qubits(a_list) or (a_segment)                      |
   |        This method causes the address manager to increment all    |
   |        usage counters for all the segments referenced inside the  |
   |        list (or for the single a_segment argument).               |
   | ====>> checkout_qubits(a_list) or (a_segment)                     |
   |        This method causes the address manager to decrement the    |
   |        usage counts for all the qubits referenced inside the list |
   |        (or for the single a_segment argument).                    |
   | ====>> fragment(target, reference, type)                          |
   |        This method causes the address manager to fragment its     |
   |        internal compact representation of the quantum memory in   |
   |        order to fulfill the "union" assumption, that is "all seg- |
   |        ments in a register are union of segments inside the       |
   |        address manager". "target" is the beginning of the new     |
   |        segment, "reference" is the beginning of an existing seg-  |
   |        ment close to it and lesser than it, and "type" is the     |
   |        fragmentation type as in the following:                    |
   |        BEGIN --> fragments the segment which contains target in   |
   |                  two parts: [.. target-1] [target ..]             |
   |        END   --> fragments the segment which contains target in   |
   |                  two parts: [.. target] [target+1 ..]             |
   |        SINGLE -> fragments the segment which contains target in   |
   |                  3 parts: [.. target-1] [target] [target+1 ..]    |
   |        (empty segments cannot be spawned).                        |
   |        This capability is indeed private, it is exported to the   |
   |        rest of the world with different names like fragment_END() |
   |        or similar. The fragment methods must be used by registers |
   |        which are to reference parts of already existing registers,|
   |        in order to prepare the internal structures of the address |
   |        manager before a call to checkin_qubits().                 |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 07 Feb 2001         |
   ********************************************************************* */
class Qaddress_manager_base {
public:
  // this is an alias for the qubit address type
  typedef Qubit_segment::address address;
  // this is an alias for the qubit segment size type
  typedef Qubit_segment::size_type size_type;
  // a class for exceptions during the destruction stage
  _Qexception(still_busy);
  // a class for problems during an allocation
  _Qexception(allocation_failed);
  // a class for problems during qubits checkin
  _Qexception(checkin_failed);
  // a class for problems during qubits checkout
  _Qexception(checkout_failed);
  // a class for problems during segment fragmentation
  _Qexception(fragment_failed);
  // a class for problems during segment search
  _Qexception(search_failed);
  // a class for errors during split operations
  _Qexception(split_error);
protected:
  // an enumeration for different behaviours of the fragment method
  enum fragment_type { BEGIN, END, SINGLE };
protected:
  // this is the default constructor
  Qaddress_manager_base() { }
  // no copy constructor please!
  Qaddress_manager_base(const Qaddress_manager_base &a_manager);
  // no assignement operator please!
  Qaddress_manager_base &operator=(const Qaddress_manager_base &a_manager);
public:
  // this is the virtual trivial destructor
  virtual ~Qaddress_manager_base() { }
public:
  // this method returns the number of managed qubits
  virtual size_type get_number_of_qubits(void) const = 0;
  // this method allocates the logical qubits
  virtual Qubit_list get_qubits(size_type qubit_number) = 0;
  // this method increments the usage count for all qubits in the list
  void checkin_qubits(const Qubit_list &a_list);
  // this method "deallocates" a list of logical qubits
  void checkout_qubits(const Qubit_list &a_list);
  // this method increments the usage counts for a single Qubit_segment
  virtual void checkin_qubits(const Qubit_segment &a_segment) = 0;
  // this method deallocates a single Qubit_segment
  virtual void checkout_qubits(const Qubit_segment &a_segment) = 0;
  // named versions of the fragment routine
  _fragment_(BEGIN); _fragment_(END); _fragment_(SINGLE);
  // an output function for the most derived class
  virtual std::ostream &output_details(std::ostream &os) const = 0;
protected:
  // this method fragments a summary appropriately
  virtual void fragment(address target,
			address reference, fragment_type type) = 0;
public:
};

/* this output function will be precious during debugging */
std::ostream &operator<<(std::ostream &os,
			 const Qaddress_manager_base *the_manager);

#endif  // __ADDRESS_MANAGER_BASE_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
