/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_REGISTER_ADDRESS_MANAGER_INLINES_
#define __QUANTUM_REGISTER_ADDRESS_MANAGER_INLINES_

/* The Qaddress_manager class has a lot of small inline methods with a
   huge amount of interspersed comments. The code has been moved to 
   this file just to be more manageable. */
#include <qmanager.h>

/* *********************************************************************
   | An helper function which inserts an entry in the busy or free     |
   | list, given the iterator to a Qubit_summary inside the main list. |
   | The list is chosen by looking at the usage count of the segment.  |
   | This function is only a wrapper for busy_insert() or free_insert()|
   | which should be used instead if the target list is already known  |
   | and/or the usage count is not in sync with it yet.                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Feb 2001         |
   ********************************************************************* */
inline void Qaddress_manager::insert(const main_iterator a_segment) {
  return ((a_segment->is_free()) ? 
	  free_insert(a_segment) : busy_insert(a_segment));
}

/* *********************************************************************
   | An helper function which removes an entry from the busy or free   |
   | list, given the iterator to a Qubit_summary inside the main list. |
   | The list is chosen by looking at the usage count of the segment.  |
   | This function is only a wrapper for busy_erase() or free_erase(), |
   | which should be used instead if the target list is already known  |
   | and/or the usage count is not in sync with it yet.                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Feb 2001         |
   ********************************************************************* */
inline void Qaddress_manager::erase(const main_iterator a_segment) {
  return ((a_segment->is_free()) ? 
	  free_erase(a_segment) : busy_erase(a_segment));
}

/* *********************************************************************
   | An helper function which inserts an entry in the busy list, given |
   | the iterator to a Qubit_summary inside the main list. This avoids |
   | a lot of typing. The cookie field of the Qubit_summary updated    |
   | too. Though it seems long, it is a single statement.              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 22 Jan 2001         |
   ********************************************************************* */
inline void Qaddress_manager::busy_insert(const main_iterator a_segment) {
  /* this will update the Qubit_summary cookie with a pointer to an
     iterator to an element in the busy list (reading it from the
     return value of "insert"). set_cookie() will automatically destroy
     an old cookie if present. */
  a_segment->set_cookie
    /* this creates a new Qubit_cookie_of<busy_list_type>; the argument is
       an iterator to the busy_list. This cookie will be destroyed either by
       another set_cookie() call or by the destructor of the Qubit_summary. */
    (new Qubit_cookie_of<busy_list_type>
     /* this actually inserts something into the busy list and return a
	pair<iterator, bool> where "iterator" links to the inserted element
	and "bool" is an error code (ignored ... it can't fail ...) */
     (busy_list.insert
      /* this creates a temporary which is a pair "lowest address of a
	 segment", "iterator to the segment in the main list" for
	 insertion. */
      (busy_list_type::value_type(a_segment->lowest(), a_segment)
       /* this reads the first field in the return value of insert();
	  it is an iterator to the newly inserted element. */
       ).first));
}

/* *********************************************************************
   | An helper function which inserts an entry in the free list, given |
   | the iterator to a Qubit_summary inside the main_list. This avoids |
   | a lot of typing. The cookie field of the Qubit_summary updated    |
   | too. This also updates the sorted map of available sizes.         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 22 Jan 2001         |
   ********************************************************************* */
inline void Qaddress_manager::free_insert(const main_iterator a_segment) {
  /* this will update the Qubit_summary cookie with a pointer to an
     iterator to an element in the free list (reading it from the
     return value of "insert"). set_cookie() will automatically destroy
     an old cookie if present. */
  a_segment->set_cookie
    /* this creates a new Qubit_cookie_of<free_list_type>; the argument is
       an iterator to the free_list. This cookie will be destroyed either by
       another set_cookie() call or by the destructor of the Qubit_summary. */
    (new Qubit_cookie_of<free_list_type>
     /* this actually inserts something into the free list and return an
	iterator to the newly inserted element. */
     (free_list.insert
      /* this creates a temporary which is a pair "size of a segment",
	 "iterator to the segment in the main list" for insertion. */
      (free_list_type::value_type(a_segment->size(), a_segment))));
  /* Now update the sorted map of available sizes. This means: try to
     see if a_segment->size() is already known to the map. If it is,
     increment its counter; if it is not, insert into the map with zero
     usage. Given the wonderful properties of operator[] for an STL
     associative container, this can be done with a single instruction
     (because the default constructor for data_type sets it to zero). */
  ++size_map[a_segment->size()];
}

/* *********************************************************************
   | An helper function which removes an entry from the busy list,     |
   | given the iterator to a Qubit_summary inside the main list.       |
   | This avoids a lot of typing. The cookie field of the              |
   | Qubit_summary is "updated" too.                                   |
   | ----------------------------------------------------------------- |
   | Being "a_segment" referenced inside the busy list we could elimi- |
   | nate the entry from the map using its key, with the statement     |
   | busy_list.erase(a_segment->first()), but since we can read it     |
   | directly from the Qubit_cookie we will take the shortest path.    |
   | Erasing the object pointed to by the cookie (an iterator to an    |
   | element of the busy list) is accomplished by the set_cookie()     |
   | function, without further pain for us, but we must take care to   |
   | eliminate the element from the map before.                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jan 2001         |
   ********************************************************************* */
inline void Qaddress_manager::busy_erase(const main_iterator a_segment) {
  /* eliminate the <key, data> pair from the busy list, taking the
     shortest path (an explicit cast is required however). */
  busy_list.erase(*(Qubit_cookie_of<busy_list_type> *)a_segment->get_cookie());
  /* eliminate the dynamically allocated iterator. */
  a_segment->set_cookie(NULL);
}

/* *********************************************************************
   | An helper function which removes an entry from the free list,     |
   | given the iterator to a Qubit_summary inside the main list.       |
   | This avoids a lot of typing. The cookie field of the              |
   | Qubit_summary is "updated" too.                                   |
   | This also updates the sorted map of available sizes.              |
   | ----------------------------------------------------------------- |
   | It would be a pain to eliminate the <key, value> pair from the    |
   | multimap (free_list) using the key, since the key is not unique!  |
   | This is the main reason for introducing the Qubit_cookies.        |
   | Erasing the object pointed to by the cookie (an iterator to an    |
   | element of the free list) is accomplished by the set_cookie()     |
   | function, without further pain for us, but we must take care to   |
   | eliminate the element from the multimap before. We also have to   |
   | update the set of available sizes.                                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jan 2001         |
   ********************************************************************* */
inline void Qaddress_manager::free_erase(const main_iterator a_segment) {
  /* eliminate the <key, data> pair from the free list, using the only
     available efficient algorithm (an explicit cast is required however). */
  free_list.erase(*(Qubit_cookie_of<free_list_type> *)a_segment->get_cookie());
  /* eliminate the dynamically allocated iterator. */
  a_segment->set_cookie(NULL);
  /* now go to the map of sizes for free segment; since this segment was
     suppose to be in the free list, the size map must contain an entry
     which is indexed with its size: decrease the usage count for this 
     entry by one unit. If it drops to zero, remove the entry from the map. */
  if ((--size_map[a_segment->size()]) == 0)
    size_map.erase(a_segment->size());
}

/* *********************************************************************
   | This method checks that the address manager is ready to be closed |
   | that is it is not managing any address. This is as easy as chek-  |
   | ing that the main qubit list has only one segment and that its    |
   | usage count is zero. Equivalently we could check that the busy    |
   | list is empty.                                                    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 20 Jan 2001         |
   ********************************************************************* */
inline bool Qaddress_manager::is_free(void) const {
  /* the simplest check is the busy list size (hope debug was good ...) */
  return busy_list.empty();
}

#endif  // __QUANTUM_REGISTER_ADDRESS_MANAGER_INLINES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
