/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <cfloat>               // for DBL_MANT_DIG
#include <qntlib_bigint.h>      // declaration for the big integer class

/* *********************************************************************
   | This method accepts a floating point number and converts it into  |
   | a big integer object. The fractional part is discarded.           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Mar 2003         |
   ********************************************************************* */
big_int big_int::from_double(double x) {
  /* the big integer to be returned at the end. Be sure that the internal
     structure is completely empty (by default it containes one basic_int
     object initialiased to zero. */
  big_int result = 0; result.drop(1);
  /* this is the size of the range for a basic_int, expressed as a double. */
  const double radix = ldexp(1.0, bits_chunk);
  /* remove the fractional part of the passed floating point number */
  modf(x, &x);
  /* cut x into pieces and save them in the basic_int container. */
  while (x != 0.0)
    result.addcarry((basic_int)ldexp(modf(x / radix, &x), bits_chunk));
  /* return the result. */
  return result;
}

/* *********************************************************************
   | This method accepts a big integer and converts it into a floating |
   | point number. Since there is a fixed number of digits in a floa-  |
   | ting point, it may lose the least significant digits.             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Mar 2003         |
   ********************************************************************* */
double big_int::to_double(const big_int &i) {
  /* the floating point to be returned at the end. */
  double result = 0.0;
  /* this loop implements the conversion. */
  for (const_reverse_iterator element = i.rbegin();
       element != i.rend(); ++element)
    (result = ldexp(result, bits_chunk)) += *element;
  /* return the result. */
  return result;
}

/* *********************************************************************
   | This private inline method transforms (a,b) into (a+b,b), return- |
   | ing the carry (0 or 1). Since the elementary integer type is un-  |
   | signed, the carry can be detected by the condition (a+b) < b.     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
inline big_int::basic_int big_int::sum_and_carry
(basic_int &a, const basic_int b) { return ((b <= (a += b)) ? 0 : 1); }

/* *********************************************************************
   | This private inline method transforms (a,b) into (a-b,b), return- |
   | ing the borrow (0 or 1). Since the elementary integer type is un- |
   | signed, the borrow can be detected by the condition that (a-b) is |
   | not greater or equal to (-b), provided that b is not zero.        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
inline big_int::basic_int big_int::subtract_and_borrow
(basic_int &a, const basic_int b) { return ((b && (a -= b) >= (-b)) ? 1 : 0); }

/* *********************************************************************
   | This private inline method transforms (a,b) into (a*b,b), return- |
   | ing the carry (which can be almost any basic_int number). In order|
   | to detect the carry, it is necessary to work with half the number |
   | of qubits in a basic_int at a time; in fact, a multiplication of  |
   | two m-digit numbers can fit in a 2m-digit number.                 |
   | ----------------------------------------------------------------- |
   | If we let Q be 2^half_bits_chunk, i.e. half of the range for a    |
   | basic_int number, and a = a0 + a1*Q, b = b0 + b1*Q, then the      |
   | product a*b is a0*b0 + a0*b1*Q + a1*b0*Q + a1*b1*Q^2. The four    |
   | products ai*bj can be easily calculated, and sum_and_carry does   |
   | the rest. I don't know how to optimise it better.                 |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
inline big_int::basic_int big_int::multiply_and_carry
(basic_int &a, const basic_int b) {
  /* calculate a0*b0, a0*b1, a1*b0 and a1*b1 which is saved in "carry". */
  basic_int p00   = low_part(a)  * low_part(b);
  basic_int p01   = low_part(a)  * high_part(b);
  basic_int p10   = high_part(a) * low_part(b);
  basic_int carry = high_part(a) * high_part(b);
  /* sum al the pieces and all the carries. */
  carry += sum_and_carry((a = p00), (low_part(p01) << half_bits_chunk));
  carry += sum_and_carry( a       , (low_part(p10) << half_bits_chunk));
  carry += (high_part(p01) + high_part(p10));
  return carry;
}

/* *********************************************************************
   | This private method implements the multiplication of a whole big  |
   | integer times a basic integer, (result,i) --> (result*i,i), and   |
   | then it returns the carry. It repeatedly calls the inline methods |
   | "sum_and_carry" and "multiply_and_carry".                         |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int::basic_int big_int::multiply_row_and_carry
(big_int &result, const basic_int i) {
  basic_int carry = 0, new_carry = 0;
  /* loop on all elements in the passed big integer. */
  for (iterator element = result.begin(); element != result.end(); ++element) {
    /* basic_int times basic_int multiplication plus carry. */
    new_carry  = multiply_and_carry(*element, i);
    /* add the previous carry, then update it. */
    new_carry += sum_and_carry(*element, carry);
    carry = new_carry;
  }
  /* return the overall carry. */
  return carry;
}

/* *********************************************************************
   | This private method calculates an estimate of the ratio of two    |
   | big integers, provided that it can fit in a basic integer. This   |
   | method is used mainly by the division engine.                     |
   | ----------------------------------------------------------------- |
   | The approximation works by building a floating point "copy" of    |
   | the integers and then converting the ratio into a basic_int.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int::basic_int big_int::ratio_estimate(const big_int &a, const big_int &b){
  double x = 0.0, y = 0.0;
  const_reverse_iterator e;
  /* this variable is the max value for a basic_int + 1. */
  double radix = ldexp(1.0, bits_chunk);
  /* convert the big integers a,b into the floating points x,y */
  for (e = a.rbegin(); e != a.rend(); ++e) (x *= radix) += *e;
  for (e = b.rbegin(); e != b.rend(); ++e) (y *= radix) += *e;
  /* convert x/y into a basic_int and return it. */
  return basic_int(x/y);
}

/* *********************************************************************
   | This is the comparison operator. If the two big integers have a   |
   | different number of digits, it returns immediately "false", other-|
   | wise it compares all the digits from the most significant ones to |
   | the least significant ones.                                       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
bool big_int::operator==(const big_int &i) const {
  /* compare the size of the two integers. Return immediately if they
     differ. Of course, this means that we should take care that there
     are no leading zeroed digits. */
  if (size() != i.size()) return false;
  /* optimisation (?) for the case of a single basic integer. */
  if (size() == 1 && front() != i.front()) return false;
  /* compare all the digits from the most significant ones downwards. */
  for (const_iterator e1 = begin(), e2 = i.begin();
       e1 != end(); ++e1, ++e2) if (*e1 != *e2) return false;
  /* if you get here, the two integers are definitely equal. */
  return true;
}

/* *********************************************************************
   | This is the "less than" operator. It first try to judge using the |
   | length of the big integers. If the size is the same, one must     |
   | resort to analyse the digits one by one, starting from the most   |
   | significant ones.                                                 |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
bool big_int::operator<(const big_int &i) const {
  /* if the two big integers differ in size, then it is easy to determine
     wether this is less than i or not. Of course, this means that we
     should take care that there are no leading zeroed digits. */
  if (size() != i.size()) return (size() < i.size());
  /* compare all the digits from the most significant ones downwards. */
  for (const_reverse_iterator e1 = rbegin(), e2 = i.rbegin();
       e1 != rend(); ++e1, ++e2) if (*e1 != *e2) return (*e1 < *e2);
  /* if you get here, then the two numbers where equal. */
  return false;
}

/* *********************************************************************
   | This is the left-shift-and-replace-operator. It introduces "i"    |
   | zeroed bits in the least significant positions of the big integer,|
   | replacing the previous value. Since the size of the bit sequence  |
   | is unlimited in this class, this operator loses no information at |
   | all. This shift is equivalent to a multiplication (*=) times 2^i. |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Mar 2003         |
   ********************************************************************* */
big_int &big_int::operator<<=(unsigned long i) {
  /* "i" can be larger than the number of bits in a basic_int (which is
     bits_chunk). In general the whole shift operation will be equivalent
     to a shift by "offset" followed by the insertion of "new_chunks"
     basic_ints at the least significant border of the bit sequence. */
  unsigned long new_chunks = i / bits_chunk;
  unsigned long offset     = i % bits_chunk;
  /* two carry variables for the "offset" shift. */
  basic_int carry = 0, new_carry = 0;
  /* shift each element of the big_int and save the carries. Since,
     if the value of the right operand is negative or greater than or
     equal to the width in bits of the promoted left operand, the behavior
     is undefined, we must check offset for not being zero. */
  if (offset)
    for (iterator element = begin(); element != end(); ++element) {
      new_carry = (*element >> (bits_chunk - offset));
      (*element <<= offset) |= carry;
      carry = new_carry;
    }
  /* if carry != 0, extend the bit sequence. */
  if (carry) addcarry(carry);
  /* now insert the "new_chunks" zeroed basic_ints at the
     beginning and return the current big integer. */ 
  return shift(new_chunks);
}

/* *********************************************************************
   | This is the right-shift-and-replace-operator. It eliminates "i"   |
   | bits from the least significant positions of the big integer,     |
   | replacing the previous value. The "i" least significant bits are  |
   | therefore lost. This shift is equivalent to an integer division   |
   | (/=) by 2^i.                                                      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Mar 2003         |
   ********************************************************************* */
big_int &big_int::operator>>=(unsigned long i) {
  /* "i" can be larger than the number of bits in a basic_int (which is
     bits_chunk). In general the whole shift operation will be equivalent
     to a shift by "offset" followed by the deletion of "drop_chunks"
     basic_ints at the least significant border of the bit sequence. */
  unsigned long drop_chunks = i / bits_chunk;
  unsigned long offset      = i % bits_chunk;
  /* two carry variables for the "offset" shift. */
  basic_int carry = 0, new_carry = 0;
  /* shift each element of the big_int and save the carries. Contrary
     to the left shift case, we must use reverse iterators here. Since,
     if the value of the right operand is negative or greater than or
     equal to the width in bits of the promoted left operand, the behavior
     is undefined, we must check offset for not being zero. */
  if (offset)
    for (reverse_iterator element = rbegin(); element != rend(); ++element) {
      new_carry = *element << (bits_chunk - offset);
      (*element >>= offset) |= carry;
      carry = new_carry;
    }
  /* now remove "drop_chunks" basic_ints at the
     beginning and return the current big integer. */ 
  return drop(drop_chunks);
}

/* *********************************************************************
   | This is the "sum-and-update" method, which replaces the current   |
   | big integer with its value plus i. The algorithm is very simple,  |
   | all the digits are summed and the carries are propagated. It is   |
   | possible that the size of *this is increased by one. The return   |
   | value is the modified *this.                                      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int &big_int::operator+=(const big_int &i) {
  /* if the size of *this is smaller than the size of i, add as many
     leading zeroes as necessary to fill the gap. */
  long length_mismatch = (i.size() - size());
  if (length_mismatch > 0) extend(length_mismatch);
  /* loop over all digits with the same position, calculate the sums
     and propagate the carries. Look out not to try to read past the
     end of the passed addend. */
  basic_int carry = 0;
  const_iterator element_2 = i.begin();
  for (iterator element_1 = begin(); element_1 != end(); ++element_1) {
    carry = sum_and_carry(*element_1, carry);
    if (element_2 != i.end()) {
      carry += sum_and_carry(*element_1, *element_2);
      ++element_2; }
  }
  /* if we still have a carry, push it into a new leading digit. */
  if (carry) addcarry(carry);
  /* I think that we don't need to call trim() here, since if all the
     addends didn't have leading zeroes, we don't have either; so,
     simply return the current big integer. */
  return *this;
}

/* *********************************************************************
   | This is the "subtract-and-update" method, which replaces the cur- |
   | rent big integer with its value minus i. The algorithm is specular|
   | to that for the addition, but we have borrows here, not carries.  |
   | The algorithm can fail if the current value is less than i (this  |
   | throws an exception of type invalid_bigint_subtraction).          |
   | Care is taken that no leading zeroes are left in the result.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int &big_int::operator-=(const big_int &i) {
  /* if we have less digits in *this than in i, the subtraction
     is impossible and an exception should be thrown. */
  if (i.size() > size()) throw invalid_bigint_subtraction();
  /* loop over the pairs of corresponding digits in the two integers.
     Since we reached this point, we know that the size of *this is
     greater or equal to the size of i. */
  basic_int borrow = 0;
  const_iterator element_2 = i.begin();
  for (iterator element_1 = begin(); element_1 != end(); ++element_1) {
    /* subtract the old borrow from the current digit of *this. */
    borrow = subtract_and_borrow(*element_1, borrow);
    /* if there is a corresponding digit in i, subtract also that. */
    if (element_2 != i.end()) {
      borrow += subtract_and_borrow(*element_1, *element_2);
      ++element_2; }
  }
  /* if we still have a borrow, we have a problem ... */
  if (borrow) throw invalid_bigint_subtraction();
  /* be sure that there are no leading zeros and return the result. */
  return trim();
}

/* *********************************************************************
   | This is the multiplication operator. The algorithm is trivial,    |
   | the current big integer is multiplied by each digit in the passed |
   | factor, and the results, once appropriately shifted, are summed.  |
   | Care is taken that no leading zeroes are left in the result.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int big_int::operator*(const big_int &i) const {
  /* this is the result to be returned at the end. */
  big_int result = 0;
  /* loop over all the digits of the passed factor. */
  for (const_iterator element = i.begin(); element != i.end(); ++element) {
    /* load the current value in the product. */
    big_int product = *this;
    /* replace product with product*i and save the carry. */
    basic_int carry = multiply_row_and_carry(product, *element);
    /* push the carry in the product if necessary. */
    if (carry) product.addcarry(carry); 
    /* shift the intermediate result appropriately and
       sum it to the overall product. */
    result += product.shift(element - i.begin());
  }
  /* eliminate the possible leading zeroes (this happens if i==0)
     and return the overall product. */
  return result.trim();
}

/* *********************************************************************
   | The integer division operator. This calls the "division" method   |
   | and then discards the reminder.                                   |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int big_int::operator/(const big_int &i) const {
  big_int result;
  big_int reminder;
  division(i, result, reminder);
  return result;
}

/* *********************************************************************
   | The reminder operator. This calls the "division" method and then  |
   | discards the integer division result.                             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int big_int::operator%(const big_int &i) const {
  big_int reminder;
  big_int result;
  division(i, result, reminder);
  return reminder;
}

/* *********************************************************************
   | This is the engine for both operator/ and operator%. It calcula-  |
   | tes the integer ratio and the reminder of the current big integer |
   | with respect to "i". The algorithm fails if i is zero.            |
   | ----------------------------------------------------------------- |
   | The algorithm calculates one digit of the result at a time. This  |
   | digit is estimated by "ratio_estimate" and then corrected by      |
   | means of multiplications and comparisons. This is probably not a  |
   | very fast algorithm, but I don't have time now to write it better.| 
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
void big_int::division(const big_int &i,
		       big_int &result, big_int &reminder) const {
  /* if i==0, the division is impossible, throw an exception. */
  if (i == 0) throw invalid_bigint_division();
  /* if i is bigger than the current big integer, the result is
     zero and the reminder is simply *this. */
  if (i > *this) { result = 0; reminder = *this; return; }
  /* if the current big integer has size 1, we are left with
     the standard integer division between basic_ints. */
  if (size() == 1) {
    result   = front()/i.front();
    reminder = front()%i.front(); return; }
  /* this is the general case. note that size() > i.size(). We are going
     to calculate one digit at a time (which will then be pushed into
     "result"), while "reminder" is the running reminder. */
  result   = 0;
  reminder = *this;
  /* calculate the difference of digits between *this and i. Then drop
     the necessary number of digits from "reminder" so that it has the
     same size as i. */
  long digit_mismatch = this->size() - i.size();
  reminder.drop(digit_mismatch);
  /* digit_mismatch is a counter which tells us if we have
     calculated all the digits of the ratio or not .*/
  while (digit_mismatch >= 0) {
    /* q is an approximation of (reminder/i). */
    basic_int q = ratio_estimate(reminder, i);
    /* adjust q, so that i*q is the largest possible integer which
       is less than reminder. Then set reminder --> reminder - i*q. */
    big_int i_times_q = i*q;
    while (i_times_q > reminder) { i_times_q -= i; --q; }
    reminder -= i_times_q;
    while (reminder > i) { reminder -= i; ++q; }
    /* insert one digit more in the result and reduce the mismatch. */
    result.insert(result.begin(), 1, q);
    --digit_mismatch;
    /* take one of the digits which were dropped from *this at the
       beginning and insert it back into "reminder" (be sure however
       that such digit exists, i.e. digit_mismatch must be >= 0. */
    if (digit_mismatch >= 0)
      reminder.insert(reminder.begin(), 1, (*this)[digit_mismatch]);
    /* be sure we don't have leading zeroes. */
    reminder.trim();
  }
  /* be sure we don't have leading zeroes, nor in the
     reminder nor in the integer division result. */
  result.trim();
  reminder.trim();
}

/* *********************************************************************
   | This method returns the value of (*this)*fpoint mod 1 as a double.|
   | It works by converting the mantissa of fpoint into a multipreci-  |
   | sion integer and subsequently running the multiplication and the  |
   | extraction of the modulus. In this way, when the big integer is   |
   | converted back into a double, only the least significant bits of  |
   | the mantissa are lost.                                            | 
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      31 Mar 2003         |
   ********************************************************************* */
double big_int::mantissa_of_multiplication_by(double fpoint) const {
  /* since our big integers are unsigned, it is better to work with a
     positive floating point integer. Therefore, store the sign and
     take the absolute value for "fpoint". */
  bool is_negative = (fpoint < 0);
  if (is_negative) fpoint = - fpoint;
  /* The frexp function express fpoint as A*2^E, where A is a fraction
     in [0.5,1[ and E is an exponent. We are interested in E in order
     to estimate the magnitude of fpoint. */
  int exponent;
  frexp(fpoint, &exponent);
  /* In the binary representation of fpoint there is a mantissa with
     DBL_MANT_DIG binary digits. Therefore, we get exactly an integer
     number if we multiply fpoint times exponent_shift. */
  int exponent_shift = (1 - exponent + DBL_MANT_DIG);
  fpoint = ldexp(fpoint, exponent_shift);
  /* transform fpoint into a big integer (without loosing precision). 
     In this way we have: integer = fpoint * 2^exponent_shift. */
  big_int integer = big_int::from_double(fpoint);
  /* calculate the range of "integer" which now corresponds to the
     fractional part, i.e. 2^exponent_shift. */
  big_int significant = 1;
  significant <<= exponent_shift;
  /* multiply "integer" times *this and then take the reminder with
     respect to "significant". This corresponds to the transformation:
     fpoint --> fpoint * 2^exponent_shift * (*this) mod 2^exponent_shift
     that is: fpoint --> ([fpoint * (*this)] mod 1) * 2^exponent_shift. */
  integer *= *this;
  integer %= significant;
  /* transform back the big integer into a floating point value. This
     can affect the value, since the big integer has a multiprecision
     representation, while the double contains only DBL_MANT_DIG significant
     digits. This, however, now affects only the least significant bits,
     since we already calculated the modulus with respect to 1. */
  double result = big_int::to_double(integer);
  /* Now put back the correct magnitude for the result. The overall tran-
     sformation now corresponds to: fpoint --> ([fpoint * (*this)] mod 1) */
  result = ldexp(result, - exponent_shift);
  /* restore the sign, if necessary; then return the result. */
  if (is_negative) result = - result;
  return result;
}

/* *********************************************************************
   | This method returns an array of digits representing the current   |
   | integer in radix-10. The first element in the returned array      |
   | corresponds to the least significant digit. If there are at least |
   | two digits in the array, the last one is guaranteed to be != 0.   |
   | ----------------------------------------------------------------- |
   | The conversion strategy is quite simple, and the run-time should  |
   | be quadratic in the number of bits of the big integer. The bit    |
   | string of *this is gradually shifted into the "digits" container, |
   | and each time the part which fits inside an element of the conta- |
   | iner execeedes the radix, it is carried to the next element.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
big_int::digit_container big_int::get_digits(void) const {
  /* we can calculate in advance the number of decimal digits
     which are needed for representing the big integer value. */
  const unsigned long radix = 10;
  const unsigned long dec_digits = (unsigned long)(1 + M_LN2*M_LOG10E*bits());
  /* this is the container to be returned at the end. */
  digit_container digits(dec_digits, 0);
  /* for each group of qubits in the big integer, shift it into the
     digit container and convert it. Remember that groups of qubits
     (i.e. basic_ints) are stored internally from the least significant
     one to the most significant one, so we must use reverse iterators
     here. Shifting a whole basic_int is MUCH faster than shifting
     bits individually. */
  for (const_reverse_iterator element = rbegin(); element!=rend(); ++element) {
    /* take the low and high part of the element to be shifted. */
    basic_int carry1 = high_part(*element);
    basic_int carry0 = low_part(*element);
    /* this variable is used and reused during the shift. It is in
       general wider than an element of "digits", so that it can
       contain both the value and the carry for partial shifts. */
    basic_int meta_digit = 0;
    /* this inner loop shifts all the digits in the "digits" container.
       The shift of a whole basic_int is done in two passes, moving
       half_bits_chunk at a time. */
    for (digit_container::iterator digit = digits.begin();
	 digit != digits.end(); ++digit) {
      /* save the current digit (< radix) in meta_digit, then shift
	 it by half_bits_chunk and add the old level-1 carry. Last,
	 remember to reset the level-1 carry. */
      ((meta_digit = *digit) <<= half_bits_chunk) += carry1;
      carry1 = 0;
      /* if a digit has overflowed (i.e. it is bigger than or equal to
	 the radix0) we recast it into the correct ranged [0, radix[
	 and save the carry in the level-1 carry variable. */
      if (meta_digit >= radix) {
      	carry1 = (meta_digit / radix);
	meta_digit %= radix; }
      /* second pass for the shift: shift and add the level-0 carry. 
	 Then reset the level-0 carry. */
      (meta_digit <<= half_bits_chunk) += carry0;
      carry0 = 0;
      /* save back the content of meta_digit in digit. */
      *digit = meta_digit;
      /* again, if an overflow took place, recast the digit and 
	 save the level-0 carry. */
      if (meta_digit >= radix) {
	carry0 = (meta_digit / radix);
	*digit = (meta_digit % radix);
      }
    }
  }
  /* be sure to eliminate leading zeroes (but leave at least one digit). */
  while (digits.size() > 1 && digits.back() == 0) digits.pop_back();
  /* return the digit container. */
  return digits;
}

/* *********************************************************************
   | This is the output operator for a big integer. It does nothing    |
   | more than calling the get_digits() method and printing the retur- |
   | ned array. We need to cast the elements of the array to ints in   |
   | order to print the digit values (and not the corresponding ASCII  |
   | table characters).                                                |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const big_int &i) {
  big_int::digit_container digits = i.get_digits();
  for (big_int::digit_container::reverse_iterator digit = digits.rbegin();
       digit != digits.rend(); ++digit) os << (int)(*digit);
  return os;
}
