/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __TIME_SLICE_INLINES_
#define __TIME_SLICE_INLINES_

#include <qop_slice.h>                          // class declaration
#include <qinterface_base.h>                    // for code generation

/* *********************************************************************
   | This method returns the number of lists in the index_lists con-   |
   | tainer. The list container is a vector, therefore calculating its |
   | size takes constant time.                                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Jan 2002         |
   ********************************************************************* */
inline Qop_slice::index_type Qop_slice::get_list_number(void) const {
  return index_lists.size();
}

/* *********************************************************************
   | This method returns a constant reference to the N-th location     |
   | list, where N is the passed argument (for inspections).           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Jun 2001         |
   ********************************************************************* */
inline const Qop_slice::list_type &Qop_slice::get_list(index_type index) const{
  return index_lists[index];
}

/* *********************************************************************
   | This method returns a mutable reference to the N-th location list |
   | where N is the passed argument (must be protected !).             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 12 Sep 2001         |
   ********************************************************************* */
inline Qop_slice::list_type &Qop_slice::get_list(index_type index) {
  return index_lists[index];
}

/* *********************************************************************
   | This method returns true if the passed slice and the current      |
   | slice are associated to two basic quantum gates which are one     |
   | the adjoint of the other. The simplest gates are self-adjoint,    |
   | hence a default method can be supplied which only tests the gate  |
   | type. The method will however be overridden by more complex time  |
   | slices (therefore it must be declared virtual).                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 05 Dec 2001         |
   ********************************************************************* */
inline bool Qop_slice::has_adjoint_basegate(const Qop_slice &a_slice) const {
  return (get_opcode() == a_slice.get_opcode());
}

/* *********************************************************************
   | This method returns the parallelisation index, that is the mini-  |
   | mum number of parallel homogeneous operations that the quantum    |
   | hardware should be capable of in order to run this time slice as  |
   | a real time slice. In this case it is simply the length of the    |
   | first address list (they must all have the same length).          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 27 Sep 2001         |
   ********************************************************************* */
inline Qop_slice::size_type Qop_slice::get_parallelisation(void) const {
  return get_list(0).size();
}

/* *********************************************************************
   | This method "shifts a part of the circuit forward" for a time     |
   | slice. This means that each formal address whose value is less    |
   | than "head" is left unaltered, while the other addresses are      |
   | shifted ahead "jump" positions.                                   |
   | ----------------------------------------------------------------- |
   | This method reduces to "splitting" all the lists independently.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   ********************************************************************* */
inline void Qop_slice::split(size_type head, size_type jump) {
  for (index_type index = 0; index < get_list_number(); ++index)
    get_list(index).run_split(head, jump);
}

/* *********************************************************************
   | This method "inverts the order of a circuit part" for time slice. |
   | This means that each formal address whose value is less than      |
   | "head" or greater or equal to "head+size" is left unaltered, while|
   | the other addresses undergo a mirror simmetry in the central re-  |
   | gion of addresses [head, size[.                                   |
   | ----------------------------------------------------------------- |
   | This method reduces to "inverting" all the lists independently.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   ********************************************************************* */
inline void Qop_slice::invert(size_type head, size_type size) {
  for (index_type index = 0; index < get_list_number(); ++index)
    get_list(index).run_invert(head, size);
}

/* *********************************************************************
   | This method performs an index  permutation for a time slice. This |
   | means that each formal address (say "i" its value) is remapped on |
   | permutation_list[i]. This transformation fails and throws an ex-  |
   | ception of type invalid_translation() if there is no i-th element |
   | in the permutation list (i.e. if it is too short).                |
   | ----------------------------------------------------------------- |
   | This method reduces to "remapping" all the lists independently.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   ********************************************************************* */
inline void Qop_slice::remap(const Qubit_list &permutation_list) {
  for (index_type index = 0; index < get_list_number(); ++index)
    get_list(index).run_remap(permutation_list);
}

/* *********************************************************************
   | This method implements the execution of a time slice. In a few    |
   | words this means we have to collect/calculate all the informa-    |
   | tions about the represented quantum operations and send them to   |
   | the quantum device through some communication channel. Remember   |
   | that these operations are by definition non-blocking.             |
   | ----------------------------------------------------------------- |
   | Coming to details:                                                |
   | a) calculate the real addresses for involved qubits by pairing    |
   |    the formal locations in the slice location list(s) with the    |
   |    real locations in the register location list.                  |
   | b) extract the operation code and the optional parameters.        |
   | c) send all data through the quantum device interface             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 18 Jul 2001         |
   ********************************************************************* */
inline void Qop_slice::operator()(const Qreg &a_register) const {
  get_quantum_interface().submit_timeslice
    (get_opcode(), get_parameter(), translate(a_register));
}

#endif // __TIME_SLICE_INLINES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
