/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iomanip>                          // C++ I/O manipulators
#include <qop_slicelist.h>                  // class declaration
#include <qop_summary.h>                    // a database for base gates

/* *********************************************************************
   | This method will return the size of the longest single address    |
   | list in a single time slice in this slice list. This size is the  |
   | parallelisation degree for the slice list (see the analogous      |
   | method in the QSlice class for further information).              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 27 Sep 2001         |
   ********************************************************************* */
Qop_slicelist::size_type Qop_slicelist::get_parallelisation(void) const {
  /* initialise with a dummy value. */
  size_type max_value = 0, parallelisation;
  /* scan the slice list. */
  for (const_iterator a_slice = begin(); a_slice != end(); ++a_slice) {
    /* get the parallelisation for this slice. */
    parallelisation = (*a_slice)->get_parallelisation();
    /* update the max value if necessary. */
    if (parallelisation > max_value) max_value = parallelisation;
  }
  /* return the max parallelisation. */
  return max_value;
}

/* *********************************************************************
   | This method returns the highest occupation in any slice of the    |
   | current slice list, calling the get_occupation() method for each  |
   | owned slice. Note that ancillae are not taken into account, i.e.  |
   | if the maximum return value from the calls to get_occupation()    |
   | for slices is 7 and get_ancillae() reports 2, this method returns |
   | 5, which is the minimum size for a register which is to be fed    |
   | into this slice list at execution time (since the ancillae are    |
   | supplied automatically).                                          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Oct 2002         |
   ********************************************************************* */
Qop_slicelist::size_type Qop_slicelist::get_occupation(void) const {
  /* reset the highest occupation sofar. */
  size_type highest_occupation = 0;
  /* iterate through all slices in this list */
  for (const_iterator a_slice = begin(); a_slice != end(); ++a_slice) {
    /* get the occupation for this slice. */
    size_type slice_occupation = (*a_slice)->get_occupation();
    /* update the overall value if necessary */
    if (slice_occupation > highest_occupation)
      highest_occupation = slice_occupation;
  }
  /* return the highest occupation minus the number of ancillae.
     If the result is negative, return zero (this means, correctly,
     that no qubit is explicitely requested in the passed register at
     execution time; this means also, though, that the circuit acts
     only on the ancilla part, which is of dubious utility. */
  if (highest_occupation > get_ancillae())
    return (highest_occupation - get_ancillae()); else return 0;
}

/* *********************************************************************
   | This is the assignment operator for Qop_slicelists. The current   |
   | list is cleared and all the elements from the passed list are     |
   | copied into it. The number of ancillae is also copied.            |
   | ----------------------------------------------------------------- |
   | More precisely, since Qop_slices cannot be instantiated, this     |
   | method clones the supplied slices using the most derived class    |
   | (reading it from the virtual table) and saves the pointers cast   |
   | to base class pointers.                                           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Sep 2002         |
   ********************************************************************* */
Qop_slicelist &Qop_slicelist::operator=(const Qop_slicelist &a_list) {
  /* flush the current slice list. Reset ancillae to zero (paranoia). */
  clear(); set_ancillae(0);
  /* now loop in the passed list and clone + append each slice. */
  for (const_iterator a_slice = a_list.begin();
       a_slice != a_list.end(); ++a_slice)
    push_back((*a_slice)->clone());
  /* set the number of ancillae. */
  set_ancillae(a_list.get_ancillae());
  /* return the current object, as usual. */
  return *this;
}

/* *********************************************************************
   | This method cannibalises the content of the passed list and can   |
   | run an automatic simplification routine. The new elements are     |
   | added at the end of the list because it is more useful for the    |
   | composition syntax: U1 & U2 means execute U1 first, then U2.      |
   | The passed list is empty after this method has been run.          |
   | ----------------------------------------------------------------- |
   | Insert here an automatic simplification routine base on border    |
   | simplification; in this way it is possible not to export an ite-  |
   | rator to a slice outside. Run the simplification only if there    |
   | was already a non emtpy list.                                     |
   | ----------------------------------------------------------------- |
   | Adjust the two lists if they don't have the same number of qubits |
   | declared as ancillae: this requires that all addresses of at most |
   | one list are shifted forward.                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Dec 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      31 Oct 2002         |
   ********************************************************************* */
void Qop_slicelist::splice(Qop_slicelist &a_list) {
  /* read the number of qubits declared as ancillae
     from each list and calculate the maximum value. */
  size_type ancillae_1   = get_ancillae();
  size_type ancillae_2   = a_list.get_ancillae();
  size_type max_ancillae = (ancillae_1 > ancillae_2 ? ancillae_1 : ancillae_2);
  /* if the current list has less ancillae than "a_list", adjust it */
  if (ancillae_1 < max_ancillae) offset(max_ancillae - ancillae_1);
  /* the opposite situation is also possible ... */
  if (ancillae_2 < max_ancillae) a_list.offset(max_ancillae - ancillae_2);
  /* fix the number of qubits declared as ancillae */
  set_ancillae(max_ancillae);
  /* this macro acts the basic splicing action */
#define DO_REAL_SPLICE slices().splice(end(), a_list.slices())
  /* if the current list is empty, splice the content of the passed 
     list and return without further processing. */
  if (empty()) { DO_REAL_SPLICE; return; }
  /* if it is non trivial, save an iterator to the last element. */
  iterator left_border = end();
  --left_border;
  /* splice the content of the passed list into the current one. This
     operation takes constant time and destroys the passed list. */
  DO_REAL_SPLICE;
  /* run the border simplification. */
  simplify(left_border);
}

/* *********************************************************************
   | This method runs "operator()" on each slice in the list, passing  |
   | the given register and the required number of ancilla qubits by   |
   | reference. Each specific slice knows how to manage its execution. |
   | ----------------------------------------------------------------- |
   | (06 Sep 2001) S.Bettelli; introduction of ancilla qubits. Each    |
   | slice has now a field which remembers how many ancilla qubits it  |
   | needs (this field also affects the swap() and split() operations).|
   | This method will spawn fresh quantum registers (initialised to    |
   | zero) when necessary and prepend them to the "regular" user regi- |
   | ster before they are passed to the slice. Already allocated an-   |
   | cillas are retained through different slices when possible (this  |
   | is just to make them useful!). Use of ancillae is reserved for    |
   | library functions, which must take care to uncompute everything   |
   | at the end and leave the qubits in the |0> state, otherwise the   |
   | slice list could not be adjoint!                                  |
   | ----------------------------------------------------------------- |
   | (30 Oct 2002) S.Bettelli; handling of ancilla qubits has been     |
   | changed: currently there is global counter for ancillae in the    |
   | slice list, not a counter for each single time slice. Therefore,  |
   | if ancilla qubits are requested, they are allocated at the begin- |
   | ning of the routine and released at the end.                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 18 Jul 2001         |
   ********************************************************************* */
void Qop_slicelist::operator()(const Qreg &a_register) const {
  /* a macro for the actual execution cycle. */
#define EXECUTE_ON(R) for (const_iterator a_slice = begin(); \
                      a_slice != end(); ++a_slice) (**a_slice)(R)
  /* if there are no ancillae, simply execute each time slice
     on the passed register (this avoids dummy copies). */
  if (get_ancillae() == 0) { EXECUTE_ON(a_register); return; }
  /* if there is any ancilla, create a new register containing
     the additional qubits at the beginning and the passed register
     after it, then execute each time slice on the ancilled register */
  Qreg ancilled_register = (Qreg(get_ancillae()) & a_register);
  EXECUTE_ON(ancilled_register);
}

/* *********************************************************************
   | This method performs an index permutation for a slice list. This  |
   | means that, for each time slice, each formal address (say "i" its |
   | value) is remapped on permutation_list[i]. This transformation    |
   | fails and throws an exception of type invalid_translation() if    |
   | there is no i-th element in the permutation list.                 |
   | ----------------------------------------------------------------- |
   | Remember that the programmer does not know about ancilla qubits,  |
   | so those lines which are labelled as ancillae (whose index is     |
   | less than get_ancillae()) must NOT be remapped. This implies a    |
   | modification of the passed list. An optimisation is provided for  |
   | the frequent situation in which we have no ancillae.              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      31 Oct 2002         |
   ********************************************************************* */
void Qop_slicelist::remap(const Qubit_list &permutation_list) { 
  /* a macro for the actual remapping cycle. */
#define REMAP_ON(L) for (iterator a_slice = begin(); \
                    a_slice != end(); ++a_slice) (*a_slice)->remap(L)
  /* If there are no ancillae, we don't need any copy. Simply remap
     each time slice with the passed argument and return. */
  if (get_ancillae() == 0) { REMAP_ON(permutation_list); return; }
  /* Remember to skip the ancilla qubits. In primis, this means that
     all the elements in the permutation list must be increased with
     the number of ancillae. Curiously enough, this can be done with
     run_split. */
  Qubit_list permutation_copy(permutation_list);
  permutation_copy.run_split(0, get_ancillae());
  /* Then create the first segment of the new permutation, which must
     be the identical permutation for the first get_ancillae() qubits. */
  Qubit_list new_permutation(0, get_ancillae());
  /* join the two parts of the new permutation list. */
  new_permutation.join(permutation_copy);
  /* remap all time slices appropriately, using the new permutation
     in order to be compliant with ancilla requirements. */
  REMAP_ON(new_permutation);
}

/* *********************************************************************
   | This method calculates the controlled version of the current list |
   | once it is given a circuit which calculates the control qubits.   |
   | Indeed it uses the control() method from each of the time slices  |
   | to generate the controlled list. control() assumes that the "h"   |
   | needed control lines have formal addresses in the range [k, k+h[, |
   | where "k" is the number of already set ancillae. Remember that a  |
   | single time slice, when controlled, tipically explodes into a     |
   | list, not a single slice). The slice list which is returned has   |
   | the number of ancillae increased by additional_ancillae.          |
   | ----------------------------------------------------------------- |
   | The control_circuit is offset to be compatible with the ancillae  |
   | already set in the current list. Then, the (copies of) the slices |
   | in the current list are offset by "additional_lines" qubit lines, |
   | thus making enough room for inserting the new ancillae and the    |
   | control register. Remember that this offset takes into account    |
   | those qubit lines which were already declared as ancillae.        |
   | Then, fill a new list with the controlled version of each slice:  |
   | control() will do this for us. Last, uncompute control_circuit    |
   | and update the ancilla count on the new list.                     |
   | ----------------------------------------------------------------- |
   | NOTE: the management of old and new ancilla lines is quite confu- |
   | sed and unsatisfactory here, it should be revised and simplified. |
   | Also, there should be a check that "control_circuit" transforms   |
   | |0>|x> (where the first register has size "additional_ancillae"   |
   | and the second register is prepared in an element of the computa- |
   | tional basis) into |f(x)>|x> (where f() is an integer to integer  |
   | function), otherwise the ancillae are not left into |0> at the    |
   | end, which is a major problem. The solution of this problem has   |
   | something in common with the problem of implementing pseudoclas-  |
   | sical functions.                                                  |
   | ----------------------------------------------------------------- |
   |                                                                   |
   |  old_ancillae         -----U-------   Schema for qubit lines:     |
   |  parallelisation - 1  ---C-X-C*----   AA is the AND circuit       |
   |  control_size - 1     -A-C---C*-A*-   CC is the COPY circuit      |
   |  control_size         -A--------A*-   * means conjugation         |
   |  register size        -----U-------   X-U means ctrl slices of U  |
   |                                           using X as controls.    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Oct 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      01 Feb 2003         |
   ********************************************************************* */
Qop_slicelist Qop_slicelist::controlled(const Qop_slicelist &control_circuit,
					size_type additional_lines,
					size_type additional_ancillae) const {
  /* prepare the list to return here, to help the named
     return value optimization (you never know!). */
  Qop_slicelist controlled_list;
  /* extract the number of ancillae already set in the current circuit. */
  size_type old_ancillae = get_ancillae();
  /* some sanity checks; the number of additional ancillae must be
     strictly less than the size of the passed circuit (zero is ok). */
  if (! (additional_ancillae < additional_lines)) throw invalid_control();
  /* also, the preparation circuit shouldn't have any explicitely
     set auxiliary qubit, for the time being. */
  if (control_circuit.get_ancillae() != 0) throw invalid_control();
  /* Get a copy of the control circuit, offset it to make it "compatible"
     with the current one and copy it into the list to be returned. 
     Use append, not splice, so we will be able to reuse it. */
  Qop_slicelist control_circuit_offset = control_circuit;
  control_circuit_offset.offset(old_ancillae);
  controlled_list.append(control_circuit_offset);
  /* offset a copy of the whole current slice list. The offset() method
     preserves the old ancillae, so we don't care about. */
  Qop_slicelist target_circuit = *this;
  target_circuit.offset(additional_lines);
  /* for each time slice in the target list, calculate its controlled
     version (which is a slice list) and push it into the appropriate
     container. Remember that these slices have no notion of ancillae. */
  for (iterator a_slice = target_circuit.begin();
       a_slice != target_circuit.end(); ++a_slice)
    (*a_slice)->control(&controlled_list, old_ancillae);
  /* we have to uncompute the preparation circuit now. Since we still
     have a copy of it, we can simply adjoin and move it into the current
     list (use splice now, since it is not going to be reused). */
  control_circuit_offset.adjoin();
  controlled_list.append(control_circuit_offset);
  /* Last, but not least, update the number of qubits declared as
     ancillae in the final list. After this modification, it is
     not necessary to supply the ancilla qubits explicitely in the
     input register. */
  controlled_list.set_ancillae(additional_ancillae + old_ancillae);
  /* return the list which has been filled. */
  return controlled_list;
}

/* *********************************************************************
   | This method will print some statistics about the slice list into  |
   | the supplied output stream. This piece of information can be      |
   | prepended to the list operator<< output to complete the simple    |
   | listing of the list content.                                      |
   | ----------------------------------------------------------------- |
   | The listing is currently quite simple. The output is divided in   |
   | two parts. The upper part hosts some global values for the slice  |
   | list: its depth (both when parallelisation of homogeneous opera-  |
   | tions is possible and when it isn't), the parallelisation index,  |
   | the occupation and the number of ancilla qubits.                  |
   | The lower part reports the number of gates which concern each     |
   | qubit line, as a table. The rows of the table represent qubit     |
   | lines (an "A" is prepended to ancilla lines). Each column repre-  |
   | sents a gate type (like Hadamard, Phase ...). Each entry in the   |
   | table is the number of gates of a given type which act on a given |
   | qubit line. The last column is the total (note that multi-qubit   |
   | gates are counted more than once, hence the sum of the totals is  |
   | likely to be larger than the circuit's depth.                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      06 Nov 2002         |
   ********************************************************************* */
void Qop_slicelist::print_statistics(std::ostream &os) const {
  /* a nice separation bar for the different parts. */
  const char separation[] = "=============================================";
  /* calculate the overall number of qubit lines in the slice list. */
  size_type lines = get_ancillae() + get_occupation();
  /* get the number of different basic gates. */
  size_type num_basegates = Qop_summary::get_gates();
  /* calculate the unparallelised depth of the circuit (i.e. the weight
     of each slice is its parallelisation index, not one, just as if 
     the underlying quantum machine could not execute in parallel. */
  size_type depth_unparallelised = 0;
  for (const_iterator slice = begin(); slice != end(); ++slice)
    depth_unparallelised += (*slice)->get_parallelisation();
  /* print into the stream the first part of the statistics. This is
     a general overview of the slice list. It uses two columns. 
     Prepend a separation line to this listing. */
  os << separation
     << "\nDepth (parall.) " << get_size()
     << "\tOccupation      " << get_occupation()
     << "\nDepth (plain)   " << depth_unparallelised
     << "\tAncillae        " << get_ancillae()
     << "\n                "
     << "\tParallelisation " << get_parallelisation();
  /* print out another separation line. */
  os << "\n" << separation << "\n\t| ";
  /* print a table header displaying the available base gates.
     Use only the first letters from each gate name. Also print
     a title for the column of per-line totals. */
  for (size_type i=0; i<num_basegates; ++i)
    os << Qop_summary::get_basegate(i)->get_name().substr(0,4) << ". ";
  os << "\ttot.\n\t|";
  /* now print a table entry for each qubit line. */
  for (size_type line = 0; line < lines; ++line) {
    /* print the line index (starting from zero) and a vertical bar.
       If the line is used as an ancilla, prepend an A. */
    os << "\n  " << ((line < get_ancillae())?'A':' ')
       << std::setw(4) << line << "\t| ";
    /* an array for holding the counts for each type of base gate.
       Be sure each entry is initialised to zero. */
    size_type count[num_basegates];
    for (size_type i=0; i<num_basegates; ++i) count[i] = 0;
    /* for each slice, test if the line index is contained in one
       of its address lists. If the test is positive, increase the
       count concerning the slice's base gate type. */
    for (const_iterator slice = begin(); slice != end(); ++slice)
      if ((*slice)->is_referencing(line))
	++count[(*slice)->get_index()];
    /* print the previously found result. Calculate the line grand
       total in the meanwhile and print it at the end. */
    size_type count_total = 0;
    for (size_type i=0; i<num_basegates; ++i) {
      os << std::setw(4) << count[i] << "  ";
      count_total += count[i];
    }
    os << '\t' << std::setw(4) << count_total << "  ";
  }
  /* print a closing separation line. */
  os << '\n' << separation << '\n';
}

/* *********************************************************************
   | This output function will display the content of the list, for    |
   | debugging purposes. The header line summarises a list identifier  |
   | (the address, sic!), the number of time slices in the list, the   |
   | number of qubit lines (both explicitely requested or declared as  |
   | ancillae) and the parallelisation index. Then each time slice is  |
   | output on a separate line using its own output function.          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 24 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      06 Nov 2002         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qop_slicelist &a_list) {
  /* first, print a list identifier (the pointer, sich..) */
  os << "\tList " << (const void *)&a_list
    /* now print the number of time slices in the list. */
     << " contains " << a_list.slices().size() << " time slice(s)"
    /* then the number of explicitely requested qubits. */
     << " <" << a_list.get_occupation() << " qub.>"
    /* and also the number of ancilla lines. */
     << " <" << a_list.get_ancillae() << " anc.>"
    /* print also the parallelisation index. */
     << " <" << a_list.get_parallelisation() << " parall.>";
  /* This variable is a shortcut for indexing time slices. */
  Qop_slicelist::size_type slice_counter = 0;
  /* Now iterate over all slices and call the appropriate output function */
  for (Qop_slicelist::const_iterator slice = a_list.begin();
       slice != a_list.end(); ++slice, ++slice_counter)
    os << "\n\t  (" << slice_counter << ")\t" << **slice;
  /* last, return the output stream. */
  return os;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
