/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <qprimitives.h>                   // all the class declarations 
#include <qop_primitives.h>                // the primitive slices
#include <qop_builders.h>                  // for specific ctor routines

/* *********************************************************************
   | Constructors for the QHadamard transformation. This Qop primitive |
   | contains a single Qop_slice_Hadamard slice, whose ctor does the   |
   | whole job. The slice is then inserted in the operation list.      |
   | 1) Ctor with 1 list: apply a Hadamard gate on all the elements.   |
   | 2) Ctor with size  : like 1) with list = [0, size[                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */

/* Default ctor (create one slice and append it, list = [0, size[). */
QHadamard::QHadamard(size_type the_size)
  : Qop(new Qop_slice_Hadamard(Qubit_list(0, the_size))) { }
/* Constructor with a preformed list (may throw exception). */
QHadamard::QHadamard(const Qubit_list &the_list)
  : Qop(new Qop_slice_Hadamard(the_list)) { }

/* *********************************************************************
   | This quantum operation applies a "NOT" transformation (the Pauli  |
   | X matrix) on some qubits in its formal register. A "NOT" gate is  |
   | the obvious thing, i.e. it brings |0> to |1> and |1> to |0>.      |
   | NOT isn't a primitive in our approach, it is a three-slice Qop    |
   | were a phase shift is sandwiched with two Hadamard transforms.    |
   | The real construction is delegated to a specific routine (the     |
   | prepended ~ is indeed operator~ for QNot_slices, which returns    |
   | the temporary by non constant reference, so that it can be spli-  |
   | ced instead of copied by the Qop constructor).                    |
   | 1) Ctor with 1 list: apply a NOT gate on all the elements.        |
   | 2) Ctor with size  : like 1) with list = [0, size[                |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 Jun 2002         |
   ********************************************************************* */

/* Default constructor */
QNot::QNot(size_type the_size)
  : Qop(~QNot_slices(Qubit_list(0, the_size))) { }
/* Constructor through preformed list (may throw exception) */
QNot::QNot(const Qubit_list &the_list)
  : Qop(~QNot_slices(the_list)) { }

/* *********************************************************************
   | Constructors for the QPhase transformation. This Qop primitive    |
   | contains a single Qop_slice_Phase slice, whose ctor does the      |
   | whole job. The slice is then inserted in the operation list.      |
   | 1) Ctor with 1 list: apply a QPhase gate on all the elements.     |
   | 2) Ctor with size  : like 1) with list = [0, size[                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */

/* Default ctor (create one slice and append it, list = [0, size[). */
QPhase::QPhase(size_type the_size,
	       ::quantum_phase_power_type the_power,
	       ::quantum_phase_floating_type the_fraction)
  : Qop(new Qop_slice_Phase(Qubit_list(0, the_size),
			    the_power, the_fraction)) { }
/* Constructor with a preformed list (may throw exception). */
QPhase::QPhase(const Qubit_list &the_list,
	       ::quantum_phase_power_type the_power,
	       ::quantum_phase_floating_type the_fraction)
  : Qop(new Qop_slice_Phase(the_list, the_power, the_fraction)) { }

/* *********************************************************************
   | Constructors for the QCondPhase transformation. This Qop primi-   |
   | tive contains a single Qop_slice_CondPhase slice, whose ctor does |
   | the whole job. The slice is then inserted in the operation list.  |
   | 1) Ctor with 2 lists: apply a conditional QPhase gate on the cor- |
   |    responding elements of the two lists (first list is "control") |
   | 2) Ctor with size: like 1) with lists [0, size[ and [size, 2*size[|
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */

/* Default constructor (create one slice and append it). */
QCondPhase::QCondPhase(size_type the_size,
		       ::quantum_phase_power_type the_power,
		       ::quantum_phase_floating_type the_fraction)
  : Qop(new Qop_slice_CondPhase(Qubit_list(0, the_size),
				Qubit_list(the_size, the_size),
				the_power, the_fraction)) { }
/* Constructor with preformed lists (may throw exception). */
QCondPhase::QCondPhase(const Qubit_list &the_controls,
		       const Qubit_list &the_targets,
		       ::quantum_phase_power_type the_power,
		       ::quantum_phase_floating_type the_fraction)
  : Qop(new Qop_slice_CondPhase(the_controls, the_targets,
				the_power, the_fraction)) { }

/* *********************************************************************
   | Constructors for controlled nots (CNOTs). CNOT isn't a primitive  |
   | in our approach, it is a three-slice Qop were a controlled phase  |
   | shift is sandwiched with two Hadamard transformations. The real   |
   | construction is delegated to a specific routine (the prepended ~  |
   | is indeed operator~ for QCnot_slices, which returns the temporary |
   | by non constant reference, so that it can be spliced instead of   |
   | copied by the Qop constructor).                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */

/* Default constructor */
QCnot::QCnot(size_type the_size)
  : Qop(~QCnot_slices(Qubit_list(0, the_size),
		      Qubit_list(the_size, the_size))) { }
/* Constructor, through preformed list (may throw exception) */
QCnot::QCnot(const Qubit_list &the_controls, 
	     const Qubit_list &the_targets)
  : Qop(~QCnot_slices(the_controls, the_targets)) { }

/* *********************************************************************
   | Constructors for the QCondCondPhase transformation. This Qop pri- |
   | mitive does not correspond to any time slice primitive in our     |
   | approach, but it can be easily built with a fixed size circuit    |
   | which is already available as the "control" routine for the sim-  |
   | ply conditioned phase gates (the prepended ~ is indeed operator~  |
   | for QCondCondPhase_slices, which returns the temporary by non     |
   | constant reference, so that it can be spliced instead of copied   |
   | by the Qop constructor).                                          |
   | 1) Ctor with 3 lists: apply a conditional QPhase gate on the cor- |
   |    responding elements of the three lists (first 2 are "control") |
   | 2) Ctor with size: like 1) with lists [0, size[ , [size, 2*size[  |
   |    and [2*size, 3*size[                                           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      02 May 2002         |
   ********************************************************************* */

/* Default constructor (create one slice and append it). */
QCondCondPhase::QCondCondPhase(size_type the_size,
			       ::quantum_phase_power_type the_power,
			       ::quantum_phase_floating_type the_fraction)
  : Qop(~QCondCondPhase_slices(Qubit_list(0, the_size),
			       Qubit_list(the_size, the_size),
			       Qubit_list(2*the_size, the_size),
			       the_power, the_fraction)) { }
/* Constructor with preformed lists (may throw exception). */
QCondCondPhase::QCondCondPhase(const Qubit_list &the_controls_1,
			       const Qubit_list &the_controls_2,
			       const Qubit_list &the_targets,
			       ::quantum_phase_power_type the_power,
			       ::quantum_phase_floating_type the_fraction)
  : Qop(~QCondCondPhase_slices(the_controls_1, the_controls_2, the_targets,
			       the_power, the_fraction)) { }

/* *********************************************************************
   | Constructors for a Toffoli gate. Toffoli is not primitive, it is  |
   | made by two Hadamards which surround a doubly controlled phase    |
   | shift with phase parameter k=1 (a controlled-controlled-Z, which  |
   | isn't primitive either). The real construction is delegated to a  |
   | specific routine (the prepended ~ is indeed operator~ for         |
   | QCnot_slices, which returns the temporary by non constant refe-   |
   | rence, so that it can be spliced instead of copied by the Qop     |
   | constructor).                                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 14 Sep 2001         |
   ********************************************************************* */

/* Default constructor */
QToffoli::QToffoli(size_type the_size) 
  : Qop(~Toffoli_slices(Qubit_list(0, the_size),
			Qubit_list(the_size, the_size),
			Qubit_list(2*the_size, the_size))) { }
/* Constructor, through preformed lists (may throw exception) */
QToffoli::QToffoli(const Qubit_list &the_controls_1,
		   const Qubit_list &the_controls_2,
		   const Qubit_list &the_targets)
  : Qop(~Toffoli_slices(the_controls_1, the_controls_2, the_targets)) { }

/* *********************************************************************
   | Constructors for swap gates. This is like a physical swap gate,   |
   | i.e. the quantum states of the qubits are interchanged; this is   |
   | however implemented by swapping the references to the physical    |
   | locations. The constructor syntax is unusual. Of course, there is |
   | a constructor with two lists (the corresponding elements of which |
   | are to be swapped); but there is also a constructor with a single |
   | list, where the first location is to be swapped with the last one,|
   | the second one with the second last one and so on ... If the list |
   | has an odd number of locations, the "central" qubit is left un-   |
   | touched. The constructor with the_size falls back on the second   |
   | ctor with list = [0, size[.                                       |
   | ----------------------------------------------------------------- |
   | (30 Jun 2002) S.Bettelli. Patched the constructor with size and   |
   | that with a single list, so that they now check that at least two |
   | addresses are present; if this is not true, return the identity.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Jun 2002         |
   ********************************************************************* */

/* Default constructor with size (build up two lists) */
QSwap::QSwap(size_type the_size)
  : Qop((the_size < 2) ? NULL :
	new Qop_slice_Swap(Qubit_list(0, the_size/2),
			   Qubit_list(the_size - (the_size/2),
				      the_size/2).reverse())) { }
/* Constructor, through a preformed list (may throw exception) */
QSwap::QSwap(const Qubit_list &the_list)
  : Qop((the_list.size() < 2) ? NULL :
	new Qop_slice_Swap(the_list(0, the_list.size()/2),
			   the_list(the_list.size() -
				    the_list.size()/2, 0).reverse())) { }
/* Constructor, through two preformed lists (may throw exception) */
QSwap::QSwap(const Qubit_list &list_1,
 	     const Qubit_list &list_2)
  : Qop(new Qop_slice_Swap(list_1, list_2)) { }

/* *********************************************************************
   | Constructors for the QFourier transformation. The transform is    |
   | exploded using Hadamard transforms and conditional phase shifts.  |
   | The real construction is delegated to a specific routine (the     |
   | prepended ~ is indeed operator~ for QCnot_slices, which returns   |
   | the temporary by non const reference, so that it can be spliced   |
   | instead of copied by the Qop constructor).                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */

/* Default constructor. */
QFourier::QFourier(size_type the_size) 
  : Qop(~QFourier_slices(Qubit_list(0, the_size))) { }
/* Constructor through preformed list (may throw exception). */
QFourier::QFourier(const Qubit_list &the_list)
  : Qop(~QFourier_slices(the_list)) { }

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
