/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */

#include <cstdlib>                          // C standard library
#include <unistd.h>                         // POSIX symbolic constants
#include <fcntl.h>                          // File control operations
#include <string.h>                         // C handling of memory areas
#include <iostream>                         // C++ I/O handling
#include <qsimulator.h>                     // access to the simulator

/* *********************************************************************
   | For the time being, the simulator is a global object. It is in-   |
   | stantiated in this file and declared as an external object in the |
   | simulator header file, so that each file including the class de-  |
   | claration will have access to the actual simulator too.           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
Qsimulator simulator;

/* *********************************************************************
   | This private inline static method returns the squared norm of the |
   | complex argument (i.e. the square of the real part plus the square|
   | of the imaginary one). The C++ header <complex> (at the time of   |
   | writing) supplies a norm() function which, despite the name, cal- |
   | culates the same quantity; this implementation is however subopti-|
   | mal, since it involves the calculation of the norm first (which   |
   | needs a sqrt) so I preferred to write my own (squared) norm.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
inline Qsimulator::rnumber Qsimulator::norm2(const cnumber &z) {
  rnumber R = z.real();
  rnumber I = z.imag();
  return (R*R + I*I);
}

/* *********************************************************************
   | The following inline functions return a new random number from    |
   | the random stream. myrandom returns a value homogeneously distri- |
   | buted in [0, 1[ while myrandom2 casts it in [-0.5, 0.5[. Both     |
   | functions currently use the C library rand(), and they must be    |
   | the only way to request a random number in this simulator.        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
inline Qsimulator::rnumber myrandom(void)  { return (rand()/(RAND_MAX + 1.)); }
inline Qsimulator::rnumber myrandom2(void) { return (myrandom() - 0.5); }

/* *********************************************************************
   | Default constructor for the simulator object. Initialise to NULL  |
   | all the internal pointer and set all the internal fields to zero. |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2003         |
   ********************************************************************* */
Qsimulator::Qsimulator() : data(NULL),             // state vector area
			   zeeman_terms(NULL),     // Zeeman perturbation
			   coupling_terms(NULL) {  // coupling among qubits
  set_numqubits(0);                                // number of qubits
  set_noise(0.0, 0.0, 0.0);                        // noise levels
}

/* *********************************************************************
   | Destructor for the simulator object. Deallocate all the internal  |
   | data (arrays) if present (test the pointer for this).             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2003         |
   ********************************************************************* */
Qsimulator::~Qsimulator() {
  if (data)           delete [] data;              // state vector area
  if (zeeman_terms)   delete [] zeeman_terms;      // Zeeman perturbation
  if (coupling_terms) delete [] coupling_terms;    // Zeeman perturbation
}

/* *********************************************************************
   | This method initialises the random number generator for the       |
   | simulator; currently, random numbers are used for simulating the  |
   | results of measurements, initialisations and the effects of noise.|
   | There is only one generator though, so that the random stream is  |
   | shared. This method reads a new random seed from the /dev/urandom |
   | device; I don't know how much this is portable.                   |
   | ----------------------------------------------------------------- |
   | (09Jan03) S.Bettelli, add the possibility of setting the random   |
   |   seed to a given value. The chosen seed is also returned as the  |
   |   output of the function. If the input value is zero (which is    |
   |   the default) the behaviour is the same as before.               |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
unsigned long Qsimulator::init_random_seed(unsigned long seed) {
  /* open the urandom device and read the correct amount of bytes, which
     are then composed and interpreted as a seed. Then, close the device.
     Don't do this if "seed" is not zero! */
  if (seed == 0) {
    int fd = open("/dev/urandom", O_RDONLY);
    read(fd, &seed, sizeof(unsigned long));
    close(fd);
  }
  /* set the seed for a new sequence of pseudo-random integers to be
     returned by rand(), which is a global random number generator from
     the C library. We could think about making it a private object. */
  srand(seed);
  /* return the value of the chosen seed. */
  return seed;
}

/* *********************************************************************
   | This method sets the number of qubits managed by the simulator.   |
   | All the pre-existing data structures are cleared and new memory   |
   | is allocated. The managed classical memory is then initialised so |
   | that it corresponds to the quantum state |0> (the memory element  |
   | which corresponds to |0> is the first (since it is indexed by the |
   | bit string 000..0)                                                |
   | ----------------------------------------------------------------- |
   | (07/Dec/2002) S.Bettelli. Since the amount of allocated space can |
   | easily be enormous, it is mandatory to check wheather the alloca- |
   | tion was successful or not.                                       |
   | ----------------------------------------------------------------- |
   | (07/May/2003) S.Bettelli. If static errors are on, regenerate the |
   | random coefficients for the natural Hamiltonian.                  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      08 Dec 2002         |
   ********************************************************************* */
void Qsimulator::set_numqubits(size_type num_qubits) {
  /* return immediately if the number is already ok. */
  if (num_qubits == qubits()) return;
  /* release any previously allocated memory area */
  if (data) delete [] data; data = NULL;
  /* calculate the amount of memory we need (2^num_qubits). */
  memsize = (0x1 << num_qubits);
  /* allocate a new big array for storing all the 2^num_qubits amplitudes.
     Since this number can be enormous, be sure that the allocation was
     succesful, and print a meaningful error message if not. */
  try { data = new cnumber[memsize]; } catch(...) {
    std::cerr << "The allocation of " << (memsize*sizeof(cnumber))
	      << " memory elements was not succesful\nProbably, there"
	      << " is not enough available memory, aborting ...\n";
    abort(); }
  /* reset the memory of the simulator. */
  reset();
  /* regenerate the random coefficients for the natural Hamiltonian. */
  generate_zeeman(zeeman_noise);
  generate_couplings(coupling_noise);
}

/* *********************************************************************
   | This method resets the memory of the quantum simulator to |0>.    |
   | This is a perfect reset, it removes all the past memory.          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      06 Jun 2003         |
   ********************************************************************* */
void Qsimulator::reset(void) {
  /* reset all amplitudes to zero, exception made for the first, which
     is set to 1. This corresponds to the initialisation to |00..00> */
  for (cnumber *i = data; i != data + size(); ++i) *i = 0.0;
  *data = 1.0;
}

/* *********************************************************************
   | This method sets the noise levels for the simulation. The "gate"  |
   | argument determines the noise intensity for noisy gates. The      |
   | "zeeman" argument determines the maximum of the modulus of the    |
   | Zeeman coefficients for the natural Hamiltonian. See the comments |
   | in the class description for references to the error model.       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      08 May 2003         |
   ********************************************************************* */
void Qsimulator::set_noise(rnumber gates, rnumber zeeman, rnumber couplings) {
  /* change the noise intensity for noisy gates. */
  gate_noise = gates;
  /* change the max for Zeeman terms and regenerate the coefficients. */
  generate_zeeman(zeeman);
  /* change the max for coupling terms and regenerate the coefficients. */
  generate_couplings(couplings);
}

/* *********************************************************************
   | This method sets up the Zeeman terms for the static interactions. |
   | The Zeeman part for the static interactions Hamiltonian is:       |
   |                                                                   |
   |     H_Zeeman = sum_i delta_i sigma_z^i       i is a qubit index   |
   |                                                                   |
   | Therefore, we must generate qubits() independent delta values,    |
   | each with flat probability in the interval [-noise/2, noise/2].   |
   | ----------------------------------------------------------------- |
   | What is actually done here is saving in the "zeeman_terms" array  |
   | the complex values exp(+2i delta_j) for each qubit index j.       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2003         |
   ********************************************************************* */
void Qsimulator::generate_zeeman(rnumber noise) {
  /* delete any previously allocate array, if present. */
  if (zeeman_terms) delete [] zeeman_terms; zeeman_terms = NULL;
  /* save the max of the modulus for a Zeeman coefficient. */
  zeeman_noise = noise;
  /* if the noise level is zero, return immediately. We must take care
     to check for this condition during the evolution, because in this
     case the "zeeman_terms" array is not present. */
  if (zeeman_noise == 0.0) return;
  /* allocate a memory area for storing the Zeeman coefficients. */
  zeeman_terms = new cnumber[qubits()];
  /* for each qubit, choose an angle "delta" in [-noise/2, noise/2],
     then calculate exp(+2i delta) and save the result in the array. */
  for (size_type index = 0; index < qubits(); ++index) {
    rnumber delta = zeeman_noise * myrandom2();
    zeeman_terms[index] = cnumber(cos(2*delta), sin(2*delta));
  }
}

/* *********************************************************************
   | This method sets up the sigma-x-coupling terms for the static     |
   | interactions. The coupling part for the static interactions       |
   | Hamiltonian in our approximation is (i is the qubit index):       |
   |                                                                   |
   |        H_coupling = sum_i J_i sigma_x^(i) x sigma_x^(i+1)         |
   |                                                                   |
   | with qubits+1=0 (i.e. the quantum memory is treated like a ring). |
   | Therefore, we must generate qubits() independent coupling values  |
   | J, each with flat probability in the interval [-noise, noise].    |
   | ----------------------------------------------------------------- |
   | What is actually done here is saving in the "coupling_terms"      |
   | array the complex values exp(+2i J_j) for each qubit index j.     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      05 Jun 2003         |
   ********************************************************************* */
void Qsimulator::generate_couplings(rnumber noise) {
  /* delete any previously allocated array, if present. */
  if (coupling_terms) delete [] coupling_terms; coupling_terms = NULL;
  /* save the max of the modulus for a coupling coefficient. */
  coupling_noise = noise;
  /* if the noise level is zero, return immediately. We must take care
     to check for this condition during the evolution, because in this
     case the "coupling_terms" array is not present. */
  if (coupling_noise == 0.0) return;
  /* allocate a memory area for storing the coupling coefficients. */
  coupling_terms = new cnumber[qubits()];
  /* for each qubit, choose an angle "J" in [-noise, noise],
     then calculate exp(+2i J) and save the result in the array. */
  for (size_type index = 0; index < qubits(); ++index) {
    rnumber J = 2 * coupling_noise * myrandom2();
    coupling_terms[index] = cnumber(cos(2*J), sin(2*J));
  }
}

/* *********************************************************************
   | This method returns the number of qubits managed by the simulator.|
   | It is actually the base-2 logarithm of the number of coefficients |
   | stored in the classical memory. Since by construction this number |
   | is a power of two, we can get it with shifts and additions.       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
Qsimulator::size_type Qsimulator::qubits(void) const {
  size_type power = 0;
  size_type sizeval = size();
  while (sizeval /= 2) ++power;
  return power;
}

/* *********************************************************************
   | This method returns the probability that the target-th qubit is   |
   | set (i.e. it is in the state |1>). This probability is the sum    |
   | of the probabilities corresponding to those basis states where    |
   | the target-th bit is 1. It is of course assumed that the repre-   |
   | sentation of the quantum state is normalised.                     |
   | ----------------------------------------------------------------- |
   | The algorithm for finding the relevant elements in the classical  |
   | memory is a bit cryptic. It however uses only assignements, in-   |
   | crements and dereferentations, so it should be very fast.         |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
Qsimulator::rnumber Qsimulator::is_set(address target) const {
  /* the probability to be returned at the end. */
  rnumber prob_is_set = 0.0;
  /* "block" is the number of consecutive integers, starting from zero,
     which do not have the target-th bit set. We are interested in those
     memory elements indexed by integers falling in odd blocks. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (const cnumber *i = data; i != data + size(); i+= block)
    /* "j" loops over all the integers in a relevant block. */
    for (const cnumber *j = i += block; j != i + block; ++j)
      /* get the contribution from this memory element. */
      prob_is_set += norm2(*j);
  /* return the calculated probability of being set. */
  return prob_is_set;
}

/* *********************************************************************
   | This method returns the amplitude of the index-th basis element   |
   | in the quantum memory. Beware that the quantum memory does not    |
   | know about any register, so "index" is an absolute index into the |
   | quantum memory as a whole.                                        |
   | ----------------------------------------------------------------- |
   | This method is not trivial since the bit order is reversed in the |
   | simulator (i.e. the most significant bit is on the right). For    |
   | instance, index=1 corresponds to the (2^(num_qubits-1))-th memory |
   | element. Therefore, a reversal of the bit order is necessary.     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      09 Jan 2003         |
   ********************************************************************* */
Qsimulator::cnumber Qsimulator::amplitude(size_type index) const {
  /* "reversed_index" will be the obviuous thing. "mem_mask"
     will act as a counter which gets spilled bit by bit. */
  size_type reversed_index = 0, mem_mask = (memsize >> 0x1);
  /* continue till the counter is non-zero. */
  while (mem_mask) {
    /* capture the rightmost bit of "index" */
    size_type test_last = index & 0x1;
    /* shift "index" right, thus loosing the rightmost bit. */
    index >>= 0x1;
    /* push the saved bit into "reversed_index", from the right. */
    (reversed_index <<= 0x1) ^= test_last;
    /* decrease the counter by shifting it one bit on the right. */
    mem_mask >>= 0x1; }
  /* return the memory element corresponding to "reversed_index". */
  return (*this)[reversed_index];
}

/* *********************************************************************
   | This method returns the squared amplitude of the index-th basis   |
   | element in the quantum memory (i.e. the probability to be measu-  |
   | red during a computational basis measurement). Have a look at the |
   | "amplitude" method for other details.                             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 Jan 2003         |
   ********************************************************************* */
Qsimulator::rnumber Qsimulator::probability(size_type index) const {
  return norm2(amplitude(index)); // should it be inline?
}

/* *********************************************************************
   | This method implements the collapse of the quantum state of the   |
   | simulated system to the state |value> for the target-th qubit.    |
   | The action of the projector |value><value| for the target-th      |
   | qubit tensor the identity for the rest of the system consists in  |
   | setting to zero all the memory elements which correspond to the   |
   | "wrong" projection and in renormalising the others. The last ar-  |
   | gument is the "probability" that the collapse takes place, which  |
   | must be known in advance.                                         |
   | ----------------------------------------------------------------- |
   | Be aware that if the current quantum state lives in the zero      |
   | eigenvalue eigenspace of the projector, the state vector after    |
   | the projection is the null vector, and the renormalisation is     |
   | not possible! This method should therefore remain private and be  |
   | called only by carefully crafted routines. The renormalisation    |
   | coefficient is calculated from "probability".                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Nov 2002         |
   ********************************************************************* */
void Qsimulator::do_collapse(address target, bool value, rnumber probability) {
  /* the unnormalised state after the collapse has a norm equal to
     the square root of the probability of the given outcome. Prepare
     a renormalisation coefficient. */
  rnumber renormalisation = 1.0 / sqrt(probability);
  /* prepare two factors for later use. renorm_zero is the appropriate
     factor for those memory indexes with the target-th bit set to zero:
     the corresponding elements must be zeroed or renormalised depending
     on "value" being true or false. renorm_one is the "reverse" ... */
  rnumber renorm_zero = (value ? 0.0 : renormalisation);
  rnumber renorm_one  = (value ? renormalisation : 0.0);
  /* "block" is the size of a block of consecutive integers
     which have the same value for the target-th bit. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (cnumber *i = data; i != data + size(); i += block) {
    /* loop over all indexes with the target-th bit set to zero. */
    for (cnumber *j = i         ; j != i + block; ++j) *j *= renorm_zero;
    /* loop over all indexes with the target-th bit set to one. */
    for (cnumber *j = i += block; j != i + block; ++j) *j *= renorm_one;
  }
}

/* *********************************************************************
   | This method implements the measurement of a qubit. In practice,   |
   | it calculates the probability that the target-th qubit is set or  |
   | unset, and then generates a random result according to this       |
   | probability. If "collapse" is set, the memory elements are then   |
   | modified and renormalised just as if a collapse of the state of   |
   | the quantum system took place. The return value is the outcome    |
   | of the fake measurement.                                          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
bool Qsimulator::do_measurement(address target, bool collapse) {
  /* the result to be returned at the end. */
  bool result;
  /* calculate the probability that the target-th qubit is set. */
  rnumber prob_is_set = is_set(target);
  /* get a random number in [0,1[. If this number is smaller than the 
     probability, assign "true" to the result, otherwise assign "false". */
  if (myrandom() < prob_is_set) result = true; else result = false;
  /* if "collapse" is true, project according to the measurement result.
     The probability for the projection is the probability that the qubit
     is set or unset depending on the result. */
  if (collapse)
    do_collapse(target, result, (result ? prob_is_set : (1.0 - prob_is_set)));
  /* return the measured value */
  return result;
}

/* *********************************************************************
   | This method evolves the system with a natural Hamiltonian given   |
   | by Zeeman terms for a unit of time. This is equivalent to say     |
   | that the system is evolved according to the following unitary op: |
   |                                                                   |
   |       U = exp(-i sum_k delta_k sigma_z^k)      k = qubit index    |
   |                                                                   |
   | where the deltas are predetermined (see generate_zeeman()).       |
   | ----------------------------------------------------------------- |
   | This is equivalent to multiplying each basis element |x> times a  |
   | phase exp(2i sum_k delta_k x_k); in fact, the only difference is  |
   | a meaningless global phase factor. The implementation, of course, |
   | is faster, since we must only multiply times some precalculated   |
   | factors for each bit which is up.                                 |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2003         |
   ********************************************************************* */
void Qsimulator::do_zeeman_terms(void) {
  /* return immediately if the error intensity for Zeeman terms is zero. */
  if (zeeman_noise == 0.0) return;
  /* loop over all the qubits in the quantum computer memory. */
  for (address target = 0; target < qubits(); ++target) {
    /* "block" is the number of consecutive integers, starting from zero,
       which do not have the target-th bit set. We are interested in those
       memory elements indexed by integers whose target-th bit is up. */
    const size_type block = 0x1 << target;
    /* "i" loops over all beginnings of double blocks. */
    for (cnumber *i = data; i != data + size(); i+= block)
      /* "j" loops over all the integers in a relevant block. */
      for (cnumber *j = i += block; j != i + block; ++j)
	/* transfer the phase given by delta_target. */
  	*j *= zeeman_terms[target];
  }
}

/* *********************************************************************
   | This method evolves the system with a natural Hamiltonian given   |
   | by the coupling terms for a unit of time. This is equivalent to   |
   | say that the system is evolved according to the following operat: |
   |                                                                   |
   | U = exp(-i sum_k J_k sigma_x^(k) x sigma_x^(k+1))   k=qubit index |
   |                                                                   |
   | where the Js are predetermined (see generate_couplings()) and     |
   | qubits+1=0 (i.e. the quantum memory is treated like a ring).      |
   | ----------------------------------------------------------------- |
   | Since the sigma_x operators are not diagonal, it is convenient to |
   | replace the unitary evolution U with a product F U' F, with F a   |
   | tensor product of Hadamard matrices on each qubit and U' a modi-  |
   | fied U where each sigma_x is replaced by a sigma_z. Infact, the   |
   | multiplication rule H sigma_z H = sigma_x is true. The effect of  |
   | U' is then equivalent to multiplying each basis element |x> times |
   | a phase exp(2i sum_k J_k x_k XOR x_(k+1)); in fact, the only dif- |
   | ference is a meaningless global phase factor. The implementation, |
   | of course, is faster, since we must only multiply times some pre- |
   | calculated factors for each consecutive pair of bits which differ.|
   | ----------------------------------------------------------------- |
   | (Special cases). If there is only 1 qubit, the are no couplings.  |
   | If there are only 2 qubits,don't count the single coupling twice! |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      05 Jun 2003         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      10 Jun 2003         |
   ********************************************************************* */
void Qsimulator::do_coupling_terms(void) {
  /* return immediately if the error intensity for coupling terms is zero. */
  if (coupling_noise == 0.0) return;
  /* return immediately if there is only one qubit (no couplings!). */
  if (qubits() == 1) return;
  /* implement the transformation F, one Hadamard matrix on each qubit. */
  do_multiHadamard();
  /* loop over all the qubits in the quantum computer memory. At each
     cycle we are interested in the pair (target, target+1). The index
     corresponding to the last qubit (i.e. qubits()-1) is not dealt with
     in this loop, since it is a bit peculiar (see later). */
  for (address target = 0; target < qubits()-1; ++target) {
    /* "block" is the number of consecutive integers, starting from zero,
       which do not have the target-th bit set. In this method, "block"
       will be used in a non-standard way, since we are interested in 
       pairs of bits, and not in single bits. */
    const size_type block = 0x1 << target;
    /* "i" loops over all beginnings of quadruple blocks. */
    for (cnumber *i = data; i != data + size(); i+= 3*block)
      /* "j" loops over all the integers in the "01" and "10" blocks. */
      for (cnumber *j = i += block; j != i + 2*block; ++j)
	/* transfer the phase given by J_target. */
  	*j *= coupling_terms[target];
  }
  /* This last section deals with the pair (qubits()-1,qubits()), which
     really means (qubits()-1,0). In this case the previous loop would
     not work without major modifications. This section is also the most
     time consuming, in my opinion, so a rough implementation could skip
     it (for a large number of qubits it shouldn't make any difference
     in the physics of the evolution. In practice, we divide the whole
     memory in two parts: in the first one (that with the (qubits()-1)-th
     bit unset) we deal with integers with the first bit set; in the 
     second part the algorithm is the opposite.
     (DANGER) Skip this section if there are only two qubits, otherwise
     we would count the unique existing coupling twice! */
  if (qubits() != 2) {
    cnumber *j = (data + 1), *halflast = (data + size()/2 - 1);
    cnumber phase = coupling_terms[0];
    for (   ; j != halflast     ; j+=2) *j *= phase;
    for (++j; j != data + size(); j+=2) *j *= phase;
  }
  /* undo the transformation F; since H is self-adjoint this is F itself. */
  do_multiHadamard();
}

/* *********************************************************************
   | This method applies one perfect Hadamard transformation on each   |
   | qubit in the simulator memory. It is used to generate couplings   |
   | in conjunction with a diagonal transformation. The implementation |
   | strategy is not terribly smart, we simply repeat the routine for  |
   | Hadamard qubits() times (is there anything faster?)               |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      23 May 2003         |
   ********************************************************************* */
void Qsimulator::do_multiHadamard(void) {
  /* loop over all the qubits in the quantum computer memory. "block" is
     the size of a block of consecutive integers which have the same value
     for the target-th bit. It is updated at every cycle; therefore we
     don't need to keep a variable "target" = log2(block), a comparison
     with the memory size will do the job for the end loop condition. */
  for (size_type block = 0x1; block < size(); block<<=1)
    /* "i" loops over all beginnings of double blocks. */
    for (cnumber *i = data; i != data + size(); i += 2*block)
      /* loop over all indexes with the target-th bit set to zero. */
      for (cnumber *j = i; j != i + block; ++j) { 
	/* The internals of this inner loop are identical to those in
	   the "hadamard" method. The comments will not be repeated. */
	cnumber *k = j;
	k  += block;
	*j += *k;
	*j *= + M_SQRT1_2;
	*k *= - M_SQRT2;
	*k += *j;
      }
}

/* *********************************************************************
   | This method implements the initialisation of the target-th qubit  |
   | to "value" (which can be |0> or |1>). This must correspond to the |
   | partial measurement (target-th qubit) of the whole quantum system |
   | possibly followed by a correction (like a "not" gate) of that     |
   | qubit if the outcome is "wrong". The state remains normalised.    |
   | ----------------------------------------------------------------- |
   | (08/Dec/2002) S.Bettelli. Don't allocate the temporary buffer on  |
   | the stack (this may cause segmentation fault errors difficult to  |
   | track down). Use dynamical allocation instead, and check explici- |
   | tely that it was succesful.                                       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      08 Dec 2002         |
   ********************************************************************* */
void Qsimulator::init(address target, bool value) {
  /* measure the target-th qubit and save the result */
  bool result = do_measurement(target, true);
  /* if the result is not what we want, we must add a correction. This
     consists in interchanging the coefficient of |x0y> and |x1y>, where
     the 0 and the 1 are in the target-th position. */
  if (result != value) {
    /* "block" is the size of a block of consecutive integers
       which have the same value for the target-th bit. */
    const size_type block = 0x1 << target;
    /* calculate the block size (in bytes) and allocate a temporary
       work space able to contain this amount of memory. Since this
       number can be enormous, be sure that the allocation was
       succesful, and print a meaningful error message if not. */
    size_type blocksize = sizeof(cnumber)*block;
    cnumber *temp = NULL;
    try { temp = new cnumber[blocksize]; } catch(...) {
      std::cerr << "The allocation of " << (blocksize*sizeof(cnumber))
		<< " memory elements was not succesful\nProbably, there"
		<< " is not enough available memory, aborting ...\n";
      abort(); }
    /* "i" loops over all beginnings of double blocks. */
    for (cnumber *i = data; i != data + size(); i += 2*block) {
      /* swap [i,i+block[ and [i+block,i+2*block[. This corresponds to
	 interchanging the coefficients of |x0y> and |x1y> with fixed x.
	 Use the C library memcpy which should be the fastest available. */
      memcpy(temp      , i         , blocksize);
      memcpy(i         , i + block , blocksize);
      memcpy(i + block , temp      , blocksize);
    }
    /* release the temporary workspace. */
    delete [] temp;
  }
}

/* *********************************************************************
   | This routine prepares the single-qubit matrix for implementing a  |
   | Hadamard rotation. The perfect Hadamard matrix is {{1,1},{1,-1}}  |
   | times 1/sqrt(2), that is a rotation of an angle PI around the     |
   | axis {1,0,1} times 1/sqrt(2) on the Bloch sphere. If the noise    |
   | level is not zero, the axis is perturbed by a kick uniformly      |
   | distributed in a solid angle with theta = [-noise/4, noise/4[.    |
   | ----------------------------------------------------------------- |
   | Note that I have changed the range for angle for the tilted axis  |
   | from [-noise/2, noise/2[ to [-noise/4, noise/4[. In this way the  |
   | perturbed Hadamard gate is a perfect Hadamard gate followed by a  |
   | rotation of an angle with flat distribution in [-noise/2, noise/2[|
   | around a random axis orthogonal to 1/sqrt(2)(1,0,1). This change  |
   | makes the errors on phase gates and Hadamard gates "the same size"|
   | ----------------------------------------------------------------- |
   | It turns out that only two matrix elements are independent in the |
   | perturbed matrix {{m00,m01},{m10,m11}}, that is m00 and m01,      |
   | since m11 = - m00 and m10 = conj(m01). For sake of efficiency,    |
   | this routine only calculates and returns m00 and m01.             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Nov 2002         |
   ********************************************************************* */
inline void Qsimulator::prepare_hadamard(cnumber &m00, cnumber &m01) {
  /* in presence of noise, we adopt an error model which perturbs the
     rotation axis, not the rotation angle. First, the axis is rotated
     in the X-Z plane of a random angle A in [-noise/4, noise/4[; then,
     the resulting axis is rotated around the original axis of a random
     angle B in [0,2*PI[. The following formulas corresponds to the 
     overall transformation, although they are a bit cryptic in order
     to gain efficiency. */
  rnumber A = 0.5 * gate_noise * myrandom2(); // random in [-noise/4, noise/4[
  rnumber B = 2.0 * M_PI * myrandom();        // random in [0, 2*PI[
  rnumber sina = sin(A), cosa = cos(A);
  rnumber sinacosb = sina * cos(B);
  m00 = cnumber(M_SQRT1_2 * ( cosa - sinacosb ), 0.0        );
  m01 = cnumber(M_SQRT1_2 * ( cosa + sinacosb ), sina * sin(B));
}

/* *********************************************************************
   | This method modifies the internal data structures according to    |
   | a Hadamard gate executed on the target-th qubit. If the noise     |
   | level for gates is non-zero, the gate is perturbed.               |
   | ----------------------------------------------------------------- |
   | The details of the gate perturbation are managed by the method    |
   | "prepare_hadamard". The iteration through the memory elements is  |
   | similar to that for is_set().                                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
void Qsimulator::hadamard(address target) {
  /* some variables for later use. m00 and m01 are the independent
     matrix elements of the perturbed Hadamard. */
  cnumber m00, m01, mim00, f, fc;
  /* this boolean variable is true if the gate is to be perturbed. */
  const bool is_noisy = (gate_noise != 0.0);
  /* If the gate noise level is not zero, we must calculate some
     quantities before entering the main loop. */
  if (is_noisy) {
    prepare_hadamard(m00, m01);  // get the independent matrix elements
    f = m01 / m00;               // f is the ratio of these two elements
    fc = conj(f);                // fc is the conjugate of f
    mim00 = - 1. / m00; }        // m(inus) (the) i(nverse) (of) m00
  /* "block" is the size of a block of consecutive integers
     which have the same value for the target-th bit. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (cnumber *i = data; i != data + size(); i += 2*block)
    /* loop over all indexes with the target-th bit set to zero. */
    for (cnumber *j = i; j != i + block; ++j) { 
      /* the Hadamard matrix is going to mix the coefficients of |x0y>
	 and |x1y>, where the 0 and the 1 are in the target-th position.
	 In the following, the coefficient "A" of |x0y> is *j and the
	 coefficient "B" of |x1y> is *k. */
      cnumber *k = j;
      k += block;
      /* General (noisy) case: mixing of "A" and "B" using the matrix
	 {{m00,m01},{m01^*,-m00}}. The following procedure is the fastest
	 I was able to devise (it uses the minimum number of temporaries,
	 and it does not involve subractions or division). */
      if (is_noisy) {
	*j += *k * f;      // *j == A + B*m01/m00     *k == B
	*j *= m00;         // *j == A*m00 + B*m01     *k == B
	*k *= mim00;       // *j == A*m00 + B*m01     *k == - B/m00
	*k += *j * fc;     // *j == A*m00 + B*m01     *k == A*m01^* - B*m00
      } 
      /* In the particular case of a perfect Hadamard matrix the procedure
	 is even simpler, since all the multiplicative coefficients are
	 constants (and sometimes they are zero). We can use the constant
	 M_SQRT1_2, defined in the C math library, which is 1/sqrt(2), as
	 well as M_SQRT2 which is sqrt(2). */
      else {
	*j += *k;          // *j == A + B             *k == B
	*j *= + M_SQRT1_2; // *j == (A + B)/sqrt(2)   *k == B
	*k *= - M_SQRT2;   // *j == (A + B)/sqrt(2)   *k == - B*sqrt(2)
	*k += *j;          // *j == (A + B)/sqrt(2)   *k == (A - B)/sqrt(2)
      }
    }
}

/* *********************************************************************
   | This routine accepts a real number "fraction" specifying a frac-  |
   | tion of an angle in [0, 2*PI], and prepares the phase for a phase |
   | (or conditional phase) rotation. This phase is set to the value   |
   | of exp(2*PI*i*fraction). If the noise level isn't zero, the angle |
   | is perturbed by a kick uniformly distrib. in [-noise/2, noise/2[. |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
inline Qsimulator::cnumber Qsimulator::prepare_phase(rnumber fraction) {
  /* convert the "fraction" into an angle in [0,2PI]. M_PI is
     the greek pi constant defined in the C math library. */
  rnumber angle = 2.0 * M_PI * fraction;
  /* if the gate noise level is not zero, add a perturbation to the
     angle, homogeneously distributed in [-noise/2, noise/2[. */
  if (gate_noise != 0.0) angle += gate_noise * myrandom2();
  /* the phase to return is exp(imaginary unit * angle). */
  return cnumber(cos(angle), sin(angle));
}

/* *********************************************************************
   | This method modifies the internal data structures according to a  |
   | phase gate executed on the target-th qubit with param. "fraction".|
   | If the gate noise level is non-zero, the gate is perturbed.       |
   | ----------------------------------------------------------------- |
   | A perturbed phase gate is a phase gate with a perturbed phase     |
   | angle. The details of this perturbation are managed by the method |
   | "prepare_phase". The iteration through the memory elements is the |
   | same as for is_set(), since the concerned elements are the same.  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
void Qsimulator::phase(address target, rnumber fraction) {
  /* get the phase parameter + noise */
  cnumber phase = prepare_phase(fraction);
  /* "block" is the size of a block of consecutive integers
     which have the same value for the target-th bit. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (cnumber *i = data; i != data + size(); i += block)
    /* "j" loops over all the integers with the target-th bit set. */
    for (cnumber *j = i += block; j != i + block; ++j)
      /* apply the phase shift. */
      *j *= phase;
}

/* *********************************************************************
   | This method modifies the internal data structures according to a  |
   | phase gate executed on the target-th qubit conditioned by the     |
   | state of the control-th qubit, with parameter "fraction".         |
   | If the gate noise level is non-zero, the gate is perturbed.       |
   | ----------------------------------------------------------------- |
   | A perturbed conditional phase gate is a conditional phase gate    |
   | with a perturbed phase angle. The details of this perturbation    |
   | are managed by the method "prepare_phase". The iteration through  |
   | the memory elements is a variation of the algorithm used for the  |
   | phase() method (indeed, a double loop).                           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
void Qsimulator::condphase(address control, address target, rnumber fraction) {
  /* get the phase parameter + noise */
  cnumber phase = prepare_phase(fraction);
  /* "block_t" and "block_c" are the sizes of blocks of consecutive
     integers which have the same value for the target-th or the 
     control-th bit. We reason as if control was less than target
     (and take the appropriate countermeasure if this is not true). */
  bool ok = control < target;
  const size_type block_t = (ok ? (0x1 << target ) : (0x1 << control));
  const size_type block_c = (ok ? (0x1 << control) : (0x1 << target ));
  /* "i" loops over all beginnings of double block_t blocks. */
  for (cnumber *i = data; i != data + size(); i += block_t)
    /* "j" loops over all beginnings of double block_c blocks. */
    for (cnumber *j = i += block_t; j != i + block_t; j += block_c)
      /* "k" loops over all the integers with the target-th and
	 the control-th bit set. */
      for (cnumber *k = j += block_c; k != j + block_c; ++k)
	/* apply the phase shift. */
	*k *= phase;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
