/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iomanip>                      // C++ I/O manipulators
#include <qsummary.h>                   // base class declaration

/* *********************************************************************
   | The constructor does nothing more than the base class constructor |
   | exception made for setting all the usage counts and the cookie    |
   | pointer to default values. The default constructor uses a single  |
   | sized segment with zero usage count, but this is mainly to fool   |
   | the compiler. The default cookie pointer is a NULL pointer.       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 20 Jan 2001         |
   ********************************************************************* */
Qubit_summary::Qubit_summary(address the_first, size_type the_size) :
  Qubit_segment(the_first, the_size),
  count(0), left_count(0), right_count(0),
  cookie(NULL)
{}

/* *********************************************************************
   | This method sets all the usage counts for this segment summary.   |
   | The three arguments are copied respectively in the segment usage  |
   | count, in the minimum address border usage count and in the       |
   | maximum address border usage count.                               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 23 Apr 2001         |
   ********************************************************************* */
void Qubit_summary::set(counter_type bulk_count,
			counter_type left_count,
			counter_type right_count) {
  set_bulk(bulk_count);
  set_left_border(left_count);
  set_right_border(right_count);
}

/* *********************************************************************
   | This function shows the content of Qubit_cookie. Indeed it does   |
   | nothing because the Qubit_cookie class is empty. It is here only  |
   | as a placeholder, replaceable by more specific output functions.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jan 2001         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qubit_cookie &a_cookie) {
  os << "(dummy)";                                // dummy string
  return os;                                      // return the stream
}

/* *********************************************************************
   | This function shows the content of a segment with usage statistic |
   | (for debugging). For each segment it prints the standard segment  |
   | output plus the usage counts. It also calls the output function   |
   | for the (most derived) Qubit_cookie (if present).                 |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 22 Jan 2001         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qubit_summary &a_summary) {
  os << (Qubit_segment)a_summary                  // show segment output
     << " usage " << std::setw(2)                 // 2 digits for the ...
     << a_summary.get_bulk()                      // segment usage count
     << " (" << std::setw(2)                      // enclose with brackets
     << a_summary.get_left_border()               // left border count
     << ',' << std::setw(2)                       // comma and space
     << a_summary.get_right_border() << ')';      // right border count
  if (a_summary.get_cookie())                     // be sure it is set
    os << " cookie " << *a_summary.get_cookie();  // cookie info
  else os << " no-cookie";                        // no cookie info
  return os;                                      // return the stream
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
