/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */

#ifndef __QLIB_MULTIPRECISION_INTEGERS
#define __QLIB_MULTIPRECISION_INTEGERS

#include <iostream>                              // for C++ I/O streams
#include <vector>                                // for STL vectors
#include <climits>                               // for CHAR_BIT
#include <cmath>                                 // mathematical functions
#include <qexception.h>                          // for "quantum" exceptions
#define BASIC_INT__TYPE unsigned long
#define BIT_CONT___TYPE std::vector<BASIC_INT__TYPE>

/* *********************************************************************
   | This is an auxiliary class for integers with an arbitrary number  |
   | of digits. The basic elementary operations are implemented, as    |
   | well as the comparison operators. The class is not optimised for  |
   | speed, but it should be anyway sufficiently fast for integers up  |
   | to some hundreds of digits.                                       |
   | ----------------------------------------------------------------- |
   | The integers are internally stored as sequences of "basic_int"    |
   | objects, and are to be interpreted as unlimited binary sequences. |
   | If n is the average lenght of the operands, then addition and     |
   | subtraction scale as n, multiplication as n^2 and division as n^3.|
   | I don't plan to develop this class further unless new needs arise |
   | for the quantum library.                                          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Mar 2003         |
   ********************************************************************* */
class big_int : private BIT_CONT___TYPE {
  // a typedef for the atomic integer type used in this class
  typedef BASIC_INT__TYPE basic_int;
  // the container type on which the class is based
  typedef BIT_CONT___TYPE bit_container;
public:
  // an exception for an invalid subtraction (a-b, with b>a)
  _Qexception(invalid_bigint_subtraction);
  // an exception for an invalid division (a/b, with b==0)
  _Qexception(invalid_bigint_division);
public:
  // this is the copy constructor for big integers
  big_int(const big_int &i) : bit_container(i) { }
  // standard constructor: builds a big integer from a basic integer
  big_int(basic_int i = 0) : bit_container(1, i) { }
public:
  // this converts from a floating point (discarding fractional part)
  static big_int from_double(double x);
  // this converts to a floating point (may lose the least significant digits)
  static double to_double(const big_int &i);
public:
  // comparison operators between two big integers
  bool     operator==(const big_int &i) const;
  bool     operator< (const big_int &i) const;
  bool     operator<=(const big_int &i) const { return (*this<i || *this==i); }
  bool     operator> (const big_int &i) const { return !(*this <= i); }
  bool     operator>=(const big_int &i) const { return !(*this < i); }
  // elementary arithmetic operations between two big integers
  big_int  operator+ (const big_int &i) const { return (big_int(*this) += i); }
  big_int  operator- (const big_int &i) const { return (big_int(*this) -= i); }
  big_int  operator* (const big_int &i) const;
  big_int  operator/ (const big_int &i) const;
  big_int  operator% (const big_int &i) const;
  // assignement operators (copies the private container)
  big_int &operator= (const big_int &i) { return copy(i);}
  // calculate-and-assign versions of the elementary operations
  big_int &operator+=(const big_int &i);
  big_int &operator-=(const big_int &i);
  big_int &operator*=(const big_int &i) { return *this = *this * i; }
  big_int &operator/=(const big_int &i) { return *this = *this / i; }
  big_int &operator%=(const big_int &i) { return *this = *this % i; }
  // shift operations on big integers
  big_int  operator<< (unsigned long i) const { return (big_int(*this)<<= i); }
  big_int  operator>> (unsigned long i) const { return (big_int(*this)>>= i); }
  // calculate-and-assign versions of the shift operations
  big_int &operator<<=(unsigned long i);
  big_int &operator>>=(unsigned long i);
public:
  // a container type for returning the digits in radix-10
  typedef std::vector<unsigned char> digit_container;
  // this converts the binary representation in radix-10 representation
  digit_container get_digits(void) const;
  // this method calculates both the integer division and the reminder
  void division(const big_int &i, big_int &result, big_int &reminder) const;
  // this returns the total number of bits in the private container
  unsigned long bits(void) const { return size()*bits_chunk; }
  // this returns the mantissa of ((*this) * fpoint)
  double big_int::mantissa_of_multiplication_by(double fpoint) const;
private:
  // this drops the "cnt" least significant basic_ints (like ">> cnt")
  big_int &drop(unsigned long cnt) { erase(begin(),begin()+cnt); return *this;}
  // this inserts "cnt" basic_ints in the least sign. positions (like "<< cnt")
  big_int &shift(unsigned long cnt) { insert(begin(), cnt, 0); return *this; }
  // this inserts "cnt" basic_ints in the most sign. positions
  big_int &extend(unsigned long cnt) { insert(end(), cnt, 0); return *this; }
  // this pushes a carry in the big_int in the most significant position
  big_int &addcarry(const basic_int i) { push_back(i); return *this; }
  // this copies the internal data structures
  big_int &copy(const big_int &i) { bit_container::operator=(i); return *this;}
  // this removes all the most significant digits which are zero
  big_int &trim(void) {while (size()>1 && back()==0) pop_back(); return *this;}
private:
  // this performs a+=b and returns the carry
  static basic_int sum_and_carry(basic_int &a, const basic_int b);
  // this performs a-=b and returns the borrow
  static basic_int subtract_and_borrow(basic_int &a, const basic_int b);
  // this performs a*=b and returns the carry
  static basic_int multiply_and_carry(basic_int &a, const basic_int b);
  // like multiply_and_carry, but the first operand is a big_int
  static basic_int multiply_row_and_carry(big_int &result, const basic_int i);
  // this extracts the least significant half of a basic_int
  static basic_int low_part(const basic_int i) { return (i & low_mask); }
  // this extracts the most significant half of a basic_int
  static basic_int high_part(const basic_int i) {return (i>>half_bits_chunk);}
  // this estimates the integer ratio of two big integers
  static basic_int ratio_estimate(const big_int &a, const big_int &b);
private:
  // this is the number of bits in a basic integer
  static const unsigned long bits_chunk = (sizeof(basic_int)*CHAR_BIT);
  // this is half the number of bits in a basic integer
  static const unsigned long half_bits_chunk = (bits_chunk / 2);
  // this is a binary mask which selects the least sign. part of a basic_int
  static const basic_int low_mask = (basic_int(-1) >> half_bits_chunk);
};

// this method prints the big integer digits in radix-10 representation
std::ostream &operator<<(std::ostream &os, const big_int &i);

#endif  // __QLIB_MULTIPRECISION_INTEGERS
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
