/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 Stefano Bettelli                     |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <qsimulator.h>               // access to the quantum simulator

typedef Qsimulator::cnumber cnumber;  // a typedef for a complex number
typedef Qsimulator::rnumber rnumber;  // a typedef for a real number

/* *********************************************************************
   | This routine finds the roots of a fourth degree polynomial (which |
   | can be done exactly, i.e. there exists a solving formula, we do   |
   | not need approximated methods). The (complex) coefficients of the |
   | polynomial are stored in the array "coefficients[4]" = (a,b,c,d), |
   | and the polynomial is def as p(x) = x^4 + a*x^3 + b*x^2 + c*x + d |
   | (the coefficient of the fourth degree term is assumed to be 1).   |
   | The roots are returned in the array passed as last argument.      |
   | ----------------------------------------------------------------- |
   | Reference: Gerolamo Cardano, "Ars Magna", 1545                    |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
void solution_4th_degree(const cnumber coefficients[4], cnumber solutions[4]) {
  /* Rename the coefficients with more standard and simple names. The
     fourth degree polynomial is p(x) = x^4 + a*x^3 + b*x^2 + c*x + d*/
  const cnumber &a = coefficients[0];
  const cnumber &b = coefficients[1];
  const cnumber &c = coefficients[2];
  const cnumber &d = coefficients[3];
  /* The first step in the solving procedure consists in translating the
     variable, i.e. setting x = y - a/4, so that the third degree term
     becomes zero. The new polynomial, where the independent variable is
     y, can be written as y^4 + A*y^2 + B*y + C. The coefficients A, B
     and C are determined by substitution: */
  cnumber A = (8.*b - 3.*a*a) / 16.0;
  cnumber B = (4.*a*b - 8.*c - a*a*a) / 8.0;
  cnumber C = (3.*a*a*a*a + 64.*a*c - 16.*a*a*b - 256.*d) / 256.0;
  /* The new polynomial can be reorganised as (y^2 + A)^2 = A^2 + B*y + C.
     Introduce now a new variable term, z, in the parenthesis on the left.
     The equality becomes (y^2+A+z)^2 = 2*z*y^2 + B*y + (A^2+C+z^2+2*z*A).
     Since the left part of this equality is a perfect square, the discri-
     minant of the right part, as a polynomial in y, must be zero, that
     is Delta = B^2 - 8*z*(A^2+C+2*z*A+z^2) = 0. We are therefore led to
     the problem of finding the roots of a third degree equation in z:
     z^3 + 2*A*z^2 + z*(A^2+C) - B^2/8 = 0. Just like before, we translate
     z = w - 2*A/3 to get rid of the second degree term; the equation is
     then reduced to w^3 + p*w + q = 0, where the numbers p and q are
     again determined by substitution: */
  cnumber p = (3.*C - A*A) / 3.0;
  cnumber q = - ((2.*A*A*A + 18.*A*C) / 27.0) - ((B*B) / 8.0);
  /* Now apply the standard procedure for solving this particular third
     degree equation. Set w = u + v. The previous equation in w becomes
     then (u^3+v^3+q) + (u+v)*(p+3*u*v) = 0, which can be satisfied
     obviously if u^3+v^3 = -q and 3*u*v = -p. All the solutions of the
     second equation satisfy also 27*u^3*v^3 = -p^3. Hence we only have
     to find u^3 and v^3 such that their sum is -q and their product is
     -p^3/27, i.e. the solutions of m^2 + q*m - p^3/27 = 0, which are
     easily found to be -q/2 +/- r (r is shown in the next line): */
  cnumber r = sqrt(((p*p*p) / 27.0) + ((q*q) / 4.0));
  /* So, let us pick up one solution, say m = r - q/2, and set for 
     instance u to one of its complex cubic root (since u^3 = m is a
     solution). Assume that 0^(1/3) is zero, don't ask the math library. */
  cnumber m = (r - (q/2.0));
  cnumber u = ((m == 0.0) ? 0.0 : pow(m, 1/3.0));  // 0^(1/3) == nan
  /* Now calculate the corresponding v from u*v = -p/3, that is set v equal
     to -p/3*u. Then calculate w from w = u + v. If u is zero (which is pos-
     sible only if p=0) then the previous formula cannot be applied; in this
     case though v^3 = -q implies w = - q^(1/3) (test q for being non-zero,
     and if it is set q^(1/3)=0.0 without asking the math library to extract
     the cubic root, since it would fail with nan). After this, it is easy
     to find a solution for Delta = 0 by setting z = w - 2*A/3. */
  cnumber w = ((u==0.) ? ((q==0.) ? 0.0 : - pow(q, 1/3.0)) : (u - p/(3.*u)));
  cnumber z = w - ((2.*A) / 3.0);
  /* When z is set to the previous value and the discriminant is (hence)
     zero, the equation in y becomes (y^2+A+z)^2 = 2*z*(y + B/4*z)^2.
     Taking the square roots gives us a second degree equation in y:
     y^2 + A + z = s1*sqrt(2*z)*(y+B/4*z), where s1=+/-1 is a sign choice.
     By setting H = sqrt(z/2) we have y^2 - s1*2*H*y + A + z - s1*B/4*H = 0. */
  cnumber H = sqrt(z/2.0);
  /* If H and z are different from zero, the previous equation is satisfied
     by setting y = s1*H + s2*sqrt(- A - z/2 + s1*B/4*H)). If H=0 the term
     with B/H is undefined; H=0 however means that z=0 hence the equation
     for the discriminant reduces to Delta = B^2 = 0, therefore B=0 and 
     the equality (y^2 + A)^2 = A^2 + B*y + C becomes (y^2 + A)^2 = A^2 + C,
     which is satisfied by y = s2*sqrt(- A + s1*sqrt(A^2 + C)). By adding
     null terms involving H and z, this last solution can be written as
     y = s1*H + s2*sqrt(- A - z/2 + s1*sqrt(A^2 + C)). We can see then
     that both types of solutions (for z != 0 and z = 0) can be expressed
     in the same form, y = s1*H + s2*sqrt(f1 + s1*f2) if we define: */
  cnumber f1 = (- A - (z/2.0));
  cnumber f2 = (H != 0.0) ? (B/(4.*H)) : sqrt(A*A + C); // degeneration
  /* Now, the solving formula y = s1*H + s2*sqrt(f1 + s1*f2) gives us four
     distinct solutions for y (by choosing s1 = 1 or -1 and s2 = 1 or -1).
     From them we obtain immediately the four solutions for x by subtract-
     ing a/4 (since y was defined as x = y - a/4). */
  solutions[0] = (+ H + sqrt(f1 + f2)) - (a/4.0);
  solutions[1] = (+ H - sqrt(f1 + f2)) - (a/4.0);
  solutions[2] = (- H + sqrt(f1 - f2)) - (a/4.0);
  solutions[3] = (- H - sqrt(f1 - f2)) - (a/4.0);
}

/* *********************************************************************
   | This routine calculates the so called "concurrence" for a bipar-  |
   | tite quantum system. The indexes "system_1" and "system_2" select |
   | the two quantum systems which are involved in the calculation.    |
   | If we call "rho" the reduced density matrix of the bipartite      |
   | system and "rhotilde" the spin flipped density matrix (that is    |
   | rhotilde = (SxS) rho^* (SxS), where ^* is the conjugation of all  |
   | the matrix elements, x is the tensor product sign and S is the    |
   | Pauli matrix Y), then the concurrence is defined as the maximum   |
   | between zero and L_1 - L_2 - L_3 - L_4, where the L_i's are the   |
   | square roots (in decreasing value order) of the eigenvalues of    |
   | R = rho rhotilde. The matrix rho rhotilde is not guaranteed to be |
   | hermitian, but it must be positive, so all its eigenvalues are    |
   | real non-negative numbers (so that taking the square root is not  |
   | ambiguous, the L_i's must be real non-negative numbers).          |
   | ----------------------------------------------------------------- |
   | If the two system indexes are out of range or equal, the routine  |
   | returns immediately with a zero value.                            | 
   | ----------------------------------------------------------------- |
   | Reference: the "concurrence" is described and shown to determine  |
   | the entanglement of formation for mixed states by W.K.Wooters,    |
   | "Entanglement of formation of an arbitrary state of two qubits",  |
   | Phys.Rev.Lett. 80 (1998) 2245-2248, quant-ph/9709029.             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
rnumber concurrence(unsigned long system_1, unsigned long system_2) {
  /* these defines save some typing later on. */
#define forall(i) for (unsigned long i = 0; i < 4; ++i)
#define max(a,b) ((a>b) ? a : b)
  /* the two systems must be distinct. */
  if (system_1 == system_2) return 0.0;
  /* the two indexes must be in the valid range. */
  unsigned long num_qubits = simulator.qubits();
  if (system_1 >= num_qubits || system_2 >= num_qubits) return 0.0;
  /* allocate three 4x4 matrices for later use. "rho" will be the
     reduced density matrix, "rhotilde" the spin flipped version of
     rho and "R" their product (this does not meet the notation of
     quant-ph/9709029 where R is actually a square root of our R). */
  cnumber rho[4][4], rhotilde[4][4], R[4][4];
  /* this section calculates the reduced density matrix rho. First, 
     prepare a bitmask "mask" with two 1's indicating the position
     of the qubits we want to select. */
  unsigned long mask = (0x1 << system_1) | (0x1 << system_2);
  /* for all the possible basis states for the bipartite system (they are
     four) prepare another bitmask with 0 or 1 in the system_i-th position
     depending on the system_i being |0> or |1>, and zero elsewhere. */
  for (unsigned long row_1 = 0; row_1 < 2; ++row_1)
    for (unsigned long row_2 = 0; row_2 < 2; ++row_2) {
      unsigned long state_row = 2 * row_1 + row_2;
      unsigned long mask_row  = (row_1 << system_1) | (row_2 << system_2);
      /* do exactly the same thing with different variables
	 (which will select columns of rho instead of rows). */
      for (unsigned long col_1 = 0; col_1 < 2; ++col_1)
	for (unsigned long col_2 = 0; col_2 < 2; ++col_2) {
	  unsigned long state_col = 2 * col_1 + col_2;
	  unsigned long mask_col  = (col_1 << system_1) | (col_2 << system_2);
	  /* I want to try to reduce numeric noise a bit here. In particular,
	     I want to kill those density matrix elements which are different
	     from zero only because of numeric noise. The order of magnitude
	     of these errors should be DBL_EPSILON, i.e. the difference
	     between 1.0 and the minimum double greater than 1.0, and 
	     simulator.size() of them could occur (they should add up 
	     quadratically). Summing and subtracting this value should
	     work, so we initialise the density matrix element with a fake
	     value which will be subtracted at the end.*/
	  double  fake_bigreal    = 2*sqrt(simulator.size());
	  cnumber fake_bigcomplex = cnumber(fake_bigreal, fake_bigreal);
	  rho[state_row][state_col] = fake_bigcomplex;
	  /* prepare all the possible n bit strings, with n being the number
	     of qubit subsystems managed by the simulator. */
	  for (unsigned long x = 0; x < simulator.size(); ++x) {
	    /* for each bit string x, erase the bits corresponding to the
	       selected subsystems and overwrite them with the content of
	       mask_row and mask_col. This algorithm generates all the 
	       n bit strings with the constraint that the system_i-th bit
	       is set to the state (|0> or |1>) of the sub-system_i. Note
	       that each pair of strings is generated four times. */
	    unsigned long x_1 = ((x & ~mask) | mask_row);
	    unsigned long x_2 = ((x & ~mask) | mask_col);
	    /* add to the rho entry the contribution from these two bit
	       strings (I assume that this entry has already been initialised 
	       to zero). Weight each contribution as 1/4 since each pair
	       of strings is generated four times. */
	    rho[state_row][state_col] +=
	      ( simulator[x_1] * conj(simulator[x_2]) ) / 4.0;
	  }
	  /* subtract the fake value which was used as initial value for
	     the density matrix element. This should dump numeric noise ... */
	  rho[state_row][state_col] -= fake_bigcomplex;
	}
    }
  /* this is the calculation of the "splin-flipped" density matrix
     (rhotilde). Since SxS, where S=sigma_Y, has only four nontrivial
     elements, the calculation is very simple: infact the (i,j) element
     of SxS is equal to kronecker_delta(i,3-i)*signs[i], so that the
     (i,j) element of rhotilde is rho(3-i,3-j) conjugated times the
     appropriate sign (signs[i]*signs[j]). */
  const rnumber signs[4] = {-1., +1., +1., -1.};
  forall(i) forall(j) rhotilde[i][j] = conj(rho[3-i][3-j]) * signs[i]*signs[j];
  /* the matrix R is rho rhotilde: we need rows by column multiplication. */
  forall(i) forall(j)
    { R[i][j] = 0.0; forall(k) R[i][j] += rho[i][k] * rhotilde[k][j]; }
  /* we need to find the eigenvalues of R now; therefore, we calculate the
     characteristic polynomial and solve it (it is a fourth degree equation).
     Prepare two arrays for holding the coefficients and the roots. */
  cnumber coefficients[4] = {0., 0., 0., 0.};
  cnumber solutions[4];
  /* There is a simple iterative procedure which finds all the
     coefficients of this polynomial, starting from the third degree term
     (the fourth degree's one is set to 1). At the end of the procedure
     the temporary matrix F must be zero (Cayley-Hamilton theorem). */
  cnumber F[4][4], G[4][4];                         // temporary matrices
  forall(i) forall(j) F[i][j] = R[i][j];            // init F to R
  forall(h) {
    forall(i) coefficients[h] -= (F[i][i] / (h + 1.0));
    forall(i) forall(j) G[i][j] = F[i][j] + ((i==j) ? coefficients[h] : 0.);
    forall(i) forall(j) { F[i][j] = 0.0; forall(k) F[i][j] += R[i][k]*G[k][j];}
  }
  /* call an appropriate routine for finding the eigenvalues, i.e.
     the roots of the characteristic polynomial. They are saved in
     no particular order in the "solutions" array. */
  solution_4th_degree(coefficients, solutions);
  /* R is a positive matrix, so its eigenvalues must be real non negative
     numbers. Calculate and save their square roots in the "eigen" array.
     These are the L_i's described in the introduction to this method, the
     eigenvalues of the hermitian matrix sqrt(sqrt(rho) rhotilde sqrt(rho)).
     Due to round-off errors, the imaginary part of the "solutions" can
     be non-zero and the real part can be negative, so we need to apply
     a small correction ... */
  rnumber eigen[4];
  forall(i) eigen[i] = sqrt(max(solutions[i].real(), 0.0));
  /* do a quick search for the largest eigenvalue. */
  rnumber max_eigen = 0.0;
  forall(i) if (eigen[i] > max_eigen) max_eigen = eigen[i];
  /* now calculate twice the largest eigenvalue minus all the eigenvalues.
     This is exactly L_1 - L_2 - L_3 - L_4 where L_1 is the largest one. */
  rnumber concurrence = 2.0 * max_eigen;
  forall(i) concurrence -= eigen[i];
  /* if the previous number is positive, it is the concurrence of the
     bipartite system. If it is negative, the concurrence is zero. */
  return max(concurrence, 0.0);
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
