/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <qop_slicelist.h>                          // class declaration

/* *********************************************************************
   | This filter tries to simplify consecutive adjoint slices (that is |
   | two slices with adjoint base gates and equivalent address lists). |
   | The slices concerned by this filter are "*a_slice" and the follo- |
   | wing one. If the simplification is possible, the two slices are   |
   | removed from the list, the "a_slice" iterator (which is passed by |
   | reference) is set to the slice preceeding the pair (if possible)  |
   | and the return value is set to "true"; otherwise it is set to     |
   | false and nothing happens.                                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Dec 2001         |
   ********************************************************************* */
bool Qop_slicelist::simplify_adjoint_slices(iterator &a_slice) {
  /* test that the current iterator is not the past-the-end one. */
  if (a_slice == end()) return false;
  /* now take an iterator to the following slice. If it is the
     past-the-end iterator, stop mumbling and return. */
  iterator next_slice = a_slice;
  ++next_slice;
  if (next_slice == end()) return false;
  /* the following test checks both that the base gates are adjoint
     and that the address lists are equivalent. If it fails, return. */
  if (! (**a_slice).is_adjoint_of(**next_slice)) return false;
  /* the simplification is possible: erase the second slice in the pair */
  erase(next_slice);
  /* erase the first slice in the pair; the return value is an iterator
     pointing to the following slice: it must be saved in order to have
     a new starting point. */
  a_slice = erase(a_slice);
  /* move the iterator one step back. Be sure however not to
     do so if we are already at the beginnig of the list. */
  if (a_slice != begin()) --a_slice;
  /* return positively */
  return true;
}

/* *********************************************************************
   | This filter tries to simplify 
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Sep 2002         |
   ********************************************************************* */
bool Qop_slicelist::simplify_adjoint_basegates(iterator &a_slice) {
  /* test that the current iterator is not the past-the-end one. */
  if (a_slice == end()) return false;
  /* now take an iterator to the following slice. If it is the
     past-the-end iterator, stop mumbling and return. */
  iterator next_slice = a_slice;
  ++next_slice;
  if (next_slice == end()) return false;
  /* test that it is a single qubit slice */
  //  if (GET_OPTYPE((*a_slice)->get_opcode()) != OPTYPE_SINGLE) return false;
  /* if the base gates are not adjoint, return immediately. */
  //if (! (*a_slice)->has_adjoint_basegate(**next_slice)) return false;
  /* annihilate common parts of the lists. */
  //  ((*a_slice)->get_list(0)).reorder_and_annihilate((*next_slice)->get_list(0));

  
  return false;
}

/* *********************************************************************
   | This method runs an algebraic simplification of the quantum ope-  |
   | rations in the slice list. It should be designed to be very fast  |
   | better than very efficient and to be extensible. The passed ite-  |
   | rator can be invalidated by the routine if any simplification     |
   | takes place (i.e. if the return value is true).                   |
   | ----------------------------------------------------------------- |
   | This routine is only an experiment, just to see what can be done  |
   | with a bit of automatic simplification. The current strategy is   |
   | very simple: try to simplify those operation at border of two     |
   | recently composed operators which are adjoint and which have the  |
   | same address lists. The slice immediately preceeding the border   |
   | is pointed to by "a_junction".                                    |
   | ----------------------------------------------------------------- |
   | TODO:                                                             |
   | - simplify the lists of adjoint operations with partial overlap.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Dec 2001         |
   ********************************************************************* */
bool Qop_slicelist::simplify(const iterator &a_junction) {
  /* the following flag will be set to true if any simplification
     takes place while this routine is running. Initialise to false. */
  bool was_simplified = false;
  /* initialise an iterator to the "left side" of the border. The
     simplification will start from here. This allows us to scan only
     the sensible region, not the whole list (this is the meaning of
     "border" simplification). */
  iterator a_slice = a_junction;
  /* loop until some condition breaks the loop. */
  do { was_simplified = simplify_adjoint_slices(a_slice); }
  while (was_simplified == true);
  /* return true if any simplification took place. */
  return was_simplified;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
