#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <vector>
#include <getopt.h>
#include <quantum_computing.h>
#include <quantum_library.h>
#include <qsimulator.h>

// calculation of concurrence
double concurrence(unsigned long system_1, unsigned long system_2);

// =============================================================
// A program for simulating some classical and quantum maps
// on a quantum computer (using the Q language and a simulator).
// =============================================================
int main(int argc, char *argv[]) {
  // enums and explanatory strings
  enum algotypes {SAWTOOTH, DWELL, ARNOLD, ALG_END};
  const char *algorithms[] = {"SAWTOOTH MAP", "DOUBLE WELL MAP", "ARNOLD MAP"};
  enum inittypes {STEP, CAT2_EXT, CAT2_INT, INI_END};
  const char *initstates[] = {"STEP DISTR. |10>+|11>", "2 QUBIT CAT |00>+|11>",
			      "2 QUBIT CAT |01>+|10>"};
  enum wavetype {NOWAVE, PROBABILITIES, AMPLITUDES, WAVE_END};
  const char *waveinfos[] = {"not printed", "probabilities", "amplitudes"};
  // all the control variables for this program
  int    type  = SAWTOOTH; // algorithm type 
  int    q     = 8;        // resources (# of qubits)
  int    c     = 4;        // number of simulated momentum cells
  int    niter = 500;      // number of map iterations
  double noise = 0.00;     // noise level for noisy gates
  double st_d  = 0.00;     // noise level for static errors (delta's)
  double st_J  = 0.00;     // noise level for static errors (J's)
  int    reps  = 0;        // number of measurements (0 = ask simulator)
  int    exps  = 1;        // number of "experiments"
  double K     = 0.04;     // K = k * T defines the classical map
  double a     = 1.6;      // "centre" of the potential (2*PI*a/L)
  int    R     = 0;        // measure in momentum (0) or position (1)
  int    I     = CAT2_EXT; // initial state
  bool   erfk  = false;    // use the same random seq. at every step
  int    wave  = NOWAVE;   // data about the wave function
  bool   Tsym  = false;    // exact or approx. time reversal symmetry

  /* Parse the option string and update the algorithm parameters. */
  while (true) {
    switch (getopt(argc, argv, "q:c:S:e:d:J:M:E:K:a:A:R:I:PTW:")) {
    case  -1: goto endopt;
    case 'q': q     = atoi(optarg);        break;
    case 'c': c     = atoi(optarg);        break;
    case 'S': niter = atoi(optarg);        break;
    case 'e': noise = strtod(optarg,NULL); break;
    case 'd': st_d  = strtod(optarg,NULL); break;
    case 'J': st_J  = strtod(optarg,NULL); break;
    case 'M': reps  = atoi(optarg);        break;
    case 'E': exps  = atoi(optarg);        break;
    case 'K': K     = strtod(optarg,NULL); break;
    case 'a': a     = strtod(optarg,NULL); break;
    case 'A': type  = atoi(optarg);        break;
    case 'R': R     = atoi(optarg);        break;
    case 'I': I     = atoi(optarg);        break;
    case 'P': erfk  = true;                break;
    case 'T': Tsym  = true;                break;
    case 'W': wave  = atoi(optarg);        break;
    }}
 endopt:
  /* some consistency checks */
  int N = (1 << q);
  int num_qubits = q;
  if (type == DWELL) ++num_qubits;
  if (type >= ALG_END) { std::cerr << "Invalid type " <<type<<'\n'; exit(-1); }
  if (q < 3) { std::cerr << "Too few qubits.\n"; exit(-1); }
  /* setup the quantum simulator (which must manage "num_qubits" qubits)
     and evolve them with noisy gates with noise intensity "noise"
     and static errors given by "st_d" and "st_J". */
  simulator.set_numqubits(num_qubits);
  simulator.set_noise(noise, st_d, st_J);
  /* We are going to use quantum primitives now ... */
  try {
    /* The following two operators commute from position to momentum
       representation and back. Since these two representations are
       conjugate, these Qops are simply Fourier transforms. */
    Qop mom_to_pos = QFourier(q);
    Qop pos_to_mom = !mom_to_pos; // this is an anti-transform
    /* Coefficient and operator for the free evolution of any quantum map.
       In a more standard notation, this is the kinetic term exp(-iTn^2/2),
       therefore we get T = 2*PI*c / N . If Tsym is true, add a few gates
       in order to have a perfect p symmetry (i.e.  i --> i+1/2). */
    double p_coeff = - c/(2.0*N);
    Qop rotator = expower(q, 2, p_coeff);
    if (Tsym) rotator &= expower(q, 1, (1-N)*p_coeff);
    /* A generic operator for the potential term of a quantum map. */
    Qop kick;
    /* This is the construction of the "kick" for the sawtooth map
       algorithm. In this case we set -(1/2)(i+D)^2 = sum_j f_j i^j.
       The choice D = 0.5*(1-N) guarantees that we take a coordinate
       range which is symmetric around zero. */
    if (type == SAWTOOTH) {
      double D = 0.5 * (1.0-N);         // x_i = (L/N)(i + D)
      
      double f  = - K / (double(c)*N);  // common factors
      double f1 = - f * D;              // f_1 * f
      double f2 = - f / 2;              // f_2 * f
      
      kick = expower(q, 1, f1) & expower(q, 2, f2);
    }
    /* This is the construction of the "kick" for the double well map
       algorithm. Remember that ((i+D)^2 - A^2)^2 = sum_j f_j i^j.
       With q=5 qubits [+1ancilla] this Qop contains 376 gates, which
       can shrink to 251 gates with further optimisations [TODO]).
       The choice D = 0.5*(1-N) guarantees that we take a coordinate
       range which is symmetric around zero. */
    if (type == DWELL) {
      double lb = 2*M_PI;           // adimensional cell size
      double A = a * (N/lb);        // scaled "centre" for the potential
      double D = 0.5 * (1-N);       // x_i = (L/N)(i + D)
      
      double f = - (lb*lb * K)/(double(c)*N*N*N); // common factors
      double f1 = f * 4*D * ( D*D - A*A );  // f_1 * f
      double f2 = f * ( 6*D*D - 2*A*A );    // f_2 * f
      double f3 = f * 4*D;                  // f_3 * f
      double f4 = f;                        // f_4 * f

      kick = (expower(q, 1, f1) & expower(q, 2, f2) &
      	      expower(q, 3, f3) & expower(q, 4, f4));
    }
    /* Arnold map: TODO */
    if (type == ARNOLD) {
      std::cerr << algorithms[type] << " not yet ready.\n"; exit(-1);
    }
    /* The following operator is a complete map step. Its definition
       depends on the initial representation (0=momentum or 1=position). */
    Qop mapstep = ((R == 1) ?
		   kick & pos_to_mom & rotator & mom_to_pos :
		   rotator & mom_to_pos & kick & pos_to_mom);
    /* The "preparation" operator will be used to prepare the initial
       state for each experiment. It is supposed to operate on the 
       q-qubits register which holds the discretisation. */
    Qop preparation[INI_END];
    preparation[STEP]     = QNot(1) & QHadamard(q-1)>>1;
    preparation[CAT2_EXT] = QHadamard(1) & QCnot(1) & QHadamard(q-2)>>2;
    preparation[CAT2_INT] = preparation[CAT2_EXT] & QNot(1)>>1;
    /* Print out some choices which are difficult to remember. */
    std::cerr << "Algorithm types: ";
    for (int j=0; j<ALG_END; ++j)
      std::cerr << "\n\t" << j << " - " << algorithms[j] << " ";
    std::cerr << "\nInitial states: ";
    for (int j=0; j<INI_END; ++j)
      std::cerr << "\n\t" << j << " - " << initstates[j] << " ";
    std::cerr << "\nWave function information:";
    for (int j=0; j<WAVE_END; ++j)
      std::cerr << "\n\t" << j << " - " << waveinfos[j] << " ";
    std::cerr << "\n";
    /* Print out some parameters as a summary. */
    std::cerr
      << "\n[A]  Algorithm        : " << algorithms[type]
      << "\n[q]  Discretisation   : " << q << " qubits"
      << "\n[c]  Number of cells  : " << c
      << "\n[K]  Value of K=kT    : " << K
      << "\n[a]  V(x) centered on : " << a
      << "\n[R]  Representation   : " <<((R)?"position (1)":"momentum (0)")
      << "\n[I]  Initial state    : " << initstates[I]
      << "\n[e]  Noisy gates      : " << noise
      << "\n[d]  Static (Zeeman)  : " << st_d
      << "\n[J]  Static (coupl.)  : " << st_J
      << "\n[M]  Measur. per step : " << reps
      << "\n[E]  Number of exp.s  : " << exps
      << "\n[S]  Number of iter.s : " << niter
      << "\n[P]  Error type       : " << ((erfk)?"pseudostatic":"random")
      << "\n[T]  Time rev. symm.  : " << ((Tsym)?"exact":"approximate")
      << "\n[W]  Wavefunction     : " << waveinfos[wave]
      << std::endl;
    /* print statistics for one map step */
    std::cerr << "\nQubit utilisation for one map step:\n";
    mapstep.print_statistics(std::cerr);
    /* Prepare some arrays for intermediate results. Also prepare
       arrays for the wave function, but dimension them trivially
       if they are not necessary (they can become very large!). */
    typedef std::vector<Qsimulator::rnumber> dblvec;
    typedef std::vector<Qsimulator::cnumber> cpxvec;
    dblvec results(niter, 0.0), concurr(niter, 0.0);
    std::vector<cpxvec> waveinfo(((wave!=NOWAVE)?niter:0), cpxvec(N, 0.0));
    /* repeat "exps" times the whole evolution (multiple steps) */
    for (int experiment = 0; experiment < exps; ++experiment) {
      /* initialise the random seed and save the result. */
      unsigned long seed = simulator.init_random_seed();
      /* if static errors are on, reinitialise them at every experiment. */
      simulator.regenerate_staterr();
      /* reset the register */
      Qreg simul(q);
      /* prepare the initial state */
      preparation[I](simul);
      /* now iterate the map "niter" times and extract
	 some physical quantities at each step. */
      for (int count = 0; count < niter; ++count) { 
	/* if erfk is true, restart the random sequence from "seed". This
	   means that the errors are random only within one sequence. */
	if (erfk) simulator.init_random_seed(seed);
	/* get the concurrence for the most significant pair */
	Qubit_list pair = simul.locate();
	concurr[count] += (concurrence(pair[0], pair[1]) / exps);
	/* if requested, save information about the wavefunction.
	   We can save the amplitudes or the probabilities (both
	   are hosted in the same array). */
	if (wave != NOWAVE) for (int index=0; index<N; ++index)
	  waveinfo[count][index] += (1 / double(exps)) *
	    ((wave == PROBABILITIES) ?
	     simulator.probability(index) : simulator.amplitude(index));
	/* get the probability of being on the left or on the right
	   (i.e. the probability that the most significant qubit is
	   |0> or |1>). Repeat the fake measurement "reps" times. */
	double prob_on_the_left; int ones = 0;
	unsigned long msb = (simul.locate())[0];
	if (reps) {
	  for (int j=0; j<reps; ++j)
	    if (simulator.fake_measure(msb)) ++ones;
	  prob_on_the_left = double(ones)/reps;
	} else
	  /* if "reps" is zero, read the exact probability. */
	  prob_on_the_left = simulator.is_set(msb);
	/* update the average probability. */
	results[count] += (prob_on_the_left / exps);
	/* run a step of the map */
	mapstep(simul);
	/* If this is the last experiment, the averaged quantities for the
	   current map step are ready, hence print them (this gives a better
	   feedback expecially when there is only one experiment). */
	if (experiment == (exps - 1)) {
	  std::cout << (count + 1) << ' ' << results[count]
		    << ' ' << concurr[count] << ' ';
	  /* Also print the the wave function information if "wave" is set. */
	  if (wave == PROBABILITIES)
	    for (int index = 0; index < N; ++index)
	      std::cout << waveinfo[count][index].real() << ' ';
	  if (wave == AMPLITUDES)
	    for (int index = 0; index < N; ++index)
	      std::cout << waveinfo[count][index] << ' ';
	  /* terminate the current line. */
	  std::cout << std::endl;
	}
      }
      /* terminate the current experiment. */
      std::cerr << "End of experiment n: " << experiment+1 << std::endl; 
    }
  } catch_qexception;
  
  return 0;
}
