/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iomanip>                           // C++ I/O manipulators
#include <qmanager.h>                        // class declaration

/* *********************************************************************
   | This is the constructor for the Qaddress_manager class. It is     |
   | simply a wrapper around a resource allocation function. It needs  |
   | to know the amount of quantum memory to be managed.               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
Qaddress_manager::Qaddress_manager(size_type the_area_size) :
  /* initialise amount of quantum memory to be managed */
  area_size(the_area_size) {
  /* Now build the address translation table. */
  allocate_resources();
}

/* *********************************************************************
   | This is the destructor for the Qaddress_manager class. Note that  |
   | destruction is only possible if all qubits are unallocated, or we |
   | can reach undefined situations. The "undestructible" case will    |
   | cause exception "still_busy()".                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
Qaddress_manager::~Qaddress_manager() {
  /* Panic if we are requested to leave but some qubit is still allocated!. */
  if (!is_free()) throw still_busy();
  /* Clean up all the internal structures. */
  deallocate_resources();
}

/* *********************************************************************
   | This method builds the address translation table. It assumes that |
   | the amount of quantum memory to be managed is already saved in    |
   | the "area_size" variable, then it initialises a structure for     |
   | each usable qubit which will remember its allocation status.      |
   | ----------------------------------------------------------------- |
   | This structure is the main qubit list. It starts with a single    |
   | segment summary element which comprises the whole qubit array and |
   | has all usage counts set to zero. The busy list starts empty, so  |
   | we must not worry about it; for the same reason the free list     |
   | starts with a single segment comprising the whole qubit vector.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
void Qaddress_manager::allocate_resources(void) {
  /* initialise the main list with a single segment with all usage counts
     set to zero. This segment will be fragmented in the future to allow
     the creation and management of quantum registers. */
  main_list.push_back(Qubit_summary(0, area_size));
  /* initialise the free list with the single segment in the main list. */
  free_insert(main_list.begin());
}

/* *********************************************************************
   | This method is the reverse of allocate_resources(). It cleans up  |
   | all the internal structures and possibly inform the quantum device|
   | that she isn't used any more.                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
void Qaddress_manager::deallocate_resources(void) {
  /* clear the main qubit list (should have only one element anyway!) */
  main_list.clear();
  /* clear the free list (should have only one element anyway!) */
  free_list.clear();
  /* clear the busy list (should be empty anyway!). */
  busy_list.clear();
  /* clear the size map (should have only one element anyway!). */
  size_map.clear();
}

/* *********************************************************************
   | This method allocates the logical qubits from the address manager.|
   | The rationale is that a (fresh) quantum register asks for a bunch |
   | of qubits, the address manager looks for a contiguous region of   |
   | that size and initializes a list with the new segment (should we  |
   | allow also for non-contiguous regions?); finally, the list is     |
   | returned to the quantum register for its plesure.                 |
   | If allocation fails, for any reason, an exception is thrown with  |
   | allocation_failed().                                              |
   | (02 Jan 2002) S.Bettelli, explicitely prohibit qubit_number = 0   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
Qubit_list Qaddress_manager::get_qubits(size_type qubit_number) {
  /* one must request at least one qubit, zero is a nonsense */
  if (qubit_number == 0) throw allocation_failed();
  /* try to get a free segment with size equal or greater than
     qubit_number from the free pool. */
  main_iterator selected_chunk = find_free_segment(qubit_number);
  /* if we didn't succed, throw an exception */
  if (selected_chunk == main_list.end()) throw allocation_failed();
  /* erase this chunk from the free list (it won't remain free!) */
  free_erase(selected_chunk);
  /* if the new chunk is oversized, split it into two and
     put the second part back into the free list. */
  if (selected_chunk->size() > qubit_number)
    free_insert(split_direct(selected_chunk, qubit_number));
  /* increment the usage counts of the correctly sized segment. add()
     increments the segment usage count and both the minimum and maximum
     address border usage counts. This is OK, since this segment is used
     (completely) by exactly one register now. */
  selected_chunk->add();
  /* enter the correctly sized segment in the busy list. */
  busy_insert(selected_chunk);
  /* create a qubit list with this single segment (downcasting
     it to Qubit_segment) and return it to the caller. */
  return Qubit_list(*selected_chunk);
}

/* *********************************************************************
   | This method finds a free segment suitable for allocation. It is   |
   | passed a size and it must find an element in the free list with   |
   | the minimum available size greater or equal to it. If this cannot |
   | be done, the past the end iterator for the main_list is returned, |
   | otherwise an iterator to the free segment is returned. This method|
   | is (de)fragmentation friendly.                                    |
   | ----------------------------------------------------------------- |
   | The algorithm is very simple, since the address manager keeps a   |
   | list of available sizes for free segments: find the smallest size |
   | which is greater than or equal to the size requested by the user, |
   | then find an element in the free_list with that size and return   |
   | it (as far as I know all this operations have logarithmic complex-|
   | ity with respect to the size of the concerned container).         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Feb 2001         |
   ********************************************************************* */
Qaddress_manager::main_iterator Qaddress_manager::find_free_segment
(Qubit_summary::size_type a_size) {
  /* find the smallest size which is greater than or equal to the size
     requested by the user; the STL sorted containers have a specialised
     member function for this, so go for it. If no suitable size is found,
     the lower_bound() method returns the past-the-end iterator for the
     size_map: in this case abort the search and return main_list.end()
     to inform the caller that no suitable free segment could found. */
  size_map_type::iterator chosen_entry = size_map.lower_bound(a_size);
  if (chosen_entry == size_map.end()) return main_list.end();
  /* if we are here we found a suitable segment size; its value is the
     first field in the selected element (the second is the multiplicity
     of this size in the map, which we currently don't care about). */
  Qubit_summary::size_type chosen_size = chosen_entry->first;
  /* Now find one element without any particular preference in the free
     list with that size. If everything is in sync, there must be at
     least one such element, so don't check the return value. Then
     return the pointer to the free segment. */
  return free_list.find(chosen_size)->second;
}

/* *********************************************************************
   | This method accepts an reference to a Qubit_segment coming from   |
   | a register destruction (supposedly) and finds all Qubit_summaries |
   | in the main list which are covered by it (there might be more     |
   | than one since the to-be-destroyed segment could overlap other    |
   | registers). For each of them it decreases all the usage counts    |
   | appropriately and, if any segment usage count drops to zero,      |
   | the segment is removed from the busy list and inserted into the   |
   | free list. After all counters have been updated, a recollection   |
   | routine is run in order to join adjacent free segments and        |
   | adjacent busy segments with the same usage count and no actively  |
   | used border between them. This will keep the main list as compact |
   | as possible.                                                      |
   | Errors during execution (which should have been eliminated during |
   | the debug cycle) cause exception checkout_failed().               |
   | ----------------------------------------------------------------- |
   | Since Qubit_summaries are stored in the main list consecutively,  |
   | that is highest() + 1 of one of them is lowest() of the next,     |
   | finding the covered Qubit_summaries is a simple as finding that   |
   | which has the same lowest() as the supplied Qubit_segment and     |
   | iterating till we find one which has the same highest().          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 27 Jan 2001         |
   ********************************************************************* */
void Qaddress_manager::checkout_qubits(const Qubit_segment &a_segment) {
  /* using the busy list, find the Qubit_summary in the main list which
     has the same lowest address as the supplied segment. Check that the
     returned iterator is valid and issue exception in case. */
  busy_list_type::iterator busy_link = busy_list.find(a_segment.lowest());
  if (busy_link == busy_list.end()) throw checkout_failed();
  main_iterator busy_segment = busy_link->second;
  main_iterator first_segment = busy_segment;
  /* decrement the minimum address ("left") border usage count for
     the first segment in this group. */
  busy_segment->del_left_border();
  /* now try to free all Qubit_summaries in the main list which are
     covered by the Qubit_segment. The termination test is that the last
     Qubit_summary must have the same highest address as the Qubit_segment. */
  while (true) {
    /* be sure we are not past the end in the main list. */
    if (busy_segment == main_list.end()) throw checkout_failed();
    /* if everything is working fine, the "busy_segment" must be really
       busy, otherwise something got out of sync. Test it and throw
       an exception if the test fails. */
    if (busy_segment->is_free()) throw checkout_failed();
    /* decrement the segment usage count (call del_bulk(), not
       del(), which would decrement all the counters!). */
    busy_segment->del_bulk();
    /* if the segment usage count drops to zero, remove it from
       the busy list and insert it into the free list. */
    if (busy_segment->is_free()) {
      busy_erase(busy_segment);
      free_insert(busy_segment);
    }
    /* if the highest address is the same as for a_segment, stop iterating */
    if (busy_segment->highest() == a_segment.highest()) break;
    /* otherwise go on with the following Qubit_summary in the main list. */
    ++busy_segment;
  }
  /* decrement the highest address border usage count for the last
     segment in this group. */
  busy_segment->del_right_border();
  /* now run our beautiful recollection routine. It only needs to know the
     range which was modified (between the initial value first_segment and
     its current final value. The routine has its own internal checks, so
     we don't have to worry. */
  recollect_segments(first_segment, busy_segment);
}

/* *********************************************************************
   | This method accepts a reference to a Qubit_segment which is sup-  |
   | posed to be already mapped and increments its usage counts.       |
   | This means it finds all Qubit_summaries in the main list which    |
   | are covered by it (there might be more than one since the segment |
   | could overlap other registers) and, for each of them, it increa-  |
   | ses the segment usage count. The routine also increments the mi-  |
   | nimum and maximum address border usage count for the first and    |
   | last corresponding segments.                                      |
   | Errors during execution (which should have been eliminated during |
   | the debug cycle) cause exception checkin_failed().                |
   | ----------------------------------------------------------------- |
   | See checkout_qubits() for details on finding Qubit_summaries      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 03 Feb 2001         |
   ********************************************************************* */
void Qaddress_manager::checkin_qubits(const Qubit_segment &a_segment) {
  /* using the busy list, find the Qubit_summary in the main list which
     has the same lowest address as this Qubit_segment. Check that the
     returned iterator is valid and issue exception in case. */
  busy_list_type::iterator busy_link = busy_list.find(a_segment.lowest());
  if (busy_link == busy_list.end()) throw checkin_failed();
  main_iterator busy_segment = busy_link->second;
  /* increment the minimum address border usage count for the first
     segment in this group. */
  busy_segment->add_left_border();
  /* now try to increment the segment usage count for all Qubit_summaries
     in the main list which are covered by the Qubit_segment. The termination
     test is that the last Qubit_summary must have the same highest
     address as the Qubit_segment. */
  while (true) {
    /* be sure we are not past the end in the main list. */
    if (busy_segment == main_list.end()) throw checkin_failed();
    /* if everything is working fine, the "busy_segment" must be really
       busy, otherwise something got out of sync. Test it and throw
       an exception if the test fails. */
    if (busy_segment->is_free()) throw checkin_failed();
    /* increment the segment usage count; call add_bulk(),
       not add(), which would increment all the counters! */
    busy_segment->add_bulk();
    /* if the highest address is the same as for a_segment, stop iterating */
    if (busy_segment->highest() == a_segment.highest()) break;
    /* go on with the following Qubit_summary in the main list. */
    ++busy_segment;
  }
  /* increment the highest address border usage count for the last
     segment in this group. */
  busy_segment->add_right_border();
}

/* *********************************************************************
   | This method receives a range inside the main list (two iterators  |
   | to the first and last element), whose elements internal counts    |
   | have been modified by a previous checkout_qubits() call, and runs |
   | a recollection procedure to keep the main list as compact as      |
   | possible. Two kinds of recollections take place:                  |
   | 1) consecutive free segments must be joined in a single (larger)  |
   |    free segment; this maximises the possibility of finding space  |
   |    for big register allocation in get_qubits().                   |
   | 2) adjacent busy segments with the same bulk usage count can be   |
   |    joined if they have zero border usage counts at the common     |
   |    border (this saves the main list from becoming more and more   |
   |    fragmented during its life cycle).                             |
   | ----------------------------------------------------------------- |
   | a) Adjacent segments are to be joined with the try_join() method. |
   |    This method cannot fail, since segments are consecutive by     |
   |    sure in the main list; therefore we will always neglect the    |
   |    return value.                                                  |
   | b) The try_join method works by "copying" the addresses from the  |
   |    "second" segment onto the "first". "second" is the segment     |
   |    with the highest addresses, and must be erased after the call, |
   |    because it remains dummy. In order not to have problems while  |
   |    dereferencing iterators we will iterate through the range in   |
   |    reverse order (due to the way erase() works).                  |
   | (01 Jan 2002) S.Bettelli, small fix for the VERY unlikely case    |
   |   that the main list has exactly one element ....                 |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 May 2001         |
   ********************************************************************* */
void Qaddress_manager::recollect_segments
(main_iterator &first_segment, main_iterator &last_segment) {
  /* The algorithm works with pairs of segments, so it really can't work
     if the main list has less than two segments! Since we assume that
     the two iterators are dereferenceable, this means that the main list
     can't be empty (it can never be emtpy anyway). The only case to 
     exclude is when the main list has exactly one segment. */
  if (main_list.begin() == --main_list.end()) return;
  /* we are going to check each segment in the supplied range to see if
     it is possible to join it with its successive. This would actually
     avoid the joining of the first segment of the range with its
     predecessor, which is a possibility. So, move the first_segment
     iterator back one position, unless it is already the first one. */
  if (first_segment != main_list.begin()) --first_segment;
  /* in analogy with the previous reasonment, we must move the last_segment
     back one position if it points to the last segment in the main list,
     in order to avoid an undefined dereferentiation. */
  if (last_segment == (--main_list.end())) --last_segment;
  /* now the range is safe, and each element of the range could be joined
     with its successive. Prepare two iterators for holding a pair. */
  main_iterator former_segment, latter_segment;
  /* A successfull join will leave a dummy segment in the list (the second
     segment of a pair) which we must remove. In order not to have problems
     with the removal, we will traverse the list in reverse order.*/
  former_segment = last_segment;
  while (true) {
    /* calculate an iterator to the successive segment with respect to
       the current segment. This is the second of the pair. */
    latter_segment = former_segment;
    ++latter_segment;
    /* 1) first possible join: if both segments are free put
       -- them togheter. This is a quite easy test. */
    if (former_segment->is_free() &&
	latter_segment->is_free()   ) {
      /* OK, the current segment and the next one are both free.
	 The strategy is: remove both from the free list, join the
	 former and the latter, remove the latter which is now dummy
	 from the main list and reinsert the former into the free list. */
      free_erase(former_segment);                // put the former in limbo
      free_erase(latter_segment);                // put the latter in limbo
      former_segment->try_join(*latter_segment); // join the two segments
      main_list.erase(latter_segment);           // erase the latter segment
      free_insert(former_segment);               // reinsert the former
    }
    /* 2) second possible join: two consecutive segments with the same
       -- bulk usage counts could be joined. But we must check that the
       -- border in the middle is not used by anyone. Don't run this
       test if the previous one was successful, because the latter_segment
       is in that case not dereferenceable. */
    else if ((former_segment->get_bulk() ==
	      latter_segment->get_bulk())              &&
	     (former_segment->get_right_border() == 0) &&
	     (latter_segment->get_left_border()  == 0)   ) {
      /* OK, the current segment and the next one have the same usage
	 count and there is no reason for keeping them apart. 
	 The strategy is: remove both from the busy list, adjust the
	 right border usage count for the former, join the former and
	 the latter, remove the latter which is now dummy from the main
	 list, and reinsert the former into the busy list. */
      busy_erase(former_segment);                // put the former in limbo
      busy_erase(latter_segment);                // put the latter in limbo
      former_segment->set_right_border           // adjust border usage
	(latter_segment->get_right_border());
      former_segment->try_join(*latter_segment); // join the two segments
      main_list.erase(latter_segment);           // erase the latter segment
      busy_insert(former_segment);               // reinsert the former
    }
    /* if the former segment is already the first one, time to leave. */
    if (former_segment == first_segment) break;
    /* otherwise continue (remember we are traversing the range in the
       reverse order). */
    --former_segment;
  }
}

/* *********************************************************************
   | This method will be called by Qreg objects inside the operator[]  |
   | or operator() methods which create quantum registers referencing  |
   | qubits of already existing registers. In these cases the address  |
   | manager must be informed because we must be sure that enough      |
   | entries are present in the busy list and that the "union"         |
   | assumption is preserved. "target" is the address where fragment   |
   | should begin; "reference" is the beginning of an existing segment |
   | (you can therefore find it in the busy list) whose address is     |
   | less or equal to "target", possibly as close to "target" as the   |
   | calling method can determine. Three types of fragmentation are    |
   | possible:                                                         |
   |    BEGIN:  fragment [... target-1] [target ...]                   |
   |    END:    fragment [... target] [target+1 ...]                   |
   |    SINGLE: fragment [... target-1] [target] [target+1 ...]        |
   | A generic problem causes exception fragment_failed().             |
   | ----------------------------------------------------------------- |
   | Note that this method doesn't increase usage counters, it only    |
   | fragments appropriately the main list.                            |
   | (28 Mar 2001) Bettelli, all fragments must have same usage count! |
   | (23 Apr 2001) Bettelli, the previous note is no more valid now    |
   |   that we have three different usage counts. It is OK to copy the |
   |   segment (bulk) usage count, but new borders must get a zero     |
   |   usage count, on both sides. In any case, counters will be in-   |
   |   creased only by a checkin_qubits() call.                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Feb 2001         |
   ********************************************************************* */
void Qaddress_manager::fragment(address target, address reference,
				fragment_type type) {
  /* due to segment fragmentation in the main list, we are not guaranteed
     that its segment which begins at "reference" (found by using the busy
     list) really contains the "target", but the containing segment must
     at least be one of its followers. Perfom a search (the called routine
     checks passed arguments and error conditions and can throw exceptions). */
  main_iterator a_segment = search_segment(target, reference);
  /* check for the segment not being free. If it is free, the routine
     cannot go on. Throw an exception if case this happens. */
  if (a_segment->is_free()) throw fragment_failed();
  /* get the usage counts for the segment we found. We will use these
     numbers later to create all fragments with the same segment usage
     count and appropriate bulk usage counts (well, given the way the
     split() method works, we don't need to save the left count). */
  Qubit_summary::counter_type usage, right_usage;
  usage       = a_segment->get_bulk();
  right_usage = a_segment->get_right_border();
  /* now erase the busy list reference to this main list segment,
     because we want to manipulate it at open hearth. */
  busy_erase(a_segment);
  /* calculate the sizes of the two parts of the current segment:
     if fragmentation type is BEGIN or SINGLE, the former will contain
     all the qubits up to but not including the target, otherwise
     (type == END) it includes also the target. */
  size_type size_former = a_segment->index(target);
  if (type == END) ++size_former;           // patch if type is END
  size_type size_latter = a_segment->size() - size_former;
  /* break the segment up into two parts; this operation is not worth the
     effort if the size of the former or latter subsegment would be zero. 
     If however we decide to break, remember to reset the border usage
     counts on both sides of the border, that is, set to zero the right
     border counter for the former segment and the left border counter
     for the right segment. Also remember to copy the segment usage count. */
  if (size_former != 0 && size_latter != 0) {
    split_direct(a_segment, size_former);   // split the two segments
    busy_insert(a_segment);                 // put the former back to busy list
    a_segment->set_right_border(0);         // reset right border count
    ++a_segment;                            // advance iter. to the latter one
    a_segment->set(usage, 0, right_usage);  // fix all usage counts
  }
  /* if fragmentation type is SINGLE, fragment again the remaining part
     with former_size = 1. Also test that the remaining size is not
     already one (skip in case). Fix usage counts as before. */
  if ((type == SINGLE) && (a_segment->size() != 1)) {
    split_direct(a_segment, 1);             // split the two segments
    busy_insert(a_segment);                 // put the former back to busy list
    a_segment->set_right_border(0);         // reset right border count
    ++a_segment;                            // advance iter. to the latter one
    a_segment->set(usage, 0, right_usage);  // fix all usage counts
  }
  /* whatever is remaining, put it back to the busy list. */
  busy_insert(a_segment);
}

/* *********************************************************************
   | This method finds a segment in the main list containing a given   |
   | address once given another address which is guaranteed to be less |
   | than or equal to it and corresponding to an entry in the busy     |
   | list. This operation is useful when we know in which segment of a |
   | register an address was located and want to find it quickly in    |
   | main list (where that segment could be fragmented). It is called  |
   | by the fragment methods. If the search fails, it throws exception |
   | search_failed().                                                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 07 Feb 2001         |
   ********************************************************************* */
Qaddress_manager::main_iterator
Qaddress_manager::search_segment(address target, address reference) {
  /* check in advance if "reference" is less or equal than "target".
     If the condition is not met, throw an exception. */
  if (reference > target) throw search_failed();
  /* resort to the busy list to find the segment beginning at "reference"
     in the main list. This operation should not fail. */
  busy_list_type::iterator an_entry = busy_list.find(reference);
  /* However, if it failed, throw an exception ... (and debug). */
  if (an_entry == busy_list.end()) throw search_failed();
  /* Now go straight to the segment in the main list. */
  main_iterator a_segment = an_entry->second;
  /* iterate inside the main list till we find the segment which contains
     the target address (hope this is fast, but I have no bound). */
  while (a_segment->highest() < target) ++a_segment;
  /* return the segment we have found. */
  return a_segment;
}

/* *********************************************************************
   | The split() method creates two entries from one in the main list, |
   | preserving the total size. After the method has been called, the  |
   | element pointed to by the iterator has size "new_size_first" and  |
   | it is followed by a new element with the remaining size. If the   |
   | requested new_size_first is larger than or equal to the size of   |
   | first_element the operation throws exception split_error().       |
   | It is assumed that segments are DIRECT (= not ordered) !          |
   | (31 Dec 2001) S.Bettelli, this method now returns an iterator to  |
   |   the second chunk. It also issue the exception if new and old    |
   |   size are the same (because split_direct is private and this     |
   |   condition shoul never happen).                                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 09 Jul 2001         |
   ********************************************************************* */
Qaddress_manager::main_iterator Qaddress_manager::split_direct
(main_iterator first_element, size_type new_size_first) {
  /* get the old size of the referenced element. */
  size_type old_size_first = first_element->size();
  /* first check that we have enough available space, otherwise die.
     Die also in case the size is already OK (should not be called!) */
  if (old_size_first <= new_size_first) throw split_error();
  /* since STL insertion works backward we must take an iterator to
     the first element and advance it by one. */
  main_iterator second_element = first_element;
  ++second_element;
  /* insert the second part of the splitted segment into the main list.
     The lowest address is the lowest address of the original segment
     offset by new_size_first; the size is the differenze between the
     old and the new size of the first part. Save the return value. */
  second_element = main_list.insert
    (second_element, Qubit_summary(first_element->lowest() + new_size_first,
				   old_size_first - new_size_first));
  /* resize the first element to have the correct new size. */
  first_element->resize(new_size_first);
  /* return the second chunk */
  return second_element;
}

/* *********************************************************************
   | This output functions dumps all the internal structure of the     |
   | address manager. These includes the main segment list and the     |
   | busy and free lists (indeed they are a map and a multimap). For   |
   | each list it prints the list size and an index + summary line for |
   | each segment.                                                     |
   | (02 Feb 2001) Added dump of the size map (entry + count).         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 20 Jan 2001         |
   ********************************************************************* */
std::ostream &Qaddress_manager::output_details(std::ostream &os) const {
  /* Print some header just to catch the reader attention! */
  os << "This is the quantum address manager speaking!";
  /* Get a reference to the main list, then print the number of segments
     and the number of available qubits for the main segment list. The
     number of segments is the main list size. The number of available
     qubits is a constant (it is area_size). */
  os << "\n\tMain list: segments = " << main_list.size()
     << ", qubits = " << get_number_of_qubits();
  /* Iterate through the list and print the index and summary for each
     segment; since these are segments with usage counts, the summary
     should include <beginning, end, size, usage> */
  main_list_type::size_type index_main = 0;
  for (main_list_type::const_iterator a_segment = main_list.begin();
       a_segment != main_list.end(); ++a_segment, ++index_main)
    os << "\n\t  (" << index_main << ")\t" << *a_segment;
  /* Now do the same thing with the busy list. Remember that this list
     is indeed a map, where the value field (i.e. ->second) is a pointer
     to a segment in the main list. */
  os << "\n\tBusy list: size = " << busy_list.size();
  main_list_type::size_type index_busy = 0;
  for (busy_list_type::const_iterator a_segment = busy_list.begin();
       a_segment != busy_list.end(); ++a_segment, ++index_busy)
    os << "\n\t  (" << index_busy << ")\t" << *(a_segment->second);
  /* Last but not least, the free list. Remember that this list is
     indeed a multimap, where the value field (i.e. ->second) is a
     pointer to a segment in the main list. */
  os << "\n\tFree list: size = " << free_list.size();
  main_list_type::size_type index_free = 0;
  for (free_list_type::const_iterator a_segment = free_list.begin();
       a_segment != free_list.end(); ++a_segment, ++index_free)
    os << "\n\t  (" << index_free << ")\t" << *(a_segment->second);
  /* (02 Feb 2001) Dump the content of the size map. Iterate through the
     container (which is in strict ascending order) and print the counter
     for each entry (it is the second field, while the first is the size). */
  os << "\n\tSize map: size = " << size_map.size();
  main_list_type::size_type index_size = 0;
  for (size_map_type::const_iterator a_size = size_map.begin();
       a_size != size_map.end(); ++a_size, ++index_size)
    os << "\n\t  (" << index_size << ")\t" 
       << std::setw(5) << a_size->first << " --> " 
       << std::setw(3) << a_size->second;
  /* return the stream */
  return os;
}

/* *********************************************************************
   | Operator form of the output function for the address manager.     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      18 Jul 2002         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os,
			 const Qaddress_manager &the_manager) {
  return the_manager.output_details(os);
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
