/*
 * Copyright (C) 2003 Klaus Frahm <frahm@irsamc.ups-tlse.fr>
 * Quantware MIPS Center, Laboratoire de Physique Theorique
 * University Paul Sabatier, Toulouse III
 * 118, route de Narbonne, 31062 Toulouse Cedex 4 - FRANCE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public Licens
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-
 *
 */

#ifndef QUANTUM

#define QUANTUM

#include "complex.h"


#define DEFAULT_NBQUBITS 10 
#define MAX_QUBITS 23
#define MAX_PHASES 1000
#define EXP2(x) (1<<(x))
#define STAT_ITER_NUM 2


class qubit_state{
    int nbqubits;  // number of qubits
    int nbstates;  // number of states = 2^nbqubits
    int nbpbits;   // number of qubits used for p values in classical case
    complex *coeffs; // complex coefficients for each state

    // initializes the global variables
    void global_initialize();
    // initializes the pointer
    void initialize(int nbq);

 public:
    // puts all coeffs to 0
    void put_to_zero();
    // --> Constructors and Destructor
    // default constructor, all qubits put to 0
    qubit_state(int nbq=DEFAULT_NBQUBITS);
    // constructor, with "coeffs" given by the vector "vec"
    qubit_state(int nbq,const complex *vec);
    // constructor, with coeffs[pos]=1 and coeffs[i]=0 if i!=pos
    qubit_state(int nbq,int pos);
    // copy construcor
    qubit_state(const qubit_state &state);
    // destructor
    ~qubit_state();
    // assignment operator
    qubit_state& operator=(qubit_state &a);

    // --> diverse quantum-gate operations
    // reverse qubit order
    void reverse_state();
    // reverse qubit order
    // restricted for qubits i with:  nmin <= i < nmax 
    void reverse_state(int nmin,int nmax);
    // binary reverse of first and second half of the qubits
    void bin_reverse_state();
    // definition of the A_gate
    void A_gate(int j);
    // definition of the perturbed A_gate
    void A_gate_noise(int j,double eps);
    // definition of the B_gate
    void B_gate(int j,int k,int back_flag);
    // definition of the perturbed B_gate
    void B_gate_noise(int j,int k,int back_flag,double eps);
    // quantum Fourier transform for back_flag=0
    // inverse quantum Fourier transform for back_flag=1
    void QFT(int back_flag);
    // quantum Fourier transform for back_flag=0
    // inverse quantum Fourier transform for back_flag=1
    // version with noise error
    void QFT_noise(int back_flag,double eps);
    // quantum Fourier transform for back_flag=0
    // inverse quantum Fourier transform for back_flag=1
    // version with static error
    void QFT_stat(int back_flag,int n);
    // inverse quantum Fourier transform for back_flag=1
    // quantum Fourier transform, second version
    // version with static error, effective version for sigma-z-error
    void QFT_stat_eff(int back_flag);
    // quantum Fourier transform for back_flag=0
    // inverse quantum Fourier transform for back_flag=1
    // restricted for qubits i with:  nmin <= i < nmax 
    void QFT_restricted(int back_flag,int nmin,int nmax);
    // put coeffs to a complex vector
    void put(complex *a);
    // get coeffs from a complex vector
    void get(complex *a);
    // set value of "nbpbits"
    void put_pbits(int p);

    // controlled not
    void cnot_gate(int j,int k);
    // perturbed controlled not
    void cnot_gate_noise(int j,int k,double eps);
    // uncontrolled phase shift
    void B1_gate(int j,complex e_iphase);
    // uncontrolled perturbed phase shift
    void B1_gate_noise(int j,complex e_iphase,double eps);
    // controlled phase shift
    void B2_gate(int j,int k,complex e_iphase);
    // controlled perturbed phase shift
    void B2_gate_noise(int j,int k,complex e_iphase,double eps);
    // double controlled phase shift
    void B3_gate(int j,int k,int l,complex e_iphase);
    // double controlled perturbed phase shift with noise error
    void B3_gate_noise(int j,int k,int l,complex e_iphase,double eps);
    // double controlled perturbed phase shift with static error
    void B3_gate_stat(int j,int k,int l,complex e_iphase,int n);
    // double controlled phase shift, cheating version
    void B3_gate_cheat(int j,int k,int m,complex e_iphase);
    // rotation with exp(i alpha sigma_j^z) 
    void sigma_z_rot(int j,complex e_iphase);
    // rotation with exp(i alpha sigma_j^x sigma_k^x) 
    void sigma_x_rot(int j,int k,complex e_iphase);
    // calculate the phases associate to a sigma_z-rotation
    complex* calc_phases(int count);
    // draw random couplings for the static error
    void draw_couplings(double delta,double J,double t,int ww=1);
    // rotation with all: exp(-i t delta_j sigma_j^z) 
    void all_sigma_z_rot(double t);
    // fast rotation with all: exp(-i t delta_j sigma_j^z) 
    void all_sigma_z_rot_fast(double t);
    // rotation with all: exp(-i t J_{jk} sigma_j^x sigma_k^x) 
    void all_sigma_x_rot(double t);
    // complete rotation for the static error 
    // t=time variable and n=number of decompostions for Trotter formula
    void stat_error(double t,int n);
    // complete rotation for the static error 
    // t=time variable and n=number of decompostions for Trotter formula
    // simple version
    void stat_error_simple(double t,int n);
    // complete rotation for the static error 
    // t=time variable 
    // simple version for only z-rotations
    void stat_error_simple_eff(int count);
    

    // adds a gaussian wave-packet at (p0,q0) 
    // of variance sigma2 to a qubit state 
    void gauss_add(double p0,double q0,double sigma2);
    // Wigner function using gaussian coarse-graining for p 
    void wig_gauss(void);
    // Wigner function using hard coarse-graining for p
    void wig_rect(void);
    // Wigner function using hard coarse-graining for p 
    // and using modified QFT-routine
    void wig_rect_p(void);
    // Wigner function using hard coarse-graining for theta 
    // and using modified QFT-routine
    void wig_rect_theta(void);
    // function to print the Wigner function in a file 
    void wig_print(char *file_name);
    // function to print the Wigner function in a file, 2nd version
    void wig_print2(char *file_name,double limit);
    // function to print the Wigner function in a file, 3rd version
    void wig_print3(char *file_name);

    // diverse administrative or other tools
    // multiplication with arbitrary phase-vector
    void phase_mult(complex *phases);
    // effective difference between two qubit-states
    double max_diff(const qubit_state &a);
    // norm of difference of two qubit-states
    double norm_diff(const qubit_state &a);
    // acces to nbstates
    int N(void){ return nbstates; }
    // acces to nbqubits
    int L(void){ return nbqubits; }
    // acces to nbpbits
    int Lp(void){ return nbpbits; }
    // simple output to console
    void print(char *message);
    // file output, textfile-version
    void print_state(char *file_name);
    // file input, textfile-version
    void read_state(char *file_name);
    // file input, textfile-version, 2nd version allowing for 
    // double numbers for L and p
    void read_state2(char *file_name);
    // file output, raw-version
    void save_state(char *file_name);
    // file input, raw-version
    void load_state(char *file_name);
    // converting from and to little Endian format
    void little_endian_convert(void);
    // norm-value
    double norm(void);
    // IPR-value
    double IPR(void);
    // normalization
    void normalize(void);
    // mean-value
    double mean(void);
    // variance
    double variance(void);
    // inverse participation ration
    double ipr(void);
    // effective volume of a state, i.e. number of sites 
    // with |psi(p)| > cut_value/sqrt(N)
    int volume(double cut_value);
    // fidelity
    double fidelity(const qubit_state &a);
};

extern void print(complex);
extern double random_rectangular_old(double w);
// provides a random phase e^{ix} with -eps/2 <= x < eps/2
complex rand_phase(double eps); 


#endif /* !QUANTUM */
