#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <complex>


using namespace std;
typedef complex<double> complexe;

double K,L,hbar;
int nq,nr;
long int N,Nr;
int p_pair,q_pair;
long int p, p_imp, q, q_imp;
int steps;

double delta, J, *deltai, *Ji, dt;
double noise;
int res1,res2;
int erreurs_statiques;
int bruit;
long int portes;



// Fast Fourier transform algorithm on a complex array of size nn (nn must be a power of 2)
// isign=1 for the direct FFT and -1 for the inverse
void FFT(complexe *data, long int nn, int isign)
{
  long int mmax,m,j,istep,i;
  double theta;
  complexe temp,wp,w;

  j=0;
  for (i=0;i<nn;i++)
    {
      if (j>i)
	{
	  temp=data[j];
	  data[j]=data[i];
	  data[i]=temp;
	}

      m=nn >> 1;
      while ((m>=2) && (j>=m))
	{
	  j-=m;
	  m>>=1;
	}
      j+=m;
    }

  mmax=1;
  while (mmax<nn)
    {
      istep=mmax << 1;
      theta=isign*(M_PI/mmax);
      wp=polar(1.0,theta);
      w=1;

      for (m=0;m<mmax;m+=1)
	{
	  for (i=m;i<nn;i+=istep)
	    {
	      j=i+mmax;
	      temp=w*data[j];
	      data[j]=data[i]-temp;
	      data[i]+=temp;
	    }
	  w*=wp;
	}
      mmax=istep;
    }

  for (i=0;i<nn;i++) data[i]/=sqrt((double)nn);
}



// Error-free Hadamard transform on the qubit j of the register n,
// used only in the static error function
void Hadamard(complexe *n, int j)
{
  long int i,masque,complement;
  complexe temp0,temp1;

  // 1 << j is equivalent to initialize masque with only the jth bit set
  masque=(long int)1 << j;
  for (i=0;i<N;i++)
    {
      if (!(i & masque))
	{
	  complement=i+masque;
	  temp0=n[i];
	  temp1=n[complement];
	  n[i]=(temp0+temp1)/sqrt(2.0);
	  n[complement]=(temp0-temp1)/sqrt(2.0);
	}
    }
}



// Computes the effect of static errors
// This function is called at the end of each gate
void Erreur_statique(complexe *n)
{
  int i;
  long int indice,masque0,masque1;
  complexe phase;
  double somme;

  // Hadamard transform to switch to the eigenbase of Xi operators
  // the terms XiXi+1 are then simple phases
  for (i=0;i<nq;i++) Hadamard(n,i);

  for (indice=0;indice<N;indice++) {
    somme=0.0;
    for (i=0;i<nq;i++) {
      masque0=1 << i;
      masque1=1 << ((i+1)%nq);
      if ( (indice & masque0) == (indice & masque1) ) somme+=Ji[i];
      else somme-=Ji[i];
    }
    phase=polar(1.0,-somme*dt);
    n[indice]=n[indice]*phase;
  }

  for (i=0;i<nq;i++) Hadamard(n,i);

  // Application of the second term sum(Zi) of the Hamiltonian
  for (indice=0;indice<N;indice++) {
    somme=0.0;
    for (i=0;i<nq;i++) {
      masque0=1 << i;
      if (indice & masque0) somme-=deltai[i];
      else somme+=deltai[i];
    }
    phase=polar(1.0,-somme*dt);
    n[indice]=n[indice]*phase;
  }
}



// Hadamard transform on the jth qubit of the register n,
// with static errors of amplitude epsilon
void Porte_Ai(complexe *n, int j, double epsilon)
{
  long int i,masque,complement;
  complexe temp0,temp1;
  complexe a11,a12,a21,a22;
  double n0x,n0z,nx,nz,njx,njy,njz;
  double ux,uz,vx,vz,wy;
  double errtheta,errphi,scalaire;

  if (bruit==0) {
    a11=1/sqrt(2.0);
    a12=a11;
    a21=a11;
    a22=-a11;
  }
  else {
    errtheta=M_PI*epsilon*random()/(double)RAND_MAX;

    n0x=1.0/sqrt(2.0);
    n0z=n0x;
    nx=(cos(errtheta)+sin(errtheta))/sqrt(2.0);
    nz=(cos(errtheta)-sin(errtheta))/sqrt(2.0);

    scalaire=n0x*nx+n0z*nz;
    ux=scalaire*n0x;
    uz=scalaire*n0z;
    vx=nx-ux;
    vz=nz-uz;
    wy=-n0x*vz+n0z*vx;

    errphi=2.0*M_PI*random()/(RAND_MAX+1.0);
    njx=ux+cos(errphi)*vx;
    njy=sin(errphi)*wy;
    njz=uz+cos(errphi)*vz;

    a11=complexe(njz,0.0);
    a12=complexe(njx,-njy);
    a21=complexe(njx,njy);
    a22=complexe(-njz,0.0);
  }

  masque=(long int)1 << j;
  for (i=0;i<N;i++)
    {
      if (!(i & masque))
	{
	  complement=i+masque;
	  temp0=n[i];
	  temp1=n[complement];
	  n[i]=a11*temp0+a12*temp1;
	  n[complement]=a21*temp0+a22*temp1;
	}
    }

  if (erreurs_statiques==1) Erreur_statique(n);
  portes++;
}



// Controlled-phase gate for the QFT, on qubits j and k
void Porte_Bjk(complexe *n, int j, int k, int signe, double epsilon)
{
  long int i,masquej,masquek;
  complexe phase;
  double erreur;

  masquej=(long int)1 << j;
  masquek=(long int)1 << k;
  if (bruit==0) {
    phase=polar(1.0,signe*M_PI/((long int)1 << (k-j)));
  }
  else {
    erreur=M_PI*epsilon*(2.0*random()/(double)RAND_MAX-1.0);
    phase=polar(1.0,signe*M_PI/((long int)1 << (k-j)) + erreur);
  }

  for (i=0;i<N;i++)
    if ((i & masquej) && (i & masquek)) n[i]*=phase;

  if (erreurs_statiques==1) Erreur_statique(n);
  portes++;
}



// General controlled-phase gate of phase theta, on qubits j and k
void Control_Phase(complexe *n, int j, int k, double theta, double epsilon)
{
  long int i,masquej,masquek;
  complexe phase;
  double erreur;

  masquej=(long int)1 << j;
  masquek=(long int)1 << k;
  if (bruit==0) {
    phase=polar(1.0,theta);
  }
  else {
    erreur=M_PI*epsilon*(2.0*random()/(double)RAND_MAX-1.0);
    phase=polar(1.0,theta + erreur);
  }

  for (i=0;i<N;i++)
    if ((i & masquej) && (i & masquek)) n[i]*=phase;

  if (erreurs_statiques==1) Erreur_statique(n);
  portes++;
}



// Swap qubits j and k
void Porte_Echange(complexe *n, int j, int k)
{
  long int i,masquej,masquek,complement;
  complexe temp;

  masquej=(long int)1 << j;
  masquek=(long int)1 << k;

  for (i=0;i<N;i++)
    if ((!(i & masquej)) && (i & masquek))
      {
	complement=i-masquek+masquej;
	temp=n[i];
	n[i]=n[complement];
	n[complement]=temp;
      }
}



// Quantum Fourier transform of register n, with static errors of amplitude epsilon
// signe=1 for the direct QFT, -1 for the inverse
void QFT(complexe *n, int signe, double epsilon)
{
  int l,m;

  for (l=nr-1;l>=0;l--)
    {
      for (m=nr-1;m>l;m--) Porte_Bjk(n, l, m, signe, epsilon);
      Porte_Ai(n, l, epsilon);
    }

  for (l=0;l<(nr/2);l++) Porte_Echange(n, l, nr-1-l);
}



// Z rotation of qubit j, with an angle coef
void Rotation_z(complexe *n, int j, double coef, double epsilon)
{
  long int i,masquej;
  complexe phase;
  double erreur;

  masquej=(long int)1 << j;
  if (bruit==0) {
    phase=polar(1.0,coef/2.0);
  }
  else {
    erreur=M_PI*epsilon*(2.0*random()/(double)RAND_MAX-1.0);
    phase=polar(1.0,coef/2.0 + erreur);
  }

  for (i=0;i<N;i++)
    {
      if (i & masquej) n[i]/=phase;
      else n[i]*=phase;
    }

  if (erreurs_statiques==1) Erreur_statique(n);
  portes++;
}



// Control U gate for the slice method
void Control_U(complexe *n, int j, int debut, long int pp, double epsilon)
{
  int k;

  for (k=0;k<=debut;k++)
    {
      Control_Phase(n, j, debut-k, pp*M_PI/pow(2.0,(double)k), epsilon);
    }
}



// Implementation of the Kick operator with the slice method on the register n :
// exp(coef*cos(m*theta)), with m=pp*2^a (pp is odd)
// steps is the chosen number of slices
void Kick(complexe *n, double coef, long int pp, int a, double epsilon)
{
  double alpha;
  int i;
  int control_qubit,debut;

  alpha=-coef/steps;
  control_qubit=nr;
  debut=nr-1-a;

  Porte_Ai(n,control_qubit,epsilon);
  Control_U(n,control_qubit,debut,-pp,epsilon);
  Porte_Ai(n,control_qubit,epsilon);

  for (i=0;i<(steps-1);i++) {
    Rotation_z(n,control_qubit,alpha/2.0,epsilon);
    Porte_Ai(n,control_qubit,epsilon);
    Control_U(n,control_qubit,debut,2*pp,epsilon);
    Porte_Ai(n,control_qubit,epsilon);
    Rotation_z(n,control_qubit,alpha,epsilon);
    Porte_Ai(n,control_qubit,epsilon);
    Control_U(n,control_qubit,debut,-2*pp,epsilon);
    Porte_Ai(n,control_qubit,epsilon);
    Rotation_z(n,control_qubit,alpha/2.0,epsilon);
  }

  Porte_Ai(n,control_qubit,epsilon);
  Control_U(n,control_qubit,debut,pp,epsilon);
  Porte_Ai(n,control_qubit,epsilon);
}



// Implementation of one complete iteration of the kicked Harper map
void Evolution(complexe *n, double epsilon)
{
  QFT(n,1,epsilon);

  Kick(n,K/hbar,q_imp,q_pair,epsilon);

  QFT(n,-1,epsilon);

  Kick(n,L/hbar,p_imp,p_pair,epsilon);
}



int main(void)
{
  complexe *n,*nh;
  long int i,j,iterations;
  long int q0,p0;
  double sigma,sigmai;
  FILE *psi;
  char output_name[100];

  double itoq,itop;
  long int l;
  complexe *husimi;
  double normalisation;
  int m;


  srandom((unsigned int)time(NULL));

  // Number of qubits of the simulated system
  nr=10;
  Nr=1 << nr;
  // Number of qubits of the quantum computer (there is one ancilla needed for the slice algorithm)
  nq=nr+1;
  N=1 << nq;
  // Number of phase space cells (q in position and p in momentum)
  p=8;
  q=8;
  hbar=2.0*M_PI*p*q/((double)Nr);
  itoq=2.0*M_PI*q/((double)Nr);   // q = itoq * i     ( with i in [0,N-1] and q in [0,q*2pi] )
  itop=2.0*M_PI*p/((double)Nr);   // p = itop * i     ( with i in [0,N-1] and p in [0,p*2pi] )
  // Number of iterations of the map
  iterations=100;
  K=0.5;
  L=0.5;

  // Set this variable to 1 to enable static errors
  erreurs_statiques=0;
  // Set this variable to 1 to enable noisy gates
  bruit=0;
  // Strength of static errors
  delta=pow(10.0,-4.0);
  J=delta;
  // Amplitude of noise in the gates
  noise=pow(10.0,-4.0);
  // Number of slices for each kick operator
  steps=40;
  // Time intervall for one slice
  dt=1.0;


  // We separate the numbers of cells q and p in even and odd parts
  // (like q=q_imp*2^q_pair)
  p_pair=0;
  p_imp=p;
  while (!(p_imp & 1))
    {
      p_imp>>=1;
      p_pair++;
    }
  q_pair=0;
  q_imp=q;
  while (!(q_imp & 1))
    {
      q_imp>>=1;
      q_pair++;
    }


  // Initialization of the random couplings for the static errors
  deltai=(double *)malloc(nq*sizeof(double));
  if (deltai==NULL)
    {
      fprintf(stderr,"Memory allocation of deltai failed.\n");
      exit(0);
    }

  Ji=(double *)malloc(nq*sizeof(double));
  if (Ji==NULL)
    {
      fprintf(stderr,"Memory allocation of Ji failed.\n");
      exit(0);
    }

  for (m=0;m<nq;m++)
    {
      deltai[m]=(random()/(double)RAND_MAX-0.5)*delta;
      Ji[m]=(2.0*random()/(double)RAND_MAX-1.0)*J;
    }

  // Initialization of the quantum register
  n=(complexe *)malloc(N*sizeof(complexe));
  if (n==NULL)
    {
      fprintf(stderr,"Memory allocation of n failed.\n");
      exit(1);
    }

  // Initialization of a buffer used in the intial state preparation
  nh=(complexe *)malloc(Nr*sizeof(complexe));
  if (nh==NULL)
    {
      fprintf(stderr,"Memory allocation of nh failed.\n");
      exit(1);
    }

  // The initial state chosen here is a gaussian wave packet
  // centered on (q0,p0) with a width of sigma
  q0=Nr/2;
  p0=Nr/2+Nr/(2*p);
  sigma=sqrt(hbar/2.0);
  sigmai=sigma/itop;

  if (p0<Nr/2) {
    for (i=0;i<(Nr/2+p0);i++) nh[i]=polar(1.0,-itoq*q0*itop*i/hbar)*exp(-pow(itop*(i-p0),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai);
    for (i=(Nr/2+p0);i<Nr;i++) nh[i]=polar(1.0,-itoq*q0*itop*i/hbar)*exp(-pow(itop*(i-(p0+Nr)),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai);
  }
  else {
    for (i=0;i<(p0-Nr/2);i++) nh[i]=polar(1.0,-itoq*q0*itop*i/hbar)*exp(-pow(itop*(i-(p0-Nr)),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai);
    for (i=(p0-Nr/2);i<Nr;i++) nh[i]=polar(1.0,-itoq*q0*itop*i/hbar)*exp(-pow(itop*(i-p0),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai);
  }

  for (i=0;i<Nr;i++) n[i]=nh[i];
  for (i=Nr;i<N;i++) n[i]=0.0;

  sprintf(output_name,"husimi_function");
  psi=fopen(output_name,"w+");
  if (psi==NULL)
    {
      fprintf(stderr,"Failed to open the file %s.\n",output_name);
      exit(1);
    }

  // Iteration of the map itself
  for (j=0;j<iterations;j++) Evolution(n,noise);

  normalisation=0.0;
  for (i=0;i<Nr;i++) normalisation+=norm(n[i]);
  for (i=0;i<Nr;i++) nh[i]=n[i]/sqrt(normalisation);


  // --------------------------------------
  // Computation of the Husimi distribution
  // --------------------------------------

  husimi=(complexe *)malloc(Nr*sizeof(complexe));
  if (husimi==NULL)
    {
      printf("Memory allocation of husimi failed.\n");
      exit(1);
    }

  // The discrete coherent state is with a good approximation a gaussian of width sigma
  // which extends Nr/2 in both directions from its maximum
  // We then have to consider several cases,
  // depending on whether the maximum is in the first half or in the second half of the Hilbert space
  sigma=sqrt(hbar/2.0);
  for (i=0;i<Nr/2;i++)
    {
      for (l=0;l<(Nr/2+i);l++)
	husimi[l]=exp(-pow(itop*(l-i),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai)*nh[l];
      for (l=(Nr/2+i);l<Nr;l++)
	husimi[l]=exp(-pow(itop*(l-(i+Nr)),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai)*nh[l];
      FFT(husimi,Nr,1);

      for (l=0;l<Nr;l++) fprintf(psi,"%e ",norm(husimi[l]));
      fprintf(psi,"\n");
    }

  for (i=Nr/2;i<Nr;i++)
    {
      for (l=0;l<(i-Nr/2);l++)
	husimi[l]=exp(-pow(itop*(l-(i-Nr)),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai)*nh[l];
      for (l=(i-Nr/2);l<Nr;l++)
	husimi[l]=exp(-pow(itop*(l-i),2.0)/(4.0*pow(sigma,2.0)))/sqrt(sqrt(2.0*M_PI)*sigmai)*nh[l];
      FFT(husimi,Nr,1);

      for (l=0;l<Nr;l++) fprintf(psi,"%e ",norm(husimi[l]));
      fprintf(psi,"\n");
    }

  free(husimi);
  fclose(psi);
  free(n);
  free(nh);
  free(deltai);
  free(Ji);

  return 0;
}
