/*
 * Copyright (C) 2003 Klaus Frahm <frahm@irsamc.ups-tlse.fr>
 * Quantware MIPS Center, Laboratoire de Physique Theorique
 * University Paul Sabatier, Toulouse III
 * 118, route de Narbonne, 31062 Toulouse Cedex 4 - FRANCE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public Licens
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-
 *
 */

#include "quantum.h"
#include "random.h"
#include <math.h>

// #include "complex.cc"



int quant_init_flag=0;
complex *e_itheta;


// initializes the global variables
void qubit_state::global_initialize(){
    int i;
    double phase;

    //    printf("MAIN INITIALIZATION !!!\n");
    quant_init_flag=1;
    e_itheta=new complex[MAX_QUBITS+10];
    
    e_itheta[0]=-1.0;
    e_itheta[1]=I;
    e_itheta[2]=complex(M_SQRT1_2,M_SQRT1_2);
    phase=M_PI/8.0;
    for(i=3;i<=MAX_QUBITS;i++){
	e_itheta[i]=complex(cos(phase),sin(phase));
	phase/=2.0;
    }
}

// initializes the pointer
void qubit_state::initialize(int nbq){
    if(!quant_init_flag) global_initialize();
    if(nbq<0 || nbq>MAX_QUBITS){
	printf("nbq  =  %d   is a ",nbq);
	fehler("wrong qubit-number !!");
    }
    nbqubits=nbq;
    nbstates=EXP2(nbq);
    nbpbits=(nbq+1)/2; // default value for number p-qubits
    coeffs=new complex[nbstates+5];
}

// puts all coeffs to 0
void qubit_state::put_to_zero(){
    int i;

    for(i=0;i<nbstates;i++) coeffs[i]=0;
}


// default constructor, all qubits put to 0
qubit_state::qubit_state(int nbq){
    initialize(nbq);
    put_to_zero();
    coeffs[0]=1.0;
}


// constructor, with "coeffs" given by the vector "vec"
qubit_state::qubit_state(int nbq,const complex *vec){
    int i;

    initialize(nbq);
    for(i=0;i<nbstates;i++) coeffs[i]=vec[i];
}


// constructor, with coeffs[pos]=1 and coeffs[i]=0 if i!=pos
qubit_state::qubit_state(int nbq,int pos){
    initialize(nbq);
    put_to_zero();
    coeffs[pos]=1.0;
}


// copy construcor
qubit_state::qubit_state(const qubit_state &state){
    int i;

    initialize(state.nbqubits);
    nbpbits=state.nbpbits;
    for(i=0;i<nbstates;i++) coeffs[i]=state.coeffs[i];
}


// destructor
qubit_state::~qubit_state(){
    delete[] coeffs;
}

// assignment operator
qubit_state& qubit_state::operator=(qubit_state &a){
    if(this!=&a){
	// adapt the vector size
	if(nbqubits!=a.nbqubits){
	    delete[] coeffs;
	    initialize(a.nbqubits);
	}
	nbpbits=a.nbpbits;
	for(int i=0;i<nbstates;i++)
	    coeffs[i]=a.coeffs[i];
    }
    return *this;
}
//assignment operator
qubit_state& qubit_state::operator+(qubit_state &a){
    if(this!=&a){
	// adapt the vector size
	if(nbqubits!=a.nbqubits){
	    delete[] coeffs;
	    initialize(a.nbqubits);
	}
	nbpbits=a.nbpbits;
	for(int i=0;i<nbstates;i++)
	    coeffs[i]=coeffs[i]+a.coeffs[i];
    }
    return *this;
}
//assignment operator
void qubit_state::const_mult(complex phi){
	for(int i=0;i<nbstates;i++)
	    coeffs[i]=phi*coeffs[i];
}
// reverse qubit order
void qubit_state::reverse_state(){
    int i,x,j,k;
    complex temp;

    for(i=0;i<nbstates;i++){
	j=0; x=i;
	for(k=0;k<nbqubits;k++){
	    j=(j<<1) | (x&1);
	    x=x>>1;
	}
	if(i<j){
	    temp=coeffs[i];
	    coeffs[i]=coeffs[j];
	    coeffs[j]=temp;
	}
    }
}

// reverse qubit order
// restricted for qubits i with:  nmin <= i < nmax 
void qubit_state::reverse_state(int nmin,int nmax){
    int i,x,j,k,mask;
    complex temp;

    mask=0;
    for(k=nmax;k<nbqubits;k++) mask=(mask<<1) | 1;
    for(k=nmin;k<nmax;k++) mask=(mask<<1);
    //    mask=mask<<(nmax-nmin);
    for(k=0;k<nmin;k++)	mask=(mask<<1) | 1;

    for(i=0;i<nbstates;i++){
	j=0; x=i>>nmin;
	for(k=nmin;k<nmax;k++){
	    j=(j<<1) | (x&1);
	    x=x>>1;
	}
	j=j<<nmin;
	j=j | (mask&i);
	if(i<j){
	    temp=coeffs[i];
	    coeffs[i]=coeffs[j];
	    coeffs[j]=temp;
	}
    }
}

// binary reverse of first and second half of the qubits
void qubit_state::bin_reverse_state(){
    int i,L,N,a,b,j,mask,N2;
    complex temp;
    qubit_state old(*this);

    L=nbqubits-nbpbits; N=EXP2(nbpbits); mask=N-1;
    N2=EXP2(L);
    for(i=0;i<nbstates;i++){
	//	a=i>>nbpbits; b=i&mask;
	a=i/N; b=i%N;
	//	j=(b<<L) | a;
	j=b*N2+a;
	coeffs[j]=old.coeffs[i];
    }
    //    put_pbits(nbqubits-nbpbits);
}

// put coeffs to a complex vector
void qubit_state::put(complex *a){
    int i;

    for(i=0;i<nbstates;i++) a[i]=coeffs[i];
}

// get coeffs from a complex vector
void qubit_state::get(complex *a){
    int i;

    for(i=0;i<nbstates;i++) coeffs[i]=a[i];
}

// set value of "nbpbits"
void qubit_state::put_pbits(int p){
    if(p<0) p=0;
    if(p>nbqubits) p=nbqubits;
    nbpbits=p;
}

// definition of the A_gate
void qubit_state::A_gate(int j){
    complex c0,c1;
    int k1,k2,q1,q2,k_index0,k_index1,j1;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    q1=1<<j;                       // q1 = 2^(number of right bits)
    q2=1<<(nbqubits-j-1);          // q2 = 2^(number of left bits)
    j1=j+1;
    for(k1=0;k1<q1;k1++) for(k2=0;k2<q2;k2++){
        k_index0=(k2<<j1) | k1;
        k_index1=k_index0 | q1;
        c0=coeffs[k_index0];
        c1=coeffs[k_index1];
        coeffs[k_index0]=M_SQRT1_2*(c0+c1);
        coeffs[k_index1]=M_SQRT1_2*(c0-c1);
    }
}

// definition of the perturbed A_gate
void qubit_state::A_gate_noise(int j,double eps){
    complex c0,c1;
    double nx,ny,nz,eps_y,eps_z,eps_r,e2,a11,a22;
    complex a12,a21;
    int k1,k2,q1,q2,k_index0,k_index1,j1;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    q1=1<<j;                       // q1 = 2^(number of right bits)
    q2=1<<(nbqubits-j-1);          // q2 = 2^(number of left bits)
    j1=j+1;

    // random vector n
    eps_r=eps*eps/4.0;
    do{
//	eps_y=random_rectangular_old(2*eps);
//	eps_z=random_rectangular_old(2*eps);
	eps_y=random_rectangular(M_PI*eps);
	eps_z=random_rectangular(M_PI*eps);
	e2=eps_y*eps_y+eps_z*eps_z;
    }while(e2>eps_r);
    nx=sqrt(1.0-e2);
    nz=nx;
    nx=M_SQRT1_2*(nx+eps_z);
    nz=M_SQRT1_2*(nz-eps_z);
    ny=eps_y;

    // rotation matrix
    a11=nz; a22=-nz;
    a12=complex(nx,-ny);
    a21=complex(nx,ny);

    for(k1=0;k1<q1;k1++) for(k2=0;k2<q2;k2++){
        k_index0=(k2<<j1) | k1;
        k_index1=k_index0 | q1;
        c0=coeffs[k_index0];
        c1=coeffs[k_index1];
        coeffs[k_index0]=a11*c0+a12*c1;
        coeffs[k_index1]=a21*c0+a22*c1;
    }
}
//-----------
void qubit_state::A1_gate_noise(int j,double eps,double eps2){
    complex c0,c1;
    double nx,ny,nz,eps_y,eps_z,eps_r,e2,a11,a22;
    complex a12,a21;
    int k1,k2,q1,q2,k_index0,k_index1,j1;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    q1=1<<j;                       // q1 = 2^(number of right bits)
    q2=1<<(nbqubits-j-1);          // q2 = 2^(number of left bits)
    j1=j+1;

    // random vector n
    eps_r=eps*eps/4.0;
    do{
	eps_y=eps2; //random_rectangular_old(eps);
	eps_z=eps2; //random_rectangular_old(eps);
	e2=eps_y*eps_y+eps_z*eps_z;
    }while(e2>eps_r);
    nx=sqrt(1.0-e2);
    nz=nx;
    nx=M_SQRT1_2*(nx+eps_z);
    nz=M_SQRT1_2*(nz-eps_z);
    ny=eps_y;

    // rotation matrix
    a11=nz; a22=-nz;
    a12=complex(nx,-ny);
    a21=complex(nx,ny);

    for(k1=0;k1<q1;k1++) for(k2=0;k2<q2;k2++){
        k_index0=(k2<<j1) | k1;
        k_index1=k_index0 | q1;
        c0=coeffs[k_index0];
        c1=coeffs[k_index1];
        coeffs[k_index0]=a11*c0+a12*c1;
        coeffs[k_index1]=a21*c0+a22*c1;
    }
}


// definition of the B_gate 
void qubit_state::B_gate(int j,int k,int back_flag){
    complex ee;
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
    if(j>=k) return;               // direct return if wrong order of j,k
    ee=e_itheta[k-j];              // transformation phase
    if(back_flag){                 // take the complex conjugate if the 
	ee=conj(ee);               //   inverse transformation is used 
    }
    mask=(1<<k) | (1<<j);

    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=ee;
    }
}

// definition of the perturbed B_gate 
void qubit_state::B_gate_noise(int j,int k,int back_flag,double eps){
    complex ee;
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
    if(j>=k) return;               // direct return if wrong order of j,k
    ee=e_itheta[k-j];              // transformation phase
    if(back_flag){                 // take the complex conjugate if the 
	ee=conj(ee);               //   inverse transformation is used 
    }
    ee*=rand_phase(eps);
    mask=(1<<k) | (1<<j);

    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=ee;
    }
}

// inverse quantum Fourier transform for back_flag=1
// quantum Fourier transform, second version
void qubit_state::QFT(int back_flag){
    int j,k;

    for(j=nbqubits-1;j>=0;j--){
	for(k=nbqubits-1;k>j;k--){
	    B_gate(j,k,back_flag);
	}
	A_gate(j);
    }
    reverse_state();
}

// inverse quantum Fourier transform for back_flag=1
// quantum Fourier transform, second version
// version with noise error
void qubit_state::QFT_noise(int back_flag,double eps){
    int j,k;

    for(j=nbqubits-1;j>=0;j--){
	for(k=nbqubits-1;k>j;k--){
	    B_gate_noise(j,k,back_flag,eps);
	}
	A_gate_noise(j,eps);
    }
    reverse_state();
}

// inverse quantum Fourier transform for back_flag=1
// quantum Fourier transform, second version
// version with static error
void qubit_state::QFT_stat(int back_flag,int n){
    int j,k;

    for(j=nbqubits-1;j>=0;j--){
	for(k=nbqubits-1;k>j;k--){
	    B_gate(j,k,back_flag);
	    stat_error_simple(1.0,n);
	}
	A_gate(j);
	stat_error_simple(1.0,n);
    }
    reverse_state();
}

// inverse quantum Fourier transform for back_flag=1
// quantum Fourier transform, second version
// version with static error, effective version for sigma-z-error
void qubit_state::QFT_stat_eff(int back_flag){
  int j,k,err_count;
  
  err_count=0;
  for(j=nbqubits-1;j>=0;j--){
    for(k=nbqubits-1;k>j;k--){
      B_gate(j,k,back_flag);
      //	    stat_error_simple(1.0,n);
      err_count++;
    }
    // doing all error-multiplication
    stat_error_simple_eff(err_count);

    A_gate(j);
    //    stat_error_simple(1.0,n);
    err_count=1;
  }
  stat_error_simple_eff(err_count);
  //  stat_error_simple(1.0,1);
  reverse_state();
}

// quantum Fourier transform for back_flag=0
// inverse quantum Fourier transform for back_flag=1
// restricted for qubits i with:  nmin <= i < nmax 
void qubit_state::QFT_restricted(int back_flag,int nmin,int nmax){
    int j,k;

    for(j=nmax-1;j>=nmin;j--){
	for(k=nmax-1;k>j;k--){
	    B_gate(j,k,back_flag);
	}
	A_gate(j);
    }
    reverse_state(nmin,nmax);
}

// controlled not
// j=target, k=control
void qubit_state::cnot_gate(int j,int k){
    int l1,l0,mask1,mask2;
    complex tt;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
    if(k==j) return; // direct return if k not different from j
    mask1=(1<<k) | (1<<j);
    mask2=~(1<<j);

    for(l1=0;l1<nbstates;l1++){
	if((l1&mask1)==mask1){
	    l0=l1&mask2;
	    tt=coeffs[l0];
	    coeffs[l0]=coeffs[l1];
	    coeffs[l1]=tt;
	}
    }

}

// perturbed controlled not
void qubit_state::cnot_gate_noise(int j,int k,double eps){
    int l1,l0,mask1,mask2;
    double nx,ny,nz,eps_y,eps_z,eps_r,e2,a11,a22;
    complex c0,c1,a12,a21;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
    if(k==j) return; // direct return if k not different from j
    mask1=(1<<k) | (1<<j);
    mask2=~(1<<j);

    // random vector n
    eps_r=eps*eps/4.0;
    do{
	eps_y=random_rectangular_old(eps);
	eps_z=random_rectangular_old(eps);
	e2=eps_y*eps_y+eps_z*eps_z;
    }while(e2>eps_r);
    nx=sqrt(1.0-e2);
    ny=eps_y;
    nz=eps_z;

    // rotation matrix
    a11=nz; a22=-nz;
    a12=complex(nx,-ny);
    a21=complex(nx,ny);
    
    for(l1=0;l1<nbstates;l1++){
	if((l1&mask1)==mask1){
	    l0=l1&mask2;
	    c0=a11*coeffs[l0]+a12*coeffs[l1];
	    c1=a21*coeffs[l0]+a22*coeffs[l1];
	    coeffs[l0]=c0;
	    coeffs[l1]=c1;
	}
    }

}

// uncontrolled phase shift
//      [1         0      ]
//	[0    exp(i phase)]
void qubit_state::B1_gate(int j,complex e_iphase){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<j);

    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
    }
}

// uncontrolled phase shift
//      [exp(-i phase)         0      ]
//	[0    		  exp(i phase)]
void qubit_state::BB1_gate(int j,complex e_iphase){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<j);

    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
           else
			   coeffs[l]*=conj(e_iphase);
    }
}

// uncontrolled perturbed phase shift
void qubit_state::B1_gate_noise(int j,complex e_iphase,double eps){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<j);

    e_iphase*=rand_phase(eps);
    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
    }
}
// uncontrolled perturbed phase shift
void qubit_state::BB1_gate_noise(int j,complex e_iphase,double eps){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<j);

    e_iphase*=rand_phase(eps);
    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
	   else 
			   coeffs[l]*=conj(e_iphase);
    }
}


// controlled phase shift
void qubit_state::B2_gate(int j,int k,complex e_iphase){
  int l,mask;
  //  double a,b;
  
  if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
  if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
  mask=(1<<k) | (1<<j);
  
  for(l=0;l<nbstates;l++){
    if((l&mask)==mask){
      coeffs[l]*=e_iphase;
    }
  }
}

// controlled phase shift version simetrica
// j=target, k=control
void qubit_state::BB2_gate(int j,int k,complex e_iphase){
  int l,mask,mask2;
  //  double a,b;
  
  if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
  if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
  mask=(1<<k)| (0<<j);
  mask2=(1<<j);
  for(l=0;l<nbstates;l++){
       if((l&mask)==mask) {
	    if((l&mask2)==mask2){
		coeffs[l]*=e_iphase;
            }else{
		coeffs[l]*=conj(e_iphase);
            }
        }
  }
}

// controlled perturbed phase shift
void qubit_state::B2_gate_noise(int j,int k,complex e_iphase,double eps){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<k) | (1<<j);

    e_iphase*=rand_phase(eps);
    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
    }
}

// controlled peruturbed phase shift version simetrica
// j=target, k=control
void qubit_state::BB2_gate_noise(int j,int k,complex e_iphase,double eps){
  int l,mask,mask2;
  //  double a,b;
  
  if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
  if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
  mask=(1<<k)| (0<<j);
  mask2=(1<<j);
//  
 e_iphase*=rand_phase(2*M_PI*eps);
//e_iphase*=rand_phase(eps);
  for(l=0;l<nbstates;l++){
       if((l&mask)==mask) {
	    if((l&mask2)==mask2){
		coeffs[l]*=e_iphase;
            }else{
		coeffs[l]*=conj(e_iphase);
            }
        }
  }
}

// double controlled phase shift, e_iphase2^2 corresponds to e_iphase !!
void qubit_state::B3_gate(int j,int k,int l,complex e_iphase2){
    cnot_gate(k,l);
    B2_gate(j,k,conj(e_iphase2));
    cnot_gate(k,l);
    B2_gate(j,k,e_iphase2);
    B2_gate(j,l,e_iphase2);
}

// double controlled perturbed phase shift with noise error
// e_iphase2^2 corresponds to e_iphase !!
void qubit_state::B3_gate_noise(int j,int k,int l,complex e_iphase2,double eps){
    cnot_gate_noise(k,l,eps);
    B2_gate_noise(j,k,conj(e_iphase2),eps);
    cnot_gate_noise(k,l,eps);
    B2_gate_noise(j,k,e_iphase2,eps);
    B2_gate_noise(j,l,e_iphase2,eps);
}

// double controlled perturbed phase shift with static error
// e_iphase2^2 corresponds to e_iphase !!
void qubit_state::B3_gate_stat(int j,int k,int l,complex e_iphase2,int n){
    cnot_gate(k,l);
    stat_error_simple(1.0,n);
    B2_gate(j,k,conj(e_iphase2));
    stat_error_simple(1.0,n);
    cnot_gate(k,l);
    stat_error_simple(1.0,n);
    B2_gate(j,k,e_iphase2);
    stat_error_simple(1.0,n);
    B2_gate(j,l,e_iphase2);
    stat_error_simple(1.0,n);
}

// double controlled phase shift, cheating version
void qubit_state::B3_gate_cheat(int j,int k,int m,complex e_iphase){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
    if(m<0 || m>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<k) | (1<<j) | (1<<m);

    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
    }
}

// rotation with exp(i alpha sigma_j^z) 
// e_iphase=exp(i alpha)
void qubit_state::sigma_z_rot(int j,complex e_iphase){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<j);

    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
	else coeffs[l]*=conj(e_iphase);
    }
}
// noisy rotation with exp(i alpha sigma_j^z) 
// e_iphase=exp(i alpha)
void qubit_state::sigma_z_rot_noise(int j,complex e_iphase,double eps){
    int l,mask;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    mask=(1<<j);
//e_iphase*=rand_phase(eps);
e_iphase*=rand_phase(2*M_PI*eps);
    for(l=0;l<nbstates;l++){
	if((l&mask)==mask) coeffs[l]*=e_iphase;
	else coeffs[l]*=conj(e_iphase);
    }
}
// rotation with exp(i alpha sigma_j^x sigma_k^x) 
// e_iphase=exp(i alpha)
void qubit_state::sigma_x_rot(int j,int k,complex e_iphase){
    int mask11,mask00,mask01,mask10,l00,l01,l10,l11;
    int i1,i2,n1,n2,jmin;
    double co,si,t0a,t0b,t1a,t1b;
    complex *a0,*a1;

    if(j<0 || j>=nbqubits) return; // direct return if bit index out of range
    if(k<0 || k>=nbqubits) return; // direct return if bit index out of range
    if(k==j) return; // direct return if  j=k
    mask11=(1<<k) | (1<<j);
    mask01=~(1<<j);
    mask10=~(1<<k);
    mask00=mask01&mask10;
    co=real_teil(e_iphase);
    si=imag_teil(e_iphase);

    if(j<k){
      n1=1<<j;
      jmin=j;
    }else{
      n1=1<<k;
      jmin=k;
    }
    n2=nbstates/n1;

    //    for(l11=0;l11<nbstates;l11++){
    for(i2=0;i2<n2;i2++){
      //      	if((l11&mask11)==mask11){
      l11=i2<<jmin;
      if((l11&mask11)==mask11){
	l00=l11&mask00;
	l01=l11&mask01;
	l10=l11&mask10;

	a0=coeffs+l00;
	a1=coeffs+l11;
	for(i1=0;i1<n1;i1++){
	  // 	  t0=co*a0[i1]+(si*a1[i1]).times_ii();
	  t0a=co*a0[i1].real()-si*a1[i1].imag();
	  t0b=co*a0[i1].imag()+si*a1[i1].real();
	  //	  t1=(si*a0[i1]).times_ii()+co*a1[i1];
	  t1a=co*a1[i1].real()-si*a0[i1].imag();
	  t1b=co*a1[i1].imag()+si*a0[i1].real();
	  //	  a0[i1]=t0;
	  //	  a1[i1]=t1;
	  a0[i1]=complex(t0a,t0b);
	  a1[i1]=complex(t1a,t1b);
	}
	a0=coeffs+l10;
	a1=coeffs+l01;
	for(i1=0;i1<n1;i1++){
	  // 	  t0=co*a0[i1]+(si*a1[i1]).times_ii();
	  t0a=co*a0[i1].real()-si*a1[i1].imag();
	  t0b=co*a0[i1].imag()+si*a1[i1].real();
	  //	  t1=(si*a0[i1]).times_ii()+co*a1[i1];
	  t1a=co*a1[i1].real()-si*a0[i1].imag();
	  t1b=co*a1[i1].imag()+si*a0[i1].real();
	  //	  a0[i1]=t0;
	  //	  a1[i1]=t1;
	  a0[i1]=complex(t0a,t0b);
	  a1[i1]=complex(t1a,t1b);
	}
      }
    }
}

// global variables for the random couplings;
double _delta_vals[MAX_QUBITS+3],_J_vals[MAX_QUBITS+3][MAX_QUBITS+3];
double _J_cos[MAX_QUBITS+3][MAX_QUBITS+3],_J_sin[MAX_QUBITS+3][MAX_QUBITS+3];
complex **_delta_phases=NULL;

// clean all the phases
void clean_phases(void){
  int i;
  for(i=0;i<MAX_PHASES;i++){
    if(_delta_phases[i]!=NULL) delete[] _delta_phases[i];
  }
  delete[] _delta_phases;
  _delta_phases=NULL;
}

// calculate the phases associate to a sigma_z-rotation
complex* qubit_state::calc_phases(int count){
    int i,j,mask;
    double t;
    complex ee,eec,*phases;

    t=(double)count;
    phases=new complex[nbstates+5];
    for(i=0;i<nbstates;i++) phases[i]=complex(1.0,0.0);
    for(j=0;j<nbqubits;j++){
	mask=(1<<j);
	ee=complex(cos(_delta_vals[j]*t),-sin(_delta_vals[j]*t));
	eec=conj(ee);
	for(i=0;i<nbstates;i++){
	    if((i&mask)==mask) phases[i]*=ee;
	    else phases[i]*=eec;
	}
    }
    return phases;  // return of pointer !!!
}
//*************************************************************
// local help function to determine if (i,j) are neighbor sites
int neighbor(int i,int j,int Lx){
  int ix,iy,jx,jy;

  ix=i/Lx; iy=i%Lx;
  jx=j/Lx; jy=j%Lx;
  if(iy==jy && (ix==jx+1 || ix==jx-1)) return 1;
  if(ix==jx && (iy==jy+1 || iy==jy-1)) return 1;
  return 0;
}
//*************************************************************
// draw random couplings for the static error
void qubit_state::draw_couplings(double delta,double J,double t,int ww){
    int i,j,l,mask,seed_flag;
    double sum2,fak;

    // load the seed for the random numbers if a seed-file exist
    // seed_flag=0 if no seed-file is found
    
seed_flag=load_seed();

  //seed_wert_1=EXP2(ww)+1;
  //seed_wert_2=EXP2(ww+1)+1;

    // factor two for symmetry J_ij=J_ji
    J=J*2.0;

    // drawing of diagonal elements
    sum2=0.0;
    for(i=0;i<nbqubits;i++){
	_delta_vals[i]=random_rectangular_old(1.0);
	sum2+=_delta_vals[i]*_delta_vals[i];
    }
    sum2+=1e-20;
    // normalization of diagonal elements
    fak=sqrt((double)nbqubits)*delta/sqrt(sum2);
    for(i=0;i<nbqubits;i++){
	_delta_vals[i]*=fak;
    }

    // drawing of coupling elements
    sum2=0.0;
    for(i=0;i<nbqubits;i++) for(j=0;j<i;j++){
      if(neighbor(i,j,1)){
	_J_vals[i][j]=random_rectangular_old(1.0);
	sum2+=_J_vals[i][j]*_J_vals[i][j];
      }else{
	_J_vals[i][j]=0;
	_J_cos[i][j]=1.0;
	_J_sin[i][j]=0;
      }
    }
    sum2+=1e-20;
    // normalization of coupling elements
    // the factor (nbquits-1) is intentional to ensure the 
    // same value of tr(dH^2) independant of the number of 
    // non-zero elements
    fak=sqrt((double)(nbqubits-1))*J/sqrt(sum2);
    for(i=0;i<nbqubits;i++) for(j=0;j<i;j++){
	if(_J_vals[i][j]!=0){
	  _J_vals[i][j]*=fak;
	  _J_cos[i][j]=cos(_J_vals[i][j]);
	  _J_sin[i][j]=sin(_J_vals[i][j]);
	}
    }

    
    if(J==0.0){
      for(i=0;i<nbqubits;i++) for(j=0;j<i;j++){
	_J_vals[i][j]=0.0;
	_J_cos[i][j]=1;
	_J_sin[i][j]=0;
      }
    }
    // special marker if all _J_vals[i][j]==0
    _J_vals[0][0]=J;

    if(delta==0.0){
      for(i=0;i<nbqubits;i++){
	_delta_vals[i]=0.0;
      }
    }

    if(_delta_phases!=NULL) clean_phases();
    _delta_phases=new complex* [MAX_PHASES];
    for(i=0;i<MAX_PHASES;i++){
      _delta_phases[i]=NULL;
    }
    _delta_phases[0]=calc_phases(1);
    
    // save seed to seed-file if seed-file exists
    if(seed_flag) save_seed();
}
//*******************************************************************
// rotation with all: exp(-i t delta_j sigma_j^z) 
void qubit_state::all_sigma_z_rot(double t){
    int i;
    complex phase;

    if(_delta_vals[0]==0.0) return;
    for(i=0;i<nbqubits;i++){
	phase=complex(cos(t*_delta_vals[i]),-sin(t*_delta_vals[i]));
	sigma_z_rot(i,phase);
    }
}
//*******************************************************************
// fast rotation with all: exp(-i t delta_j sigma_j^z) 
void qubit_state::all_sigma_z_rot_fast(double t){
    if(_delta_vals[0]==0.0) return;
    phase_mult(_delta_phases[0]);
}

// rotation with all: exp(-i t J_{jk} sigma_j^x sigma_k^x) 
void qubit_state::all_sigma_x_rot(double t){
    int i,j;
    complex phase;
    // verify special marker
    if(_J_vals[0][0]==0.0) return;

    for(i=0;i<nbqubits;i++) for(j=0;j<i;j++){
      if(_J_vals[i][j]!=0.0){
	//	phase=complex(cos(t*_J_vals[i][j]),-sin(t*_J_vals[i][j]));
	// the following line only works for t=1 !!!
	// otherwise one has to use the previous line and comment this line !
	phase=complex(_J_cos[i][j],-_J_sin[i][j]);
	sigma_x_rot(i,j,phase);
      }
    }
    //    for(i=1;i<nbqubits;i++){
    //	phase=complex(cos(t*_J_vals[i][i-1]),-sin(t*_J_vals[i][i-1]));
    //	sigma_x_rot(i,i-1,phase);
    //    }
}

// complete rotation for the static error 
// t=time variable and n=number of decompostions for Trotter formula
void qubit_state::stat_error(double t,int n){
    int i;
    double tn,tn2;
    
    if(n<1) return; // direct return if n<1 
    tn=t/(double)n; tn2=tn/2.0;
    all_sigma_z_rot(tn2);
    all_sigma_x_rot(tn);
    for(i=1;i<n;i++){
	all_sigma_z_rot(tn);
	all_sigma_x_rot(tn);
    }
    all_sigma_z_rot(tn2);
}

// complete rotation for the static error 
// t=time variable and n=number of decompostions for Trotter formula
// simple version
void qubit_state::stat_error_simple(double t,int n){
  //    int i;
  //    double tn;
    
  //    if(n<1) return; // direct return if n<1 
  //    tn=t/(double)n;
  //    for(i=0;i<n;i++){
  //	all_sigma_z_rot(tn);
  all_sigma_z_rot_fast(1.0);
  all_sigma_x_rot(1.0);
  //    }
}

// complete rotation for the static error 
// t=time variable 
// simple version for only z-rotations
void qubit_state::stat_error_simple_eff(int count){
    int i;
    
    if(count<1 || count>MAX_PHASES){
      if(count!=0){
	printf("Range-error for count = %5d\n",count);
      }
      return;
    }
    if(_delta_phases[count-1]==NULL){
      _delta_phases[count-1]=calc_phases(count);
      printf("Created phases for count = %5d\n",count);
    }
    phase_mult(_delta_phases[count-1]);
}

// adds a gaussian wave-packet at (p0,q0) 
// of variance sigma2 to a qubit state 
void qubit_state::gauss_add(double p0,double q0,double sigma2){
    int p;
    double angle,fak,norm,xx,NN,N2;
    complex val;
    
    norm=1.0/sqrt(sqrt(2.0*M_PI*sigma2));
    NN=(double)nbstates;
    N2=NN/2.0;
    for(p=0;p<nbstates;p++){
	xx=(double)p-p0;
	if(xx>N2) xx-=NN;
	if(xx<-N2) xx+=NN;
	xx=xx*xx/4.0/sigma2;
	fak=exp(-xx)*norm;
	angle=-q0*(double)p;
	val=complex(fak*cos(angle),fak*sin(angle));
	coeffs[p]+=val;
    }
}

// Wigner function using gaussian coarse-graining for p 
void qubit_state::wig_gauss(void){
    int i,q,p,p0,m,Nc,N2,N2q,x;
    double sum,*g,a,sigma2;
    qubit_state old(*this);

    // number of extra bits, should be between 0 and 3:
    m=3;
    N2=nbpbits;   N2=EXP2(N2);
    N2q=nbstates>>nbpbits;
    Nc=N2q<<m;
    g=new double[Nc+5];
    qubit_state b(nbqubits-nbpbits+m);

    sigma2=(double)(N2q);
    sigma2=sigma2*sigma2/12.0;
    for(p=0;p<Nc;p++){
	a=(double)p-(double)Nc/2.0;
	g[p]=exp(-0.25*a*a/sigma2);
    }
    sum=0.0;
    for(p=0;p<Nc;p++) sum+=g[p]*g[p];
    sum=sqrt((double)Nc/sum);
    for(p=0;p<Nc;p++) g[p]*=sum;
    for(i=0;i<N2;i++){
	p0=N2q*i+N2q/2-Nc/2;
	for(p=0;p<Nc;p++){
	    x=(p+p0+nbstates)%nbstates;
	    b.coeffs[p]=g[p]*old.coeffs[x];
	}
	b.QFT(0);
	p0=N2q*i;
	for(q=0;q<N2q;q++) coeffs[p0+q]=b.coeffs[q<<m];
    }
    normalize();

    delete[] g;
}

// Wigner function using hard coarse-graining
// this function applies individually the fourier transform 
// to N2=sqrt(N) cells of size N2 !
void qubit_state::wig_rect(void){
    int L2,N2,i;
    complex *keep;

    L2=nbqubits-nbpbits; N2=EXP2(L2);
    qubit_state work(L2);
    // some pointer gymnastics
    // keep the original pointer to ensure later the save destruction
    keep=work.coeffs;
    for(i=0;i<nbstates;i+=N2){
	// temporal assignment of a small piece of size N2 of the 
	// actual class to the qubit_state "work"
	work.coeffs=coeffs+i;
	// fourier transform of the small piece of size N2;
	work.QFT(0);
    }
    // restauration of original pointer to ensure the save destruction
    work.coeffs=keep;
}

// Wigner function using hard coarse-graining for p 
// and using modified QFT-routine
void qubit_state::wig_rect_p(void){
    QFT_restricted(0,0,nbqubits-nbpbits);
}

// Wigner function using hard coarse-graining for theta 
// and using modified QFT-routine
void qubit_state::wig_rect_theta(void){
    QFT(0);
    QFT_restricted(1,0,nbpbits);
    bin_reverse_state();
}

// function to print the Wigner function in a file 
void qubit_state::wig_print(char *file_name){
    int L2,N2,i,p,t;
    double theta,dtheta,w,pp,dp;
    FILE *fp;

    L2=nbqubits-nbpbits; N2=EXP2(L2);
    dtheta=2.0*M_PI/(double)N2;
    dp=2.0*M_PI/(double)(EXP2(nbpbits));
    fp=fopen(file_name,"w");
    for(i=0;i<nbstates;i++){
	t=i%N2; pp=dp*(double)(i/N2);
	theta=dtheta*(double)t;
	w=absquad(coeffs[i]);
	fprintf(fp,"%20.10lf  %20.10lg  %20.10lg\n",
		theta,pp,w);
    }
    fclose(fp);
}

// function to print the Wigner function in a file, 2nd version
void qubit_state::wig_print2(char *file_name,double limit){
    int L2,N2,i,p,t;
    double theta,dtheta,w,w2,pp,dp;
    FILE *fp;

    L2=nbqubits-nbpbits; N2=EXP2(L2);
    dtheta=2.0*M_PI/(double)N2;
    dp=2.0*M_PI/(double)(EXP2(nbpbits));
    fp=fopen(file_name,"w");
    for(i=0;i<nbstates;i++){
	t=i%N2; p=i/N2;
	theta=dtheta*(double)t;
	pp=dp*(double)p;
	w=absquad(coeffs[i]);
	if(w>limit) fprintf(fp,"%20.10lf  %20.10lf\n",theta,pp);
    }
    fclose(fp);
}

// function to print the Wigner function in a file, 3rd version
void qubit_state::wig_print3(char *file_name){
    int L2,N2,i,p,t;
    double w;
    FILE *fp;

    L2=nbqubits-nbpbits; N2=EXP2(L2);
    fp=fopen(file_name,"w");
    for(i=0;i<nbstates;i++){
	t=i%N2;
	w=absquad(coeffs[i]);
	if(t==0) fprintf(fp,"\n");
	fprintf(fp,"%20.10lg\n",w);
    }
    fclose(fp);
}


// multiplication with an arbitrary phase-vector
void qubit_state::phase_mult(complex *phases){
    int l;
    double a,b;

    for(l=0;l<nbstates;l++){
      //      coeffs[l]*=phases[l];
      a=coeffs[l].real()*phases[l].real()-coeffs[l].imag()*phases[l].imag();
      b=coeffs[l].real()*phases[l].imag()+coeffs[l].imag()*phases[l].real();
      coeffs[l]=complex(a,b);
    }
}
    
// effective difference between two qubit-states
double qubit_state::max_diff(const qubit_state &a){
    double dmax,diff;
    int l;

    if(nbqubits!=a.nbqubits) return 9.999E+99;
    dmax=0.0;
    for(l=0;l<nbstates;l++){
	diff=abs(coeffs[l]-a.coeffs[l]);
	if(diff>dmax) dmax=diff;
    }
    return dmax;
}

// norm of difference of two qubit-states
double qubit_state::norm_diff(const qubit_state &a){
    double sum;
    int i;

    sum=0.0;
    if(nbstates!=a.nbstates) return 9.999E+99;
    for(i=0;i<nbstates;i++){
	sum+=absquad(coeffs[i]-a.coeffs[i]);
    }
    return sqrt(sum);
}

// simple output
void qubit_state::print(char *message){
    int i;

    printf("%s\n----------------\n",message);
    for(i=0;i<nbstates;i++)
	printf("%5d  %16.8lf  +  i * %16.8lf\n",
	       i,coeffs[i].real(),coeffs[i].imag());
    printf("\n");
}

// file output, textfile-version
void qubit_state::print_state(char *file_name){
    FILE *fp;
    int i;

    fp=fopen(file_name,"w");
    fprintf(fp,"%d %d\n",nbqubits,nbpbits);
    for(i=0;i<nbstates;i++){
	fprintf(fp,"%25.17le %25.17le\n",coeffs[i].real(),coeffs[i].imag());
    }
    fclose(fp);
}

// file output, textfile-version
void qubit_state::print_state_real(char *file_name){
    FILE *fp;
    int i;

    fp=fopen(file_name,"w");
  //  fprintf(fp,"%d %d\n",nbqubits,nbpbits);
    for(i=0;i<nbstates;i++){
	fprintf(fp,"%d %25.17le \n",i,coeffs[i].real());
    }
    fclose(fp);
}

// prob. file output, textfile-version
void qubit_state::print_prob(char *file_name){
    FILE *fp;
    int i;

    fp=fopen(file_name,"w");
  //  fprintf(fp,"%d %d\n",nbqubits,nbpbits);
    for(i=0;i<nbstates;i++){
	fprintf(fp,"%d %25.17le \n",i,absquad(coeffs[i]));
    }
    fclose(fp);
}

// overload of previous fucntion...
// prob. suming over last nbr qubits.
// file output, textfile-version
void qubit_state::print_prob(char *file_name,int nbr){
    FILE *fp;
    int i,j,num,dim,dim1;
    double suma;
num=nbqubits-nbr;
dim=EXP2(num);
dim1=EXP2(nbr);
    fp=fopen(file_name,"w");
  //  fprintf(fp,"%d %d\n",nbqubits,nbpbits);
    for(i=0;i<dim;i++){
      suma=0.;
       for(j=0;j<dim1;j++){
	suma+=real_teil(coeffs[i<<nbr|j]*conj(coeffs[i<<nbr|j]));
		}
	fprintf(fp,"%d %25.17le \n",i,suma);
    }
    fclose(fp);
}

// overload of previous fucntion...
// prob. suming over last nbr qubits.
// file output, textfile-version
//also prints epsilon column
void qubit_state::print_prob(char *file_name,int nbr,double eps){
    FILE *fp;
    int i,j,num,dim,dim1;
    double suma;
num=nbqubits-nbr;
dim=EXP2(num);
dim1=EXP2(nbr);
    fp=fopen(file_name,"w");
  //  fprintf(fp,"%d %d\n",nbqubits,nbpbits);
    for(i=0;i<dim;i++){
      suma=0.;
       for(j=0;j<dim1;j++){
	suma+=real_teil(coeffs[i<<nbr|j]*conj(coeffs[i<<nbr|j]));
		}
	fprintf(fp,"%9.7le %d %25.17le \n",eps,i,suma);
    }
    fclose(fp);
}
//
// couting peaks after applying shor gate seqcuence
// (Hadamard+Modular exp+FT)
int qubit_state::peak_count(int nbr){
    int i,j,count;
    double suma;
    double *buff;
    if(nbr>nbqubits)
	fehler("2nd register is too big!!!");
    int nz=EXP2(nbqubits-nbr);
    int nr=EXP2(nbr);
    buff= new double[nz + 5];
    for(i=0;i<nz;i++){
      suma=0.;
       for(j=0;j<nr;j++){
	suma+=real_teil(coeffs[i<<nbr|j]*conj(coeffs[i<<nbr|j]));
		}
	buff[i]=suma;
    }
    
    count=0;
    if(buff[0]>buff[1])count++;
    for(i=1;i<nz-1;i++){
	if(buff[i-1]<buff[i]&&buff[i+1]<buff[i])count++;
      }
    if(buff[nz-2]<buff[nz-1])count++;
   delete[] buff;
  return count;
}

// file input, textfile-version
void qubit_state::read_state(char *file_name){
    FILE *fp;
    int i,L,p;
    double x,y;

    fp=fopen(file_name,"r");
    if(fp==NULL) fehler("Inputfile not found !!");
    fscanf(fp,"%d%d",&L,&p);
    // adapt the vector size
    if(nbqubits!=L){
	delete[] coeffs;
	initialize(L);
    }
    put_pbits(p);
    for(i=0;i<nbstates;i++){
	fscanf(fp,"%lf%lf",&x,&y);
	coeffs[i]=complex(x,y);
    }
    fclose(fp);
}

// file input, textfile-version, 2nd version allowing for 
// double numbers for L and p
void qubit_state::read_state2(char *file_name){
    FILE *fp;
    int i,L,p;
    double x,y,LL,pp;

    fp=fopen(file_name,"r");
    if(fp==NULL) fehler("Inputfile not found !!");
    //    fscanf(fp,"%d%d",&L,&p);
    fscanf(fp,"%lf%lf",&LL,&pp);
    L=(int)(LL+0.1);
    p=(int)(pp+0.1);
    // adapt the vector size
    if(nbqubits!=L){
	delete[] coeffs;
	initialize(L);
    }
    put_pbits(p);
    for(i=0;i<nbstates;i++){
	fscanf(fp,"%lf%lf",&x,&y);
	coeffs[i]=complex(x,y);
    }
    fclose(fp);
}

// file output, raw-version
void qubit_state::save_state(char *file_name){
    FILE *fp;
    int i;

    fp=fopen(file_name,"w");
    fwrite(coeffs,sizeof(complex),nbstates,fp);
    fclose(fp);
}

// file input, raw-version
void qubit_state::load_state(char *file_name){
    FILE *fp;
    int i;

    fp=fopen(file_name,"r");
    if(fp==NULL) fehler("Inputfile for raw-loading not found !!");
    fread(coeffs,sizeof(complex),nbstates,fp);
    // test if file is in Big-Endian format
    if(isnan(norm())){
      printf("Converting to Little-Endian format.\n");
      little_endian_convert();
    }

    fclose(fp);
}

// converting from and to little Endian format
void qubit_state::little_endian_convert(void){
  char *t,tmp;
  int i,j;

  for(i=0;i<nbstates;i++){
    t=(char*)(&(coeffs[i].real()));
    for(j=0;j<4;j++){
      tmp=t[j];
      t[j]=t[7-j];
      t[7-j]=tmp;
    }
    t=(char*)(&(coeffs[i].imag()));
    for(j=0;j<4;j++){
      tmp=t[j];
      t[j]=t[7-j];
      t[7-j]=tmp;
    }
  }
}


// norm-value
double qubit_state::norm(void){
    double sum;
    int i;
    
    sum=0.0;
    for(i=0;i<nbstates;i++) sum+=absquad(coeffs[i]);
    return sqrt(sum);
}

// IPR-value
double qubit_state::IPR(void){
    double sum,x;
    int i;
    
    sum=0.0;
    for(i=0;i<nbstates;i++){
	x=absquad(coeffs[i]);
	sum+=x*x;
    }
    return 1.0/sum;

}

// normalization
void qubit_state::normalize(void){
    double fak;
    int i;

    fak=1.0/norm();
    for(i=0;i<nbstates;i++) coeffs[i]*=fak;
}

// mean-value
double qubit_state::mean(void){
    double mean;
    int i,N2,x;

    N2=nbstates/2;
    mean=0.0;
    for(i=0;i<nbstates;i++){
      x=i;
      //      if(x>N2) x-=nbstates;
      mean+=((double)x*absquad(coeffs[i]));
    }
    return mean;
}

// variance
double qubit_state::variance(void){
    double vv,mm,xx,L2,L;
    int i;

    mm=mean();
    vv=0.0;
    L=(double)nbstates;
    L2=L/2.0;
    for(i=0;i<nbstates;i++){
	xx=(double)i-mm;
	if(xx<-L2){ xx+=L;}
	if(xx>L2){xx-=L;}
	vv+=(xx*xx*absquad(coeffs[i]));
    }
    return vv;
}

// inverse participation ration
double qubit_state::ipr(void){
  double sum,x;
  int i;

  sum=0.0;
  for(i=0;i<nbstates;i++){
    x=absquad(coeffs[i]);
    sum+=x*x;
  }
  return 1.0/sum;
}
//
// measure nr-th qubit in a quatum state
int qubit_state::measure(int nr){
  int i,j;
  double prob[2];
  double x;
  // direct return with bad value if bit index out of range
  if(nr<0 || nr>=nbqubits) return -1; 
  // calculate the two probabilites
  prob[0]=0.0;
  prob[1]=0.0;
  for(i=0;i<nbstates;i++){
    j=(i>>nr)&1;
    prob[j]+=absquad(coeffs[i]);
  }
  // draw the random qubit value: 
  // "0" with probability prob[0]
  // "1" with probability prob[1]
  if((x=rand_double())<prob[0]) j=0; else j=1;
    // projection, put coeffs[i] to zero if nr-th bit != j
    // for(i=0;i<nbstates;i++){
    //   if(((i>>nr)&1)!=j){
    //     coeffs[i]=0;
    //   }
    // }
    // normalization after projection
    //normalize();
    // return measured qubit value
  return j;
}

// effective volume of a state, i.e. number of sites 
// with |psi(p)| > cut_value/sqrt(N)
int qubit_state::volume(double cut_value){
  int i,number;

  number=0;
  cut_value=cut_value*cut_value/(double)nbstates;
  for(i=0;i<nbstates;i++){
    if(absquad(coeffs[i])>cut_value) number++;
  }
  return number;
}

// fidelity
double qubit_state::fidelity(const qubit_state &a){
    complex sum;
    int i;

    sum=0.0;
    if(nbstates!=a.nbstates) return 0.0;
    for(i=0;i<nbstates;i++){
	sum+=conj(coeffs[i])*a.coeffs[i];
    }
    return absquad(sum);
}

// provides a random phase e^{ix} with -eps/2 <= x < eps/2

complex rand_phase(double eps){
    double x=random_rectangular_old(eps);
    return complex(cos(M_PI*x),sin(M_PI*x));
}
//**********************************************************************
//Controlled Modular multiplicator
//	U_x|s>|y>=|s>|x*y Mod(n)>
void qubit_state::Umod_control(int x, int n, int nbr, int l){
    int z,Y,Ynew,nz,mask,nr;
    complex *buff;	
    if(nbr>nbqubits){
	fehler("wrong qubit-number for register!!");
    }
//also it should be 2^(nbr-1)<n<2^nbr
    nz=EXP2(nbqubits-nbr);
    nr=EXP2(nbr);
    buff=new complex[n+5]; //the dimension eventually might need to be larger...
    mask=1<<l;
    for(z=0;z<nz;z++){
        if((z&mask)==0)continue;
// 	for(Y=0;Y<nr;Y++){
        for(Y=0;Y<n;Y++){
            buff[Y]=coeffs[(z<<nbr)|Y];
	}
//    	  for(Y=0;Y<nr;Y++){
        for(Y=0;Y<n;Y++){
  	    Ynew=( (z<<nbr) | (Y*x)%n );
	    coeffs[Ynew]=buff[Y];
	}
    }    
    delete[] buff;
}  
//close Umod_control
//*************************************************************************
//Modular multiplicator (NO CONTROL)
//	U_x|y>=|x*y Mod(n)>
void qubit_state::Umod(int x,int n){
    int z,Y,Ynew,nz,mask,nr;
    complex *buff;	

//also it should be 2^(nbr-1)<n<2^nbr
    buff=new complex[n+5]; //the dimension eventually might need to be larger...
        for(Y=0;Y<n;Y++){
            buff[Y]=coeffs[Y];
	}
//    	  for(Y=0;Y<nr;Y++){
        for(Y=0;Y<n;Y++){
  	    Ynew=( (Y*x)%n );
	    coeffs[Ynew]=buff[Y];
	}
       
    delete[] buff;
}  
//close Umod
//*************************************************************************
//false control Modular multiplicator 
//	
void qubit_state::Umod_false_control(int x,int n,int l){
    int z,Y,Ynew,nz,mask,nr;
    complex *buff;	
nz=2*nbqubits;
//also it should be 2^(nbr-1)<n<2^nbr
    buff=new complex[n+5]; //the dimension eventually might need to be larger...


if(l==1){
        for(Y=0;Y<n;Y++){
            buff[Y]=coeffs[Y];
	}
//    	  for(Y=0;Y<nr;Y++){
        for(Y=0;Y<n;Y++){
  	    Ynew=( (Y*x)%n );
	    coeffs[Ynew]=buff[Y];
	}
    }   
    delete[] buff;
}  
//close Umod
/*********************************************************************/
//	the following are for errors on the rightmost nbr qubits...
/*********************************************************************/
// draw random couplings for the static error for the rightmost nbr qubits
void qubit_state::draw_nbr_couplings(double delta,double J,double t,int x,int nbr,int ww){
    int i,j,l,mask,seed_flag;
    double sum2,fak;
	int max_random=2147483647;
    // load the seed for the random numbers if a seed-file exist
    // seed_flag=0 if no seed-file is found
   if(x<0)
	seed_flag=load_seed();
   else{
   seed_wert_1=EXP2(3*x)%max_random;
   seed_wert_2=(EXP2(3*x+1)+1)%max_random;
   }
    // factor two for symmetry J_ij=J_ji
    J=J*2.0;

//int ndif=nbqubits-nbr;
    // drawing of diagonal elements
    sum2=0.0;
    for(i=0;i<nbr;i++){
	_delta_vals[i]=random_rectangular_old(1.0);
	sum2+=_delta_vals[i]*_delta_vals[i];
    }
    sum2+=1e-20;
    // normalization of diagonal elements
    fak=sqrt((double)nbr)*delta/sqrt(sum2);
    for(i=0;i<nbr;i++){
	_delta_vals[i]*=fak;
    }

    // drawing of coupling elements
    sum2=0.0;
    for(i=0;i<nbr;i++) for(j=0;j<i;j++){
      if(neighbor(i,j,ww)){
	_J_vals[i][j]=random_rectangular_old(1.0);
	sum2+=_J_vals[i][j]*_J_vals[i][j];
      }else{
	_J_vals[i][j]=0;
	_J_cos[i][j]=1.0;
	_J_sin[i][j]=0;
      }
    }
    sum2+=1e-20;
    // normalization of coupling elements
    // the factor (nbquits-1) is intentional to ensure the 
    // same value of tr(dH^2) independant of the number of 
    // non-zero elements
    fak=sqrt((double)(nbr-1))*J/sqrt(sum2);
    for(i=0;i<nbr;i++) for(j=0;j<i;j++){
	if(_J_vals[i][j]!=0){
	  _J_vals[i][j]*=fak;
	  _J_cos[i][j]=cos(_J_vals[i][j]);
	  _J_sin[i][j]=sin(_J_vals[i][j]);
	}
    }

    
    if(J==0.0){
      for(i=0;i<nbr;i++) for(j=0;j<i;j++){
	_J_vals[i][j]=0.0;
	_J_cos[i][j]=1;
	_J_sin[i][j]=0;
      }
    }
    // special marker if all _J_vals[i][j]==0
    _J_vals[0][0]=J;

    if(delta==0.0){
      for(i=0;i<nbr;i++){
	_delta_vals[i]=0.0;
      }
    }

    if(_delta_phases!=NULL) clean_phases();
    _delta_phases=new complex* [MAX_PHASES];
    for(i=0;i<MAX_PHASES;i++){
      _delta_phases[i]=NULL;
    }
    _delta_phases[0]=calc_phases(1);
    
    // save seed to seed-file if seed-file exists
    if(seed_flag) save_seed();
}
//***********************************************************************//
// rotation with rightmost nbr qubits: exp(-i t delta_j sigma_j^z) 
void qubit_state::some_sigma_z_rot(double t,int nbr){
    int i;
    complex phase;

//comment added 9/8/06    if(_delta_vals[0]==0.0) return;
    for(i=0;i<nbr;i++){
	phase=complex(cos(t*_delta_vals[i]),-sin(t*_delta_vals[i]));
	sigma_z_rot(i,phase);
    }
}

// rotation with all: exp(-i t J_{jk} sigma_j^x sigma_k^x) ***************************
void qubit_state::some_sigma_x_rot(double t,int nbr){
    int i,j;
    complex phase;
    // verify special marker
// comment added 9/8/06    if(_J_vals[0][0]==0.0) return;

    for(i=0;i<nbr;i++) for(j=nbqubits-nbr;j<i;j++){
      if(_J_vals[i][j]!=0.0){
	//	phase=complex(cos(t*_J_vals[i][j]),-sin(t*_J_vals[i][j]));
	// the following line only works for t=1 !!!
	// otherwise one has to use the previous line and comment this line !
	phase=complex(_J_cos[i][j],-_J_sin[i][j]);
	sigma_x_rot(i,j,phase);
      }
    }
}
//***************************************************************************************
// complete rotation for the static error for the rightmost nbr qubits...
// t=time variable and n=number of decompostions for Trotter formula
void qubit_state::some_stat_error(double t,int n,int nbr){
    int i;
    double tn,tn2;
    
    if(n<1) return; // direct return if n<1 
    tn=t/(double)n; tn2=tn/2.0;
    some_sigma_z_rot(tn2,nbr);
    some_sigma_x_rot(tn,nbr);
    for(i=1;i<n;i++){
	some_sigma_z_rot(tn,nbr);
	some_sigma_x_rot(tn,nbr);
    }
    some_sigma_z_rot(tn2,nbr);
}
//*********************************************************
// Pauli gates
//********************************************************
void qubit_state::global_phase(complex phase){
   complex tt;
    for(int l=0;l<nbstates;l++){
	tt=coeffs[l]*phase;
	coeffs[l]=tt;
    }
}
//**************************
void qubit_state::Pauli_z(int m){
   complex phase1;
phase1=complex(0.,-1.);
sigma_z_rot(m,phase1);
global_phase(phase1);
}
//**************************
void qubit_state::Pauli_x(int m){
    int l0,mask2,l1;
    complex tt;
mask2=~(1<<m);
for(l1=0;l1<nbstates;l1++){
	l0=l1&mask2;
	tt=coeffs[l0];
	coeffs[l0]=coeffs[l1];
	coeffs[l1]=tt;
      }
}
//------------------------------
void qubit_state::Pauli_y(int m){
complex phase=complex(0.,1.);
 Pauli_x(m);
 Pauli_z(m);
 global_phase(phase);
}
//--------------

