/* Simulation of the Shor alogrith with the one-qubit-trick */
/* version with loop for different eps-values for ipr */

#include "complex.h"
#include "random.h"
#include "quantum.h"

extern complex *e_itheta;

/* ########### THEORETICAL-PHYSICAL PART ############ */
/***************************************************************/

double **get_matrix(int num,int length){
  int i;
  double **a;

  a=new double*[num+3];
  for(i=0;i<num;i++) a[i]=new double[length+3];
  return a;
}

/***************************************************************/

void delete_matrix(double **a,int num){
  int i;

  for(i=num-1;i>=0;i--) delete[] a[i];
  delete[] a;
}

/***************************************************************/

void draw_parameters(double *par,double par_max,int num){
  int i;
  double sum2,fak;

  if(num<0) return;
  // drawing of parameters
  sum2=0.0;
  for(i=0;i<num;i++){
    par[i]=zufall_recht_alt(1.0);
    sum2+=par[i]*par[i];
  }
  sum2+=1e-20;
  // normalization of parameters
  fak=sqrt((double)num)*par_max/sqrt(sum2);
  for(i=0;i<num;i++){
    par[i]*=fak;
  }
}

/***************************************************************/

void renormalize_parameters(double *par,double par_max,int num){
  int i;
  double sum2,fak;

  if(num<0) return;
  // summing of parameters
  sum2=0.0;
  for(i=0;i<num;i++){
    //    par[i]=zufall_recht_alt(1.0);
    sum2+=par[i]*par[i];
  }
  sum2+=1e-20;
  // normalization of parameters
  fak=sqrt((double)num)*par_max/sqrt(sum2);
  for(i=0;i<num;i++){
    par[i]*=fak;
  }
}

/***************************************************************/

int period(int x,int N){
  int r,f;
  
  if((f=gcd(x,N))>1){
    x/=f; N/=f;
  }
  r=1;
  f=x;
  while(f!=1){
    f=(x*f)%N;
    r++;
  }
  return r;
}

/***************************************************************/

// direct simulation of Shor's Alogorithm without mesuring
// nbpbits is used as value for nq
void simple_shor(qubit_state &a,int x,int N){
  int nq,nl,ntot,Nq,Nl,Ntot;
  int i,xfac,fak;

  nq=a.Lp();
  ntot=a.L();
  nl=ntot-nq;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);

  // verification that x and N are mutally prime
  if((fak=gcd(x,N))>1){
    printf("Warning 2: gcd(x,N) = %d and not 1.\n",fak);
    N=N/fak;
    printf("Replacing: N -> N/gcd(x,N) = %d\n",N);
  }

  // 1. initialization: |\psi> = |0...0> |0...1>
  a.put_to_one(1);

  // initial Hardamargates:
  // |\psi> = 1/\sqrt(N) \sum_a |a> |0...1>
  for(i=0;i<nl;i++){
    a.A_gate(i+nq);
  }

  // controled multiplication:
  // |\psi> = 1/\sqrt(N) \sum_a |a> |x^a mod N>
  xfac=x;
  for(i=0;i<nl;i++){
    a.cond_mult(nq,i,xfac,N);
    xfac=(xfac*xfac)%N;
  }

  // final fourier transform
  a.QFT_restricted(0,nq,ntot);
}

/***************************************************************/

// direct simulation of Shor's Alogorithm without mesuring
// nbpbits is used as value for nq
// modified version for reordering of operations
void simple_shor_modif(qubit_state &a,int x,int N){
  int nq,nl,ntot,Nq,Nl,Ntot;
  int i,*xfac,fak;
  int j,k,nmin,nmax;

  nq=a.Lp();
  ntot=a.L();
  nl=ntot-nq;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);

  // verification that x and N are mutally prime
  if((fak=gcd(x,N))>1){
    printf("Warning 3: gcd(x,N) = %d and not 1.\n",fak);
    N=N/fak;
    printf("Replacing: N -> N/gcd(x,N) = %d\n",N);
  }

  // preparation of x^(2^i) expressions
  xfac=new int[nl+5];
  xfac[0]=x;
  for(i=1;i<nl;i++){
    xfac[i]=(xfac[i-1]*xfac[i-1])%N;
  }

  // 1. initialization: |\psi> = |0...0> |0...1>
  a.put_to_one(1);

  // global loop: start
  for(i=nl-1;i>=0;i--){
    // initial Hardamargates:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |0...1>
    a.A_gate(i+nq);

    // controled multiplication:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |x^a mod N>
    a.cond_mult(nq,i,xfac[i],N);

    // final fourier transform, manual version:
    // a.QFT_restricted(0,nq,ntot);
    j=i+nq;
    for(k=ntot-1;k>j;k--){
      a.B_gate(j,k,0);
    }
    a.A_gate(j);

  }   // <--- global loop: end

  a.reverse_state(nq,ntot);


  delete[] xfac;
}

/***************************************************************/

// direct simulation of Shor's Alogorithm without mesuring
// nbpbits is used as value for nq
// modified version for reordering of operations
// version with errors
void simple_shor_modif_err(qubit_state &a,int x,int N,double **delta,double **J){
  int nq,nl,ntot,Nq,Nl,Ntot;
  int i,*xfac,fak;
  int j,k,nmin,nmax;

  nq=a.Lp();
  ntot=a.L();
  nl=ntot-nq;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);

  // verification that x and N are mutally prime
  if((fak=gcd(x,N))>1){
    printf("Warning 4: gcd(x,N) = %d and not 1.\n",fak);
    N=N/fak;
    printf("Replacing: N -> N/gcd(x,N) = %d\n",N);
  }

  // preparation of x^(2^i) expressions
  xfac=new int[nl+5];
  xfac[0]=x;
  for(i=1;i<nl;i++){
    xfac[i]=(xfac[i-1]*xfac[i-1])%N;
  }

  // 1. initialization: |\psi> = |0...0> |0...1>
  a.put_to_one(1);

  // global loop: start
  for(i=nl-1;i>=0;i--){
    // initial Hardamargates:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |0...1>
    a.A_gate(i+nq);

    // controled multiplication:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |x^a mod N>
    a.cond_mult(nq,i,xfac[i],N);
    // application of error operator
    a.stat_error_restricted(0,nq,delta[i],J[i]);

    // final fourier transform, manual version:
    // a.QFT_restricted(0,nq,ntot);
    j=i+nq;
    for(k=ntot-1;k>j;k--){
      a.B_gate(j,k,0);
    }
    a.A_gate(j);

  }   // <--- global loop: end

  a.reverse_state(nq,ntot);


  delete[] xfac;
}

/***************************************************************/

// one-qubit simulation of Shor's Alogorithm with mesuring

long one_qubit_shor(qubit_state &a,int x,int N,int nl){
  int nq,ntot,Nq,Nl,Ntot;
  int i,*xfac,fak,*alpha;
  int j,k,nmin,nmax;
  int asum;

  nq=a.L()-1;
  ntot=nq+nl;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);

  // verification that x and N are mutally prime
  if((fak=gcd(x,N))>1){\
    printf("Warning 5: gcd(x,N) = %d and not 1.\n",fak);
    N=N/fak;
    printf("Replacing: N -> N/gcd(x,N) = %d\n",N);
  }

  // preparation of x^(2^i) expressions
  xfac=new int[nl+5];
  alpha=new int[nl+5];
  xfac[0]=x;
  alpha[0]=0;
  // important: here we need that alpha[nl]=0 for later use 
  // in the algorithm
  for(i=1;i<=nl;i++){
    xfac[i]=(xfac[i-1]*xfac[i-1])%N;
    alpha[i]=0;
  }

  // 1. initialization: |\psi> = |0> |0...1>
  a.put_to_one(1);

  // global loop: start
  for(i=nl-1;i>=0;i--){
    // initial Hardamargates:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |0...1>
    //    if(alpha[i+1]) a.Abis_gate(nq); else a.A_gate(nq);
    a.A_gate(nq);
    if(alpha[i+1]) a.B1_gate(nq,e_itheta[0]);

    // controled multiplication:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |x^a mod N>
    a.cond_mult(nq,0,xfac[i],N);

    // final fourier transform, semiclassical version:
    j=i+nq;
    for(k=nl-1;k>i;k--){
      if(alpha[k]){
	a.B1_gate(nq,e_itheta[k-i]);
      }
    }
    a.A_gate(nq);
    // measuring the nq-qubit
    alpha[i]=a.measure(nq);
  }   // <--- global loop: end

  // recalculation of the full a-value, taking into account
  // the reverse bit-swap operation
  // a = \sum_{i=0}^{nl-1} alpha[nl-i-1]*2^i
  asum=0; fak=1;
  for(i=0;i<nl;i++){
    asum+=alpha[nl-1-i]*fak;
    fak*=2;
  }
  
  delete[] alpha;
  delete[] xfac;
  
  return asum;
}

/***************************************************************/

// one-qubit simulation of Shor's Alogorithm with mesuring
// version with errors

long one_qubit_shor_err(qubit_state &a,int x,int N,int nl,double **delta,double **J){
  int nq,ntot,Nq;
  long Nl,Ntot;
  int i,*xfac,*alpha;
  long fak;
  int j,k,nmin,nmax;
  long asum;

  nq=a.L()-1;
  ntot=nq+nl;
  Ntot=a.N();
  if(nl<32){Nl=EXP2(nl);}else{Nl=(long)pow((double)2,(double)nl);}
//  Nl=EXP2(nl);
  Nq=EXP2(nq);
fak=gcd(x,N);
  // verification that x and N are mutally prime
  if((fak=gcd(x,N))>1){
    printf("Warning 1: gcd(x,N) = %d and not 1.\n",fak);
    N=N/fak;
    printf("Replacing: N -> N/gcd(x,N) = %d\n",N);
  }

  // preparation of x^(2^i) expressions
  xfac=new int[nl+5];
  alpha=new int[nl+5];
  xfac[0]=x;
  alpha[0]=0;
  // important: here we need that alpha[nl]=0 for later use 
  // in the algorithm
  for(i=1;i<=nl;i++){
//    xfac[i]=(xfac[i-1]*xfac[i-1])%N;
      xfac[i]=prod_modN(xfac[i-1],xfac[i-1],N);
    alpha[i]=0;
  }

  // 1. initialization: |\psi> = |0> |0...1>
  a.put_to_one(1);

  // global loop: start
  for(i=nl-1;i>=0;i--){
    // initial Hardamargates:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |0...1>
    //    if(alpha[i+1]) a.Abis_gate(nq); else a.A_gate(nq);
    a.A_gate(nq);
    if(alpha[i+1]) a.B1_gate(nq,e_itheta[0]);

    // controled multiplication:
    // |\psi> = 1/\sqrt(N) \sum_a |a> |x^a mod N>
    a.cond_mult(nq,0,xfac[i],N);
    // application of error operator
   // a.stat_error_restricted(0,nq,delta[i],J[i]);
 a.stat_error_restricted(0,nq,delta[0],J[0]);

    // final fourier transform, semiclassical version:
    j=i+nq;
    for(k=nl-1;k>i;k--){
      if(alpha[k]){
	a.B1_gate(nq,e_itheta[k-i]);
      }
    }
    a.A_gate(nq);
    // measuring the nq-qubit
    alpha[i]=a.measure(nq);
  }   // <--- global loop: end

  // recalculation of the full a-value, taking into account
  // the reverse bit-swap operation
  // a = \sum_{i=0}^{nl-1} alpha[nl-i-1]*2^i
  asum=0; fak=1;
  for(i=0;i<nl;i++){
    asum+=alpha[nl-1-i]*fak;
    fak*=2;
  }
  
  delete[] alpha;
  delete[] xfac;
  
  return asum;
}

/***************************************************************/

// theoretical probability

double prob_theoretical(int c,int r,int Nl){
  double y,y2,A,A_sum;
  int Mk,k,i,cr,mkcr;

  cr=(c*r)%Nl;
  if(cr+cr>Nl) cr=Nl-cr;
  mkcr=(Mk*cr)%Nl;
  if(mkcr+mkcr>Nl) mkcr=Nl-mkcr;
  y=(M_PI*cr)/(double)Nl;
  y2=(M_PI*mkcr)/(double)Nl;
  A_sum=0;
  for(k=0;k<r;k++){
    Mk=(int)(((double)(Nl-k-1))/(double)r)+1;
    if(cr==0){
      A=(double)Mk/(double)Nl;
    }else{
      mkcr=(Mk*cr)%Nl;
      if(mkcr+mkcr>Nl) mkcr=Nl-mkcr;
      y2=(M_PI*mkcr)/(double)Nl;
      A=sin(y2)/sin(y)/(double)Nl;
    }
    A_sum+=(A*A);
  }
  return A_sum;
}

/* ########### FOLDING PART ############ */
/***************************************************************/
// refolding transformation function 
// to one peak at zero and shift of N/(2r)
// N es en realidad Nl
int refolding_trans3(int i,int r,long N){
  long N2,i1,j;
 int i2;
  double Ri2;
  N2=(N/r+1)/2;
  i1=i+N2;
  j=(i1*r)/N;
  i2=(2*i1*r-2*j*N+r)/(2*r);
  return i2;
}


long refolding_trans1(long i,int r,long N){ //esta anda bien
  long N2,i1,j;
 long i2;
  double Ri2;
  N2=(N/r+1)/2;
  i1=i+N2;
  j=(i1*r)/N;
  i2=((long)2*i1*r-(long)2*j*N+r)/(2*r);
printf("i %ld\t N2 %ld \t i1 %ld \t j %ld \t i2 %ld\n",i,N2,i1,j,i2);
  return i2;
}

long refolding_trans(long i,int r,long N){
  long N2,i1,j;
 long i2;
  double w;
  N2=(N/r+1)/2;
   w=(double)N/((double)r);
  i1=i+N2;
  //j=(i1*r)/N;
  j=(long)((double)i1/w);
   i2=(long)i1-(long)(j*w+0.5);
//printf("i %ld\t N2 %ld \t i1 %ld \t j %ld \t i2 %ld\n",i,N2,i1,j,i2);
  return i2;
}
/***************************************************************/
// refolding of a histogram from r peaks at distance N/r
// to one peak at zero and shift of N/(2r)
// integer version for histogramm values

void refolding(int r,int &N,int **hist){
  int *buff,*orig,Nnew,i,i2;

  Nnew=N/r+1;
  buff=new int[Nnew+10];
  orig=(*hist);
  for(i=0;i<Nnew;i++) buff[i]=0;
  for(i=0;i<N;i++){
    i2=refolding_trans(i,r,N);
    buff[i2]+=orig[i];
  }
  delete[] orig;
  (*hist)=buff; // pointer copy !!!
  N=Nnew;
}

/***************************************************************/
// refolding of a histogram from r peaks at distance N/r
// to one peak at zero and shift of N/(2r)
// double version for histogramm values

void refolding(int r,int &N,double **hist){
  double *buff,*orig;
  int Nnew,i,i2;

  Nnew=N/r+1;
  buff=new double[Nnew+10];
  orig=(*hist);
  for(i=0;i<Nnew;i++) buff[i]=0;
  for(i=0;i<N;i++){
    i2=refolding_trans(i,r,N);
    buff[i2]+=orig[i];
  }
  delete[] orig;
  (*hist)=buff; // pointer copy !!!
  N=Nnew;
}

/***************************************************************/

double prob_theoretical_fold(int i,int r,int Nl){
  int c;
  double sum;

  sum=0.0;
  for(c=0;c<Nl;c++){
    if(i==refolding_trans(c,r,Nl)){
      sum+=prob_theoretical(c,r,Nl);
    }
  }
  return sum;
}

/* ########### PRINTING PART ############ */
/***************************************************************/

// printing function 
void print_shor(qubit_state &a,int x,int N,char *filename){
  int nq,nl,ntot,Nq,Nl,Ntot;
  int i,xfac,fak,r,c,Mk,k;
  complex *buff;
  double y,A,B;
  FILE *fp;
  

  nq=a.Lp();
  ntot=a.L();
  nl=ntot-nq;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);
  buff=new complex[Ntot+10];
  a.put(buff);

  r=period(x,N);
  k=0;
  Mk=(int)(((double)(Nl-k-1))/(double)r)+1;
  fp=fopen(filename,"w");
  for(i=1;i<Ntot;i+=Nq){
    c=i/Nq;
    y=(M_PI*c*r)/(double)Nl;
    if(y==0){
      A=(double)Mk/(double)Nl;
    }else{
      A=sin(Mk*y)/sin(y)/(double)Nl;
    }
    A=A*A;
    B=absquad(buff[i]);
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",c,B,A,fabs(B-A)/B);
  }
  fclose(fp);

  delete[] buff;
}

/***************************************************************/

// printing function, version with sum for k=0,...,N-1
void print_shor_sum(qubit_state &a,int x,int N,char *filename){
  int nq,nl,ntot,Nq,Nl,Ntot;
  int i,xfac,fak,r,c,k;
  complex *buff;
  double y,A,B,A_sum,B_sum;
  FILE *fp;
  

  nq=a.Lp();
  ntot=a.L();
  nl=ntot-nq;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);
  buff=new complex[Ntot+10];
  a.put(buff);

  r=period(x,N);
  fp=fopen(filename,"w");
  for(i=0;i<Ntot;i+=Nq){
    c=i/Nq;
    A_sum=prob_theoretical(c,r,Nl);
    B_sum=0.0;
    for(k=0;k<Nq;k++) B_sum+=absquad(buff[i+k]);
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",c,B_sum,A_sum,
	    fabs(B_sum-A_sum)/B_sum);
  }
  fclose(fp);

  delete[] buff;
}

/***************************************************************/

// printing function, version with sum for k=0,...,N-1
// folding version

void print_shor_sum_fold(qubit_state &a,int x,int N,char *filename){
  int nq,nl,ntot,Nq,Nl,Ntot,Nnew;
  int i,xfac,fak,r,c,k;
  complex *buff;
  double *prob;
  double y,A,B,A_sum,B_sum;
  FILE *fp;
  

  nq=a.Lp();
  ntot=a.L();
  nl=ntot-nq;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);
  buff=new complex[Ntot+10];
  prob=new double[Nl+10];
  a.put(buff);

  r=period(x,N);
  for(i=0;i<Ntot;i+=Nq){
    c=i/Nq;
    B_sum=0.0;
    for(k=0;k<Nq;k++) B_sum+=absquad(buff[i+k]);
    prob[c]=B_sum;
  }

  Nnew=Nl;
  refolding(r,Nnew,&prob);
  fp=fopen(filename,"w");
  for(i=0;i<Nnew;i++){
    A_sum=prob_theoretical_fold(i,r,Nl);
    B_sum=prob[i];
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",i,B_sum,A_sum,
	    fabs(B_sum-A_sum)/B_sum);
  }
  fclose(fp);

  delete[] prob;
  delete[] buff;
}

/***************************************************************/

// printing function, version with sum for k=0,...,N-1
void shor_ipr(qubit_state &a,int x,int N,char *filename){
  int nq,nl,ntot,Nq,Nl,Ntot;
  int i,xfac,fak,r,c,k;
  complex *buff;
  double y,A,B,A_sum,B_sum,xi;
  FILE *fp;
  

  nq=a.Lp();
  ntot=a.L();
  nl=ntot-nq;
  Ntot=a.N();
  Nl=EXP2(nl);
  Nq=EXP2(nq);
  buff=new complex[Ntot+10];
  a.put(buff);

  r=period(x,N);
  xi=0.0;
  for(i=0;i<Ntot;i+=Nq){
    c=i/Nq;
    B_sum=0.0;
    for(k=0;k<Nq;k++) B_sum+=absquad(buff[i+k]);
    xi+=B_sum*B_sum;
  }
  fp=fopen(filename,"w");
  fprintf(fp,"%20.10lf  %20.10lf\n",xi,1.0/xi);
  fclose(fp);

  delete[] buff;
}

/***************************************************************/

void print_positions(int x,int N,int nl,char *filename){
  int m,r,Nl;
  FILE *fp;
  double c;

  Nl=EXP2(nl);
  r=period(x,N);
  fp=fopen(filename,"w");
  for(m=0;m<r;m++){
    c=(double)Nl/(double)r*(double)m;
    fprintf(fp,"%10.3lf\n",c);
  }
  fclose(fp);
}

/* ########### IPR PART ############ */
/***************************************************************/

double __ipr_global;

int analyze_ipr(FILE* fp,int N,int *hist,double err){
  int i,R;
  double sum,xi,xi2,xiR,xiR2,delta2_xiR,delta_rel,alpha,alpha2,A1,A2,p,p2;

  R=0;
  for(i=0;i<N;i++) R+=hist[i];
  if(R<3) return 0;
  alpha=1.0/(double)R;
  alpha2=alpha*alpha; // = alpha^2
  A1=1.0-alpha;  // = 1-alpha
  A2=(A1-alpha)*A1; // = (1-alpha)*(1-2*alpha)

  xiR=0.0; xiR2=0.0;
  for(i=0;i<N;i++){
    p=(double)hist[i]*alpha;
    p2=p*p;
    xiR+=p2;
    xiR2+=(p*p2);
  }

  xi=(xiR-alpha)/A1;
  xi2=(xiR2-3.0*alpha*xiR+2*alpha2)/A2;

  delta2_xiR=2.0*alpha2*A1*(xi-xi*xi)+4*alpha*A2*(xi2-xi*xi);
  delta_rel=sqrt(delta2_xiR)/xiR;

  // file output only if fp!=NULL
  if(fp!=NULL){
    fprintf(fp,"%10d  %12.4lf  %12.4lf  %12.4lf  %12.6lf  %12.6lf\n",
	    R,1.0/xiR,1.0/xi,delta_rel/xi,delta_rel,(xi2-xi*xi)/xi/xi);
  }
  __ipr_global=1.0/xi;
  return(delta_rel<err && delta_rel!=0.0 && err*(double)R > 1.0);
}

/***************************************************************/

double get_ipr_from_hist(FILE* fp,int N,int *hist,double &rel_err){
  int i,R;
  double sum,xi,xi2,xiR,xiR2,delta2_xiR,delta_rel,alpha,alpha2,A1,A2,p,p2;

  R=0;
  rel_err=1.0;
  for(i=0;i<N;i++) R+=hist[i];
  // direct return with rel_err=1 if R=1
  if(R<=3) return 1.0;
  alpha=1.0/(double)R;
  alpha2=alpha*alpha; // = alpha^2
  A1=1.0-alpha;  // = 1-alpha
  A2=(A1-alpha)*A1; // = (1-alpha)*(1-2*alpha)

  xiR=0.0; xiR2=0.0;
  for(i=0;i<N;i++){
    p=(double)hist[i]*alpha;
    p2=p*p;
    xiR+=p2;
    xiR2+=(p*p2);
  }

  xi=(xiR-alpha)/A1;
  // direct return with rel_err=1 if extrapolated xi does not work because zero
  if(fabs(xi)<1e-14) return 1.0/xiR;
  xi2=(xiR2-3.0*alpha*xiR+2*alpha2)/A2;

  delta2_xiR=2.0*alpha2*A1*(xi-xi*xi)+4*alpha*A2*(xi2-xi*xi);
  delta_rel=sqrt(fabs(delta2_xiR))/xiR;
  rel_err=delta_rel;

  // file output for testing only if fp!=NULL
  if(fp!=NULL){
    fprintf(fp,"%10d  %12.4lf  %12.4lf  %12.4lf  %12.6lf  %12.6lf\n",
	    R,1.0/xiR,1.0/xi,delta_rel/xi,delta_rel,(xi2-xi*xi)/xi/xi);
  }
  return 1.0/xi;
}

/***************************************************************/

void print_ipr(int x,int N,int nq,int ntot,double rel_err,char *filename){
  qubit_state b(nq+1);
  int i,nl,r;
  long Nl;
  long *hist_buff;
  double B,A,err,xi;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  hist_buff=new long[Nl+10];

  r=period(x,N);
  for(i=0;i<Nl;i++) hist_buff[i]=0;
  fp=fopen(filename,"w");
  do{
    hist_buff[one_qubit_shor(b,x,N,nl)]++;
//   xi=get_ipr_from_hist(fp,Nl,hist_buff,err);
    //    printf("%20.10lf  %20.10lf  %20.10lf\n",xi,err,rel_err);
  }
  while(err>rel_err);
  fclose(fp);
  delete[] hist_buff;
}

/***************************************************************/

void print_ipr_err(int x,int N,int nq,int ntot,double rel_err,
			double **delta,double **J,char *filename){
  qubit_state b(nq+1);
  int i,nl,Nnew,r;
  long Nl;
  long *hist_buff;
  double B,A,err,xi;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  r=period(x,N);
  Nnew=Nl/r+1;

  hist_buff=new long[Nl+10];

  for(i=0;i<Nl;i++) hist_buff[i]=0;
  fp=fopen(filename,"w");
  do{
    //    hist_buff[refolding_trans(one_qubit_shor_err(b,x,N,nl,delta,J),r,Nl)]++;
    hist_buff[one_qubit_shor_err(b,x,N,nl,delta,J)]++;
//    xi=get_ipr_from_hist(fp,Nl,hist_buff,err);
    printf("%20.10lf  %20.10lf  %20.10lf\n",xi,err,rel_err);
  }
  while(err>rel_err);
  fclose(fp);
  delete[] hist_buff;
}

/***************************************************************/

void print_ipr_fold_err(int x,int N,int nq,int ntot,double rel_err,
			double **delta,double **J,char *filename){
  qubit_state b(nq+1);
  int i,nl,Nnew,r;
  long Nl;
  long *hist_buff;
  double B,A,err,xi;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  r=period(x,N);
  Nnew=Nl/r+1;

  hist_buff=new long[Nnew+10];

  for(i=0;i<Nnew;i++) hist_buff[i]=0;
  fp=fopen(filename,"w");
  do{
    //    hist_buff[one_qubit_shor(b,x,N,nl)]++;
    hist_buff[refolding_trans(one_qubit_shor_err(b,x,N,nl,delta,J),r,Nl)]++;
//    xi=get_ipr_from_hist(fp,Nnew,hist_buff,err);
    printf("%20.10lf  %20.10lf  %20.10lf\n",xi,err,rel_err);
  }
  while(err>rel_err);
  fclose(fp);
  delete[] hist_buff;
}

/***************************************************************/

void calc_ipr_fold_err(int x,int N,int nq,int ntot,int count,double rel_err,
		       double eps_max,double deps,double ipr_max,
		       char *filename){
  qubit_state b(nq+1);
  int i,nl,r;
  long Nl,Nnew;
  int *hist_buff;
  double B,A,err,xi,eps,ipr_null;
  double **delta,**J;
  FILE *fp;

  nl=ntot-nq; 
  if(nl<32){Nl=EXP2(nl);}else{Nl=(long)pow((double)2,(double)nl);}
  printf("Nl %ld\n",Nl);
  r=period(x,N);
  Nnew=Nl/r+1;
  printf("r %d\t Nnew %ld\n",r,Nnew);
  hist_buff=new int[Nnew+10];
  // create matrices for error parameters
  delta=get_matrix(nl,nq);
  J=get_matrix(nl,nq-1);


  fp=fopen(filename,"w");
  fclose(fp);
  for(eps=0.0;eps<=eps_max+deps/2.0;eps+=deps){
    // draw random parameters
 //  reset_random();  
    for(i=0;i<nl;i++){
      draw_parameters(delta[i],eps,nq);
      // factor two for symmetry J_ij=J_ji
      draw_parameters(J[i],2.0*eps,nq-1);
    }
//    reset_random();//preguntar a Klaus sobre esto 11/06/07
    for(i=0;i<Nnew;i++) hist_buff[i]=0;
    i=0;
    do{
      i++;
      hist_buff[refolding_trans(one_qubit_shor_err(b,x,N,nl,delta,J),r,Nl)]++;
      xi=get_ipr_from_hist(NULL,Nnew,hist_buff,err);
      //      printf("%20.10lf  %20.10lf  %20.10lf\n",xi,err,rel_err);
    }
    while(err>rel_err || i<=count);

    if(eps==0.0){
      ipr_null=xi;
    }
    fprintf(stdout,"%12.6lf  %12.6lf  ",eps,xi/ipr_null);
    xi=get_ipr_from_hist(stdout,Nnew,hist_buff,err);
    fp=fopen(filename,"a");
    fprintf(fp,"%12.6lf  %12.6lf  ",eps,xi/ipr_null);
    xi=get_ipr_from_hist(fp,Nnew,hist_buff,err);
    fclose(fp);
    // stop calculation if maximal value of ipr_max is reached
    if(xi/ipr_null>ipr_max) break;
  }

  // delete matrices for error parameters
  delete_matrix(J,nl);
  delete_matrix(delta,nl);
  delete[] hist_buff;
}

/* ########### HISTROGRAMM PART ############ */
/***************************************************************/

void print_histogramm(int x,int N,int nq,int ntot,int count,char *filename){
  qubit_state b(nq+1);
  int i,nl,Nl,r,*hist_buff,Nnew;
  double B,A;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  r=period(x,N);
  Nnew=Nl/r+1;
  hist_buff=new int[Nl+10];
  //  hist_buff=new int[Nnew+10];

  for(i=0;i<Nl;i++) hist_buff[i]=0;
  for(i=0;i<count;i++){
    hist_buff[one_qubit_shor(b,x,N,nl)]++;
  }
  fp=fopen(filename,"w");
  for(i=0;i<Nl;i++){
    B=(double)hist_buff[i]/(double)count;
    A=prob_theoretical(i,r,Nl);
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",i,B,A,fabs(B-A));
    //    fprintf(fp,"%5d  %20.10le\n",i,B);
  }
  fclose(fp);
  delete[] hist_buff;
}

/***************************************************************/

void print_histogramm_err(int x,int N,int nq,int ntot,int count,
			  double **delta,double **J,char *filename){
  qubit_state b(nq+1);
  int i,nl,Nl,r,*hist_buff,Nnew;
  double B,A;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  r=period(x,N);
  Nnew=Nl/r+1;
  hist_buff=new int[Nl+10];
  //  hist_buff=new int[Nnew+10];

  for(i=0;i<Nl;i++) hist_buff[i]=0;
  for(i=0;i<count;i++){
    hist_buff[one_qubit_shor_err(b,x,N,nl,delta,J)]++;
  }
  fp=fopen(filename,"w");
  for(i=0;i<Nl;i++){
    B=(double)hist_buff[i]/(double)count;
    A=prob_theoretical(i,r,Nl);
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",i,B,A,fabs(B-A));
    //    fprintf(fp,"%5d  %20.10le\n",i,B);
  }
  fclose(fp);


  delete[] hist_buff;
}

/***************************************************************/

void print_histogramm_fold2(int x,int N,int nq,int ntot,int count,char *filename){
  qubit_state b(nq+1);
  int i,nl,Nl,r,*hist_buff,Nnew;
  double B,A;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  r=period(x,N);
  Nnew=Nl/r+1;
  hist_buff=new int[Nl+10];
  //  hist_buff=new int[Nnew+10];

  for(i=0;i<Nl;i++) hist_buff[i]=0;
  for(i=0;i<count;i++){
    hist_buff[one_qubit_shor(b,x,N,nl)]++;
  }
  Nnew=Nl;
  refolding(r,Nnew,&hist_buff);
  fp=fopen(filename,"w");
  for(i=0;i<Nnew;i++){
    B=(double)hist_buff[i]/(double)count;
    A=prob_theoretical_fold(i,r,Nl);
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",i,B,A,fabs(B-A));
    fprintf(fp,"%5d  %20.10le\n",i,B);
  }
  fclose(fp);
  delete[] hist_buff;
}

/***************************************************************/

void print_histogramm_fold(int x,int N,int nq,int ntot,int count,char *filename){
  qubit_state b(nq+1);
  int i,nl,Nl,r,*hist_buff,Nnew;
  double B,A;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  r=period(x,N);
  Nnew=Nl/r+1;
  //  hist_buff=new int[Nl+10];
  hist_buff=new int[Nnew+10];

  for(i=0;i<Nnew;i++) hist_buff[i]=0;
  for(i=0;i<count;i++){
    hist_buff[refolding_trans(one_qubit_shor(b,x,N,nl),r,Nl)]++;
  }
  //  refolding_int(r,Nl,&hist_buff);
  fp=fopen(filename,"w");
  for(i=0;i<Nnew;i++){
    B=(double)hist_buff[i]/(double)count;
    A=prob_theoretical_fold(i,r,Nl);
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",i,B,A,fabs(B-A));
    fprintf(fp,"%5d  %20.10le\n",i,B);
  }
  fclose(fp);
  delete[] hist_buff;
}

/***************************************************************/

void print_histogramm_fold_err(int x,int N,int nq,int ntot,int count,
			       double **delta,double **J,char *filename){
  qubit_state b(nq+1);
  int i,nl,Nl,r,*hist_buff,Nnew;
  double B,A;
  FILE *fp;

  nl=ntot-nq; Nl=EXP2(nl);
  r=period(x,N);
  Nnew=Nl/r+1;
  //  hist_buff=new int[Nl+10];
  hist_buff=new int[Nnew+10];

  for(i=0;i<Nnew;i++) hist_buff[i]=0;
  for(i=0;i<count;i++){
    hist_buff[refolding_trans(one_qubit_shor_err(b,x,N,nl,delta,J),r,Nl)]++;
  }
  //  refolding_int(r,Nl,&hist_buff);
  fp=fopen(filename,"w");
  for(i=0;i<Nnew;i++){
    B=(double)hist_buff[i]/(double)count;
    A=prob_theoretical_fold(i,r,Nl);
    fprintf(fp,"%5d  %20.10le  %20.10le  %20.10le\n",i,B,A,fabs(B-A));
    fprintf(fp,"%5d  %20.10le\n",i,B);
  }
  fclose(fp);
  delete[] hist_buff;
}

/***************************************************************/

int main_try1(int argc,char **argv){
  int x,N,i,Nq,Ntot,asum,count,ntot,nq,nl,fak;
  double **delta,**J;
  double rel_err,eps;

  x=2;
  N=23;
  count=1000;
  rel_err=0.1;
  eps=0.01;
  

  // use above default values if argc=1
  if(argc<6 && argc>1) 
    fehler("syntax: x N count rel_err eps");
  if(argc>=6){
    if(sscanf(argv[1],"%d",&x)==0) 
      fehler("syntax: x N count rel_err eps");
    if(sscanf(argv[2],"%d",&N)==0) 
      fehler("syntax: x N count rel_err eps");
    if(sscanf(argv[3],"%d",&count)==0) 
      fehler("syntax: x N count rel_err eps");
    if(sscanf(argv[4],"%lf",&rel_err)==0) 
      fehler("syntax: x N count rel_err eps");
    if(sscanf(argv[5],"%lf",&eps)==0) 
      fehler("syntax: x N count rel_err eps");
  }

  fak=N;
  nq=0;
  while(fak){
    nq++;
    fak=fak>>1;
  }
  ntot=3*nq;
  if(ntot>32) ntot=31;
  nl=ntot-nq;

  // create matrices for error parameters
  delta=get_matrix(nl,nq);
  J=get_matrix(nl,nq-1);

  // draw random parameters
  reset_random();
  for(i=0;i<nl;i++){
    draw_parameters(delta[i],eps,nq);
    // factor two for symmetry J_ij=J_ji
    draw_parameters(J[i],2.0*eps,nq-1);
  }

  // create qubit state, full size
  qubit_state a(ntot);


  Nq=EXP2(nq);
  Ntot=EXP2(ntot);
  a.put_pbits(nq);
  simple_shor_modif_err(a,x,N,delta,J);
  print_shor_sum(a,x,N,"plot_err.dat");
  print_shor_sum_fold(a,x,N,"plot_fold_err.dat");
  shor_ipr(a,x,N,"ipr_shor.dat");
  /*  reset_random();
      print_histogramm_err(x,N,nq,ntot,count,delta,J,"hist_err.dat");
      reset_random();
      print_histogramm_fold_err(x,N,nq,ntot,count,delta,J,"hist_fold_err.dat");
  */
  //reset_random();
  print_ipr_err(x,N,nq,ntot,rel_err,delta,J,"ipr_err.dat");
  //reset_random();
  print_ipr_fold_err(x,N,nq,ntot,rel_err,delta,J,"ipr_fold_err.dat");
  /*
    print_positions(x,N,ntot-nq,"pos.dat");
    reset_random();
    print_histogramm(x,N,nq,ntot,count,"hist.dat");
    reset_random();
    print_histogramm_fold(x,N,nq,ntot,count,"hist_fold.dat");
    reset_random();
    print_histogramm_fold2(x,N,nq,ntot,count,"hist_fold2.dat");
    reset_random();
    print_ipr(x,N,nq,ntot,rel_err,"ipr.dat");
  */

  // delete matrices for error parameters
  delete_matrix(J,nl);
  delete_matrix(delta,nl);
}

/***************************************************************/

int main(int argc,char **argv){
  int x,N,i,Nq,asum,count,ntot,nq,nl,fak;
  double **delta,**J;
  double rel_err,eps_max,deps,ipr_max;
  char filename[200],syntax[200];

  // default values
  x=2;
  N=23;
  count=1000;
  rel_err=0.04;
  eps_max=0.5;
  deps=0.01;
  ipr_max=100;
  
load_seed();
  sprintf(syntax,"syntax: x N count rel_err eps_max deps ipr_max");
  // use above default values if argc=1
  if(argc<8 && argc>1) 
    fehler(syntax);
  if(argc>=8){
    if(sscanf(argv[1],"%d",&x)==0) 
      fehler(syntax);
    if(sscanf(argv[2],"%d",&N)==0) 
      fehler(syntax);
    if(sscanf(argv[3],"%d",&count)==0) 
      fehler(syntax);
    if(sscanf(argv[4],"%lf",&rel_err)==0) 
      fehler(syntax);
    if(sscanf(argv[5],"%lf",&eps_max)==0) 
      fehler(syntax);
    if(sscanf(argv[6],"%lf",&deps)==0) 
      fehler(syntax);
    if(sscanf(argv[7],"%lf",&ipr_max)==0) 
      fehler(syntax);
  }

  sprintf(filename,"ipr_%d_%d_%5.3lf.dat",x,N,rel_err);

  fak=N;
  nq=0;
  while(fak){
    nq++;
    fak=fak>>1;
  }
  ntot=3*nq;
//  if(ntot>32) ntot=31;
  nl=ntot-nq;
  //========================
  long Nl=(long)pow((double)2,(double)nl);//(1<<(nl));
  long Ntot= (long)pow((double)2,(double)ntot); //(1<<(ntot));
printf("%d \t %d \t %d \n",nq,nl,ntot);
printf("%d \t %ld \t %ld \n",EXP2(nq),Nl,Ntot);
   printf("\n");

calc_ipr_fold_err(x,N,nq,ntot,count,rel_err,eps_max,deps,ipr_max,filename);

save_seed();
}
