#!/usr/bin/python
import sys
import argparse
import cartopy.crs as ccrs
import cartopy.io.shapereader as shpreader
import matplotlib as mpl
from pylab import *
from scipy.sparse.linalg import eigs
from tqdm import tqdm

parser = argparse.ArgumentParser(sys.argv[0], description="""
Plots a polical world map with countries colored according to their sensitivity
to the specified bank.
""")
parser.add_argument('-b,--bank', dest='bank', help='Bank name to test sensitivity against')
parser.add_argument('-c,--countries', dest='countries', help='File with country titles ordered by pagerank')
args = parser.parse_args(sys.argv[1:])

def read_matrix(fname):
    with open(fname, 'rb') as f:
        n = fromfile(f, dtype=int32, count=1)[0]
        g = fromfile(f, dtype=float64, count=n*n).reshape((n,n))
        return g

def pagerank(G):
    w,v = eigs(G, k=1)
    p = v[:,0].real
    p /= sum(p)
    return p

def sensitivity(bank, country):
    ib = banks.index(bank)
    ic = nbanks + countries.index(country)

    G = GR.copy()
    G[ic,ib] *= 1 + delta
    G[: ,ib] /= sum(G[:,ib])
    p = pagerank(G)

    return (log(p[ic]) - log(p0[ic])) / delta

banks     = [l.strip() for l in open('banks_by_pagerank.txt').readlines()]
countries = [l.strip() for l in open('countries_by_pagerank.txt').readlines()]

Grr = read_matrix('Grr.bin')
Gpr = read_matrix('Gpr.bin')
Gqr = read_matrix('Gqr.bin')

GR = Grr + Gpr + Gqr
p0 = pagerank(GR)

nbanks = len(banks)
ncountries = len(countries)
delta = 1e-3

shapename = 'admin_0_countries'
shp = shpreader.Reader(shpreader.natural_earth(
        resolution='110m', category='cultural', name=shapename))

cmap = cm.get_cmap('plasma')

fig = figure(figsize=(16,8))
ax = fig.add_axes([0.05,0.10,0.90,0.85], projection=ccrs.PlateCarree())

nc = len(list(shp.records()));
order = list(range(nc))

country_map = {
        'Russian Federation': 'Russia',
        'Bahamas': 'The Bahamas',
        'Timor-Leste': 'East Timor',
        "Côte d'Ivoire": 'Ivory Coast',
        'Palestine': 'State of Palestine',
        'Lao PDR': 'Laos',
        'Dem. Rep. Korea': 'North Korea',
        'Republic of Korea': 'South Korea',
        'Ireland': 'Republic of Ireland',
        'Georgia': 'Georgia (country)',
        'Brunei Darussalam': 'Brunei',
        'Northern Cyprus': 'Cyprus',
        'Somaliland': 'Somalia',
        'Macedonia': 'Republic of Macedonia',
        }
S = empty(nc, dtype=float64)
data = []
for i,country in enumerate(shp.records()):
    name = country.attributes['NAME_LONG']
    name = country_map.get(name, name)
    S[i] = sensitivity(args.bank, name) if name in countries else 0
    if name in countries:
        data.append((S[i], name))

data.sort(reverse=True, key=lambda t: t[0])

Smin = S.min()
Sdelta = S.max() - Smin

order.sort(key=lambda i: S[i])

for i,country in enumerate(shp.records()):
    name = country.attributes['NAME_LONG']
    name = country_map.get(name, name)
    c = cmap(order.index(i) / nc) if name in countries else 'w'
    #c = cmap((S[i] - Smin) / Sdelta) if name in countries else 'w'
    ax.add_geometries(country.geometry, ccrs.PlateCarree(), facecolor=c,
            edgecolor='k')

title('Country sensitivity to {}'.format(args.bank))

ax2  = fig.add_axes([0.25,0.05,0.50,0.05])
norm = mpl.colors.Normalize(vmin=0, vmax=1)
#norm = mpl.colors.Normalize(vmin=S.min(), vmax=S.max())
mpl.colorbar.ColorbarBase(ax2,cmap=cmap,norm=norm,orientation='horizontal')

savefig('sensitivity_{}.pdf'.format(args.bank))
with open('sensitivity_{}.txt'.format(args.bank), 'wt') as f:
    for s,c in data:
        f.write('{:20s}\t{:e}\n'.format(c, s))
