/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <cmath>                         // for ldexp
#include <iomanip>                       // for C++ I/O manipulators
#include <qinterface.h>                  // class definition
#include <qmanager.h>                    // a particular address manager
#include <qop_codes.h>                   // constants for operation codes

/* *********************************************************************
   | Setup a typedef to an actual address manager. Currently we only   |
   | have the Qaddress_manager class, so use that; any class deriving  |
   | from Qaddress_manager_base and fulfilling its requirements would  |
   | however do the job. This means that it will be very easy to plug  |
   | in a new address manager in the future without touching the rest  |
   | of the library.                                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   ********************************************************************* */
typedef Qaddress_manager ACTUAL_ADDRESS_MANAGER;

/* *********************************************************************
   | The following line of code initialises a static instance of a     |
   | Qinterface. Untill we will need more than one interface this is   |
   | a simple compromise. This object is created before main() begins  |
   | to be executed.                                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      06 Nov 2002         |
   ********************************************************************* */
static Qinterface quantum_interface;

/* *********************************************************************
   | This global function, defined in qinterface_base.h, exports the   |
   | static global instance of the quantum interface to the rest of    |
   | the world.                                                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      06 Nov 2002         |
   ********************************************************************* */
Qinterface_base &get_quantum_interface(void) { return quantum_interface; }

/* *********************************************************************
   | This static method is the start routine for the consumer thread,  |
   | when we use a threaded simulation or interface to quantum device. |
   | It simply calls run_evolution() within an infinite loop; the loop |
   | is broken only when the run_consumer variable becomes false and   |
   | the bytecode buffer is empty.                                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      15 Jul 2002         |
   ********************************************************************* */
#ifdef QINTERFACE_USES_PTHREADS
void *Qinterface::consumer(void *interface_pt) {
  /* correctly cast the passed value to a quantum interface */
  Qinterface &interface = *(Qinterface *)interface_pt;
  /* enter an infinite loop which can only be broken by the
     "run_consumer" variable (when it is false). */
  while (true) {
    /* test if any bytecode is present in the command queue. Reading this
       boolean value should be atomic in the STL library, so we don't care
       to get and release the mutex. The value can of course be incremented
       by the other thread during this cycle, but this is not a problem
       (the bytecode will be consumed during next cycle). */
    bool no_commands = interface.command_queue.empty();
    /* if the command queue is empty, either the producer has finished
       its job or it is busy preparing new data. */
    if (no_commands) {
      /* test the "run_consumer" variable. If it is false, then the
	 producer has finished. We can safely return since the command
	 queue is empty. No return value is needed. */
      if (interface.run_consumer == false) pthread_exit(NULL);
      /* if "run_consumer" is true, then the consumer was simply faster
	 than the producer. In this case we shoud return control to the
	 OS and wait, so that the buffer can be filled a bit. This avoids
	 that we spend a lot of CPU time in this loop testing the 
	 "run_consumer" variable again and again. */
      else sched_yield();
    }
    /* if there are commands in the bytecode buffer, it is time to
       consume it. Call our fake simulator. */
    else interface.run_evolution();
  }
}
#endif

/* *********************************************************************
   | Constructor for a quantum interface:                              |
   | 1) create an address manager and pass its pointer to the base     |
   |    class. The address manager must be informed about the quantum  |
   |    memory it must handle.                                         |
   | 2) if requested, spawn a thread which will consume the bytecode   |
   |    FIFO in an asyncronous way. (S.Bettelli, 15 Jul 2002)          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   ********************************************************************* */
Qinterface::Qinterface() :
  /* For the time being, fake the quantum device and assume we have 2^16
     qubits (!). Dynamically create an address manager, with this size. */
  Qinterface_base(new ACTUAL_ADDRESS_MANAGER(1<<16)) {
#ifdef QINTERFACE_USES_PTHREADS
  /* initialise the mutex objects (they are initially free). */
  pthread_mutex_init(&command_mutex, NULL);
  pthread_mutex_init(&feedback_mutex, NULL);
  /* turn on the consumer switch (tell it to run) */
  run_consumer = true;
  /* spawn a thread which will take care of the bytecode FIFO. */
  pthread_create(&consumer_thread, NULL, consumer, this);
#endif
}

/* *********************************************************************
   | Destructor for a quantum interface: it should shut down the byte- |
   | code buffer. Since the queue is an STL container we don't have to |
   | care about it explicitely. But it is safer to check that the      |
   | queue is indeed empty and consume it completely if it is not.     |
   | ----------------------------------------------------------------- |
   | If we are using the threaded approach, the consumer is told to    |
   | finish its job and then to return. After this, all thread related |
   | objects are destroyed. If we are using the serialised approach,   |
   | run_evolution() is called once, just to be sure that the buffer   |
   | is completely empty.                                              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   ********************************************************************* */
Qinterface::~Qinterface() {
#ifdef QINTERFACE_USES_PTHREADS
  /* "switch off" the consumer thread: when the bytecode buffer will
     be empty, the consumer thread will call pthread_exit. */
  run_consumer = false;
  /* wait for the consumer thread to terminate. */
  pthread_join(consumer_thread, NULL);
  /* destroy the mutex objects. */
  pthread_mutex_destroy(&command_mutex);
  pthread_mutex_destroy(&feedback_mutex);
#else
  /* be sure the bytecode buffer is completely empty. */
  run_evolution();
#endif
  /* if the bytecode buffer is still not empty, we have a problem. */
  if (! command_queue.empty())
    std::cerr << "Bytecode buffer not empty at shutdown!\n";
  /* test also if someone forgot some feedback on the queue. */
  if (! feedback_queue.empty())
    std::cerr << "Feedback buffer not empty at shutdown!\n";
}

/* *********************************************************************
   | This method waits until there is data in the feedback buffer, and |
   | it is only useful in the threaded implementation. Of course, lock |
   | the feedback buffer, so that we don't try to get it while someone |
   | is writing some feedback (in this case the result of the test     |
   | could be "false" but this wouldn't guarantee that there are enough|
   | message_type objects).                                            |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      17 Feb 2003         |
   ********************************************************************* */
inline void Qinterface::poll_feedback(void) {
  /* loop until there is data ... */
  while (true) {
    /* lock the feedback channel. */
    channel_feed_busy();
    /* test it */
    bool no_feedback = feedback_queue.empty();
    /* unlock the feedback channel. */
    channel_feed_free();
    /* if there is no feedback, wait a bit, otherwise exit. */
    if (no_feedback) sched_yield(); else break;
  }
}

/* *********************************************************************
   | This method tries to collect a measurement result from the quan-  |
   | tum device. It prepares an appropriately sized Qbitset (the size  |
   | is the passed argument, which we get from the calling register)   |
   | and fills it. If the size we read from the feedback channel does  |
   | not meet that specified as argument, the method throws an excep-  |
   | tion of type measurement_mismatch. Remember that this call is     |
   | blocking (i.e. it does not return until the device, or the simu-  |
   | lator, has performed the measurement and returned the result).    |
   | ----------------------------------------------------------------- |
   | In this class the results of measurements are found in another    |
   | FIFO queue. It is supposed that writing or reading a measurement  |
   | is an atomic operation. This implies some work with mutexes for   |
   | the threaded simulation.                                          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 20 Jul 2002         |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      17 Feb 2003         |
   ********************************************************************* */
Qbitset Qinterface::get_measurement_result(Qbitset::size_type bits) {
  /* the POPM macro is a shortcut for getting a value out of a
     buffer and delete it at the same time from the buffer. */
#define POPM read_queue(feedback_queue)
  /* prepare a qbitset with size=bits and initialise it to zero. */
  Qbitset measurement_result(bits);
  /* block till the feedback channel is ready. */
  poll_feedback();
  /* inhibit other accesses to the feedback FIFO */
  channel_feed_busy();
  /* the first value in the feedback buffer now is the number
     of bits which were measured. This number must be the same
     as the number of significant bits in the bitsets, otherwise
     there was an error somewhere and we should better stop here. */
  if (POPM != bits) throw measurement_mismatch();
  /* for each message_type object containing boolean values, read the
     object and store it in the appropriate position in our bitset. */
  for (message_type j = 0; j < measurement_result.elements(); ++j)
    measurement_result.set_element(j, POPM);
  /* allow accesses to the feedback FIFO again */
  channel_feed_free();
  /* return the Qbitset to the caller. */
  return measurement_result;
}

/* *********************************************************************
   | This method tries to collect the result of a locate request,      |
   | which is a Qubit_list holding the "physical addresses" of the     |
   | qubits in the device (whatever this means). This method is mainly |
   | useful in conjunction with simulators, in order to interpret the  |
   | content of the raw memory. The number of elements in the returned |
   | list must match "size", otherwise the method throws an exception  |
   | of type location_mismatch. Remember that this call is blocking    |
   | (i.e. it does not return until the device, or the simulator, has  |
   | calculated the physical locations and returned them).             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      17 Feb 2003         |
   ********************************************************************* */
Qubit_list Qinterface::get_locate_result(Qubit_list::size_type size) {
  /* prepare the Qubit_list to be returned. */
  Qubit_list location_result;
  /* block till the feedback channel is ready. */
  poll_feedback();
  /* inhibit other accesses to the feedback FIFO */
  channel_feed_busy();
  /* the first value in the feedback buffer now is the number
     of segments in the list to be returned. */
  Qubit_list::size_type num_segments = POPM;
  /* for each pair of message_type objects build a segment
     and push it into the "location_result" list. */
  for (message_type j = 0; j < num_segments; ++j) {
    Qubit_segment::address first = POPM;
    Qubit_segment::address last  = POPM;
    Qubit_list new_segment(first, last, true);
    location_result.join(new_segment);
  }
  /* allow accesses to the feedback FIFO again */
  channel_feed_free();
  /* be sure that the list to be returned is minimal. */
  location_result.make_minimal();
  /* The number of addresses (not segments!) in the "location_result"
     list must be the same as the passed argument, otherwise
     there was an error somewhere and we should better stop here. */
  if (location_result.size() != size) throw location_mismatch();
  /* return the calculated locations. */
  return location_result;
}

/* *********************************************************************
   | This routine is an example of decoder for the bytecode buffer. It |
   | swaps the buffer into a local structure and then consumes it. It  |
   | is assumed here that the buffer at the beginning contains only    |
   | complete initialisations, measurements, time slice operations or  |
   | control commands for the quantum device.                          |
   | In each cycle of the main loop a bytecode element is read (the    |
   | operation code) and its value is used to process the follosing    |
   | elements. Every unrecognisable code gets printed to stderr.       |
   | ----------------------------------------------------------------- |
   | (8 sep 2002) S.Bettelli Don't call POP twice in the same function |
   | call! The call order is undefined in this case! I'm going to pro- |
   | provide a solution for this in the form of a "define" statement.  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      14 May 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      19 Jul 2002         |
   ********************************************************************* */
void Qinterface::run_evolution(void) {
  /* If we are running a threaded run_evolution(), the safest approach
     is to move all the code from the bytecode buffer to a secondary buffer
     which will be then consumed without interference wiht the bytecode
     producer. Of course, the buffer is marked as busy during the bytecode
     swapping. If everything is serial, the following lines are actually
     unuseful, but we don't care since swapping is very fast. */
  command_queue_type queue_copy;
  channel_busy();
  queue_copy.swap(command_queue);
  channel_free();
  /* the POP macro is a shortcut for getting a value out of a bytecode
     buffer and delete it at the same time from the buffer. The READ_SEGMENT
     macro solves a problem of an undefined call sequence (POP should be
     called twice in the argument list of the Qubit_segment constructor).
     The last macro turns on and off the output to stdout (our "simulator") */
#define POP read_queue(queue_copy)
#define READ_SEGMENT(s) Qubit_segment::address first_##s = POP;  \
                        Qubit_segment::address last_##s  = POP;  \
			Qubit_segment s(first_##s, last_##s, true)
#define PRINTME true 
  /* be sure all digits are written out. */
  std::cout << std::setprecision(15);
  /* Now enter the real "simulation" loop. Each cycle will consume a
     complete time slice or an initialisation or a measurement or a
     control for the quantum device driver. This is safe since
     Qinterface_base blocks access to the bytecode buffer while
     it is writing one complete block of information. */
  while (! queue_copy.empty()) {
    /* pop the first element from the bytecode buffer. It should be a
       command, which we use to decide about further processing. */
    message_type command_code = POP;
    switch (command_code) {
      /* ************************************************************
	 | INITIALISATION OF A QUANTUM REGISTER                     |
	 ************************************************************ */
    case OPCODE_INITIALISATION: {
      /* the first value in the buffer now is the number of bits to be
	 initialised. Build a Qbitset with the appropriate size. */
      Qbitset init(POP);
      /* for each message_type object containing boolean values, read
	 the object and copy it into the "init" container. */
      for (message_type j = 0; j < init.elements(); ++j)
	init.set_element(j, POP);
      /* the next value is the number of [first,last] pairs which will
	 follow, i.e. the number of registers. Then, read the data for
	 the registers and "initialise" them with the previous bits. */
      message_type bit_counter = 0;
      message_type reg_segments = POP;
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	for (message_type k = 0; k < segment.size(); ++k, ++bit_counter)
	  /* REPLACE THE FOLLOWING LINE */
	  if (PRINTME) std::cout << std::setw(12) << "Init"
				 << "\ton " << segment[k] << "\t value "
				 << init.read(bit_counter) << std::endl;
      }
    } break;
    /* ************************************************************
       | MEASUREMENT OF A QUANTUM REGISTER                        |
       ************************************************************ */
    case OPCODE_MEASURE: {
      /* the next value is the number of [first,last] pairs which will
	 follow, i.e. the number of segments. Then, read the data for
	 the registers and "measure" them. We should keep a count of
	 how many qubits we have to measure for later use. */
      message_type count = 0;
      message_type reg_segments = POP;
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	count += segment.size();
	for (message_type k = 0; k < segment.size(); ++k)
	  /* REPLACE THE FOLLOWING LINE */
	  if (PRINTME) std::cout << std::setw(12) << "Measure"
				 << "\ton " << segment[k] << std::endl;
      }
      /* now generate fake data for the feedback buffer. You should
	 REPLACE THE FOLLOWING PIECE OF CODE with the real measurements. */
      Qbitset results(count);
      for (message_type j = 0; j < results.elements(); ++j)
	results.set_element(j, 0xAAAAAAAA);
      /* write these fake data to the feedback buffer. */
      channel_feed_busy();
      write_feedback(results.size());
      for (message_type j = 0; j < results.elements(); ++j)
	write_feedback(results.get_element(j));
      channel_feed_free();
    } break;
    /* ************************************************************
       | PHYSICAL LOCATION OF A QUANTUM REGISTER                  |
       ************************************************************ */
    case OPCODE_LOCATE: {
      /* the next value is the number of [first,last] pairs which will
	 follow, i.e. the number of segments. Then, read the data for
	 the registers and "locate" them. For the time being, this task
	 is very simple since we only have to copy the request into the
	 feedback channel (i.e. a trivial device does not apply any
	 further mapping to qubits). You should anyway REPLACE THE
	 FOLLOWING PIECE OF CODE with a real location routine if needed. */
      channel_feed_busy();
      message_type reg_segments = POP;
      write_feedback(reg_segments);
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	for (message_type k = 0; k < segment.size(); ++k)
	  /* REPLACE THE FOLLOWING LINE */
	  if (PRINTME) std::cout << std::setw(12) << "Locate"
				 << "\ton " << segment[k] << std::endl;
	write_feedback(segment.first());
	write_feedback(segment.last());
      }
      channel_feed_free();
    } break;
    /* ************************************************************
       | TENSOR PRODUCT OF HADAMARD GATES                         |
       ************************************************************ */
    case OPCODE_HADAMARD: {
      /* The Hadamard gate requires just one list of registers, so its
	 "simulation" is very similar to the procedure for measurements. */
      message_type reg_segments = POP;
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	for (message_type k = 0; k < segment.size(); ++k)
	  /* REPLACE THE FOLLOWING LINE */
	  if (PRINTME) std::cout << std::setw(12) << "Hadamard"
				 << "\ton " << segment[k] << std::endl;
      }
    } break;
    /* ************************************************************
       | TENSOR PRODUCT OF PHASE GATES                            |
       ************************************************************ */
    case OPCODE_PHASE: {
      /* The phase gate "simulation" works exactly like that of the
	 Hadamard gate, but we need to read and decode an additional
	 element from the buffer at the beginning: the phase parameter.
	 See the Qop_phase_like class for details on the encoding. */
      message_type parameter = POP;
      message_type reg_segments = POP;
      double phase = ((double)parameter/ldexp(1.0,QUANTUM_MESSAGE_TYPE_SIZE));
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	for (message_type k = 0; k < segment.size(); ++k) {
	  /* REPLACE THE FOLLOWING LINE */
	  if (PRINTME) std::cout << std::setw(12) << "Phase"
			    << "\ton " << segment[k]
			    << "\t" << phase << std::endl;
	}
      }
    } break;
      /* ************************************************************
	 | TENSOR PRODUCT OF CONDITIONAL PHASE GATES                |
	 ************************************************************ */
    case OPCODE_COND_PHASE: {
      /* Conditional phase gates have and additional parameter, like
	 simple phase gates, and they also need two lists of registers
	 (we can find corresponding registers in pairs). */
      message_type parameter = POP;
      double phase = ((double)parameter/ldexp(1.0,QUANTUM_MESSAGE_TYPE_SIZE));
      message_type twice_reg_segments = POP;
      for (message_type j = 0; j < twice_reg_segments/2; ++j) {
	READ_SEGMENT(segment_control);
	READ_SEGMENT(segment_target);
	for (message_type k = 0; k < segment_control.size(); ++k)
	  /* REPLACE THE FOLLOWING LINE */
	  if (PRINTME) std::cout << std::setw(12) << "Cond Phase"
				 << "\ton " << segment_control[k]
				 << "-" << segment_target[k]
				 << "\t" << phase << std::endl;
      }
    } break;
      /* ************************************************************
	 | YOU SHOULD'T REACH THIS POINT!                           |
	 ************************************************************ */
    case OPCODE_SWAP: 
      std::cerr << "This code should not be sent!\n"; exit(-1);
    default:
      std::cerr << "I should not reach this point!\n"; exit(-1);
    }
  }

}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
