/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */

#ifndef __QLIB_EXPOWER_AUXILIARY
#define __QLIB_EXPOWER_AUXILIARY

#include <cfloat>                           // for DBL_MANT_DIG
#include <vector>                           // for STL vectors
#include <utility>                          // STL utilities (like pair)
#include <qinterface_types.h>               // for the typedef's saga
#include <qexception.h>                     // for "quantum" exceptions
#include <qoperator.h>

/* *********************************************************************
   | This is an auxiliary class whose goal is to "calculate" the gates |
   | which are necessary to build up the circuit for the "expower"     |
   | function in the quantum library. In practice, the ctor of this    |
   | class fills in a data structure with one record for each gate     |
   | (each gate is a multicontrolled phase gate); that record contains |
   | an array of integers, specifying which qubits the gate acts on,   |
   | and a floating point value, which is the phase factor of the gate.|
   | The circuit is built in such a way that the first bit in each     |
   | register passed to the circuit will be considered, as usual, the  |
   | most significant bit (MSB).                                       | 
   | ----------------------------------------------------------------- |
   | Say q="qubits" the number of qubits, p="power" and f="factor";    |
   | then the basic idea for building a circuit which transforms the   |
   | computational basis state |n> into exp(2 PI i f n^p)|n>, with n   |
   | in the range [0, 2^q[, is to expand n into its binary representa- |
   | tion sum_j a_j 2^j, with j in [0, q-1]. This gives the following  |
   | expression for exp(i F n^p), with F = (2 PI f):                   |
   |   prod_{j_1 ... j_p} exp(i F a_j_1 ... a_j_p 2^(j_1 + ... + j_p)) |
   | Each product of binary digits a_j_? is either zero or one. Indeed |
   | it is 1 only if all digits are 1s. Then, each factor in the over- |
   | all product corresponds to a multicontrolled phase gate, acting   |
   | on the p qubits indexed by j_1 ... j_p, with phase factor equal   |
   | to exp(i F 2^(j_1 + ... + j_p)). This algorithm is described in:  |
   |                                                                   |
   | A.D.Chepelianskii and D.L.Shepelyansky, "Schroedinger cat anima-  |
   | ted on a quantum computer", quant-ph/0202113.                     |
   | ----------------------------------------------------------------- |
   | With the previous construction, the number of gates is obviously  |
   | q^p, which, for a fixed p, is polynomial in q. However, this class|
   | uses an optimised construction, which involves the minimal possi- |
   | ble number of multicontrolled gates, when each gate is considered |
   | as a primitive (the decomposition into one- and two-qubit primi-  |
   | tives is performed at a later stage): two different gates do NOT  |
   | concern the same (unordered) set of qubits.                       |
   | It is easy to calculate the number of gates in this construction; |
   | if we say (q k) the binomial coefficient "q choose k", we have:   |
   |       Ngates = sum_k (q k) for k in the range [1, min(q,p)]       |
   | For q<=p this sum gives (2^q - 1), while for q >> p it can be     |
   | approximated by q^p/p!. Altough the behaviour for fixed p is the  |
   | same as before, it allows for a significant reduction of the num- |
   | ber of gates: already for p=4, as is the case in the previously   |
   | cited paper, ~95.8% of the multicontrolled phase gates are saved  |
   | in the limit of large q. For q=5 and p=4 the saving is 95.2%.     |
   | ----------------------------------------------------------------- |
   | This construction involves the multiplication of "factor" by very |
   | large integers (growing like the factorial of "power" and expo-   |
   | nential in "qubits"); since the number of significant digits in   |
   | "factor" is limited, this means that the fractional part of this  |
   | multiplication (which is what interests us, since angles are cy-  |
   | clical) rapidly becomes pure numerical noise. It is not clear yet |
   | if there is any way to overcome this difficulty; currently, the   |
   | ctor and its subroutines try to detect when the result degenera-  |
   | tes into noise and warn the programmer. Any suggestion about a    |
   | possible solution is welcome!                                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2002         |
   ********************************************************************* */
class gate_calculator {
public:
  // a typedef for a qubit address (type for "qubits")
  typedef ::qubit_address_type address_type;
  // a typedef for a power value (type for "power")
  typedef unsigned int power_type;
  // a typedef for the floating point factor
  typedef double factor_type;
  // number of significant binary digits in factor_type
  static const unsigned int factor_digits = DBL_MANT_DIG;
public:
  // the most important ctor for this class
  gate_calculator(address_type the_qubits,
		  power_type   the_power,
		  factor_type  the_factor);
  // this method reads the gate description and creates the circuit
  Qop get_Qop(void);
private:
  // an exception for an overflow during an addition
  _Qexception(addition_overflow);
  // an exception for an overflow during a multiplication
  _Qexception(multiplication_overflow);
  // an exception for an overflow during a shift operation
  _Qexception(shift_overflow);
  // a generic exception stating that the gate construction was aborted
  _Qexception(gate_construction_aborted);
  // a generic unsigned integer (should contain power_type & address_type)
  typedef address_type number;
  // an unsigned integer type for storing VERY big integers
  typedef unsigned long long bignumber;
  // a vector of integers for keeping a partition of "power"
  typedef std::vector<power_type> partition_type;
  // a vector of integers for keeping a set of qubit addresses
  typedef std::vector<address_type> selection_type;
  // each record can contain a partition and its multiplicity
  typedef std::vector<std::pair<partition_type, bignumber> > partition_list;
  // each record can contain the description of a multicontrolled phase gate
  typedef std::vector<std::pair<selection_type, factor_type> > gate_list;
  // the range of qubit indexes will be [0, qubits[
  const address_type qubits;
  // this is the number of indexes which are combined toghether in each gate
  const power_type power;
  // this is the additional factor passed to the constructor
  const factor_type factor;
  // a list of partition descriptions filled by calculate_partitions
  partition_list partitions;
  // the list of all multicontrolled phase gates to be built
  gate_list gates;
  // this vectors stores all the factorials from 0 up to power
  std::vector<bignumber> factorials;
private:
  // calculates all the part. of p into k groups (save them in "partitions")
  void calculate_partitions(number k, power_type p);
  // calculates the multicontrolled phase gate list
  void calculate_gates(number k, address_type q);
  // calculates the phase factor for a gate given an index selection
  factor_type calculate_factor(selection_type a_selection);
  // throws exception if a+b is more than a bignumber
  bignumber sum_with_overflow_check(bignumber a, bignumber b);
  // throws exception if a*b is more than a bignumber
  bignumber multiply_with_overflow_check(bignumber a, bignumber b);
  // throws exception if a*2^b is more than a bignumber
  bignumber power_with_overflow_check(bignumber a, bignumber b);
};

/* *********************************************************************
   | This little inline function tries to understand if the sum of its |
   | two arguments can be performed within the limits of the "bignum-  |
   | ber" data type. This time we can't simply add and subtract the    |
   | second argument to the first to see if we get the first back,     |
   | because this is always true, even in presence of an otherflow.    |
   | But we can check that a+b is greater than or equal to a (right?). |
   | If there is an overflow, throw the appropriate exception.         |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 May 2002         |
   ********************************************************************* */
inline gate_calculator::bignumber
gate_calculator::sum_with_overflow_check(bignumber a, bignumber b) {
  if ((a + b) < a) throw addition_overflow();
  return (a + b);
}

/* *********************************************************************
   | This little inline function tries to understand if the multipli-  |
   | cation of its arguments can be performed within the limits of the |
   | "bignumber" data type. Since it is an integer type, the only way  |
   | for the multiplication to fail is through overflowing. In this    |
   | case, a division by b does not return a: throw an exception.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 May 2002         |
   ********************************************************************* */
inline gate_calculator::bignumber
gate_calculator::multiply_with_overflow_check(bignumber a, bignumber b) {
  if ((a * b)/b != a) throw multiplication_overflow();
  return (a * b);
}

/* *********************************************************************
   | This little inline function tries to understand if the multipli-  |
   | cation of its first argument times two to the second argument     |
   | can be performed within the limits of the "bignumber" data type.  |
   | Since multiplying times a power of two is a shift, one should     |
   | simply check that the reverse shift gives the first argument back |
   | (i.e. that no significant bits get lost on the left). If the test |
   | fails, throw an exception.                                        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 May 2002         |
   ********************************************************************* */
inline gate_calculator::bignumber
gate_calculator::power_with_overflow_check(bignumber a, bignumber b) {
  if ((a << b) >> b != a) throw shift_overflow();
  return (a << b);
}

#endif  // __QLIB_EXPOWER_AUXILIARY
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
