/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __TIME_SLICE_BASEGATES_
#define __TIME_SLICE_BASEGATES_

#include <string>                     // STL strings
#include <qinterface_types.h>         // for basic types
#include <qop_codes.h>                // macros for dealing with opcodes

/* *********************************************************************
   | An instance of this class describes (at high level) the base gate |
   | associated to a time slice. In some cases it could be possible to |
   | retrive the matrix representation of the gate. This class doesn't |
   | include a gate parameter, since the parameter itself is likely    |
   | more variable with respect to the other pieces of information     |
   | stored here. It is however possible to know if the gate needs a   |
   | parameter or not. In the original design, only a few objects from |
   | this class are supposed to be instantiated, as constants: they    |
   | will be then referenced by "real" slices (without duplication of  |
   | information).                                                     |
   | ----------------------------------------------------------------- |
   | Current fields:                                                   |
   |   index: an index in the gate database linking to the object      |
   |   opcode: the operation code for the associated time slice        |
   |   name: a human-readable string describing the associated gate    |
   |   parametrised: true if the gate needs an additional parameter    |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      10 Nov 2002         |
   ********************************************************************* */
class Qop_slice_basegate {
public:
  // the type for an operation code (must go through the interface)
  typedef ::quantum_message_type opcode_type;
  // a type for the index in the gate database
  typedef unsigned long size_type;
public:
  // inhibit the default constructor
  Qop_slice_basegate();
  // this ctor sets all the arguments and registers the base gate
  Qop_slice_basegate(opcode_type the_opcode,
		     const char *the_name, bool has_parameter);
public:
  // this returns the index of this base gate in the gate database
  size_type get_index(void) const { return index; }
  // this returns the operation code for the associated time slice
  opcode_type get_opcode(void) const { return opcode; }
  // this returns human-readable text describing the gate
  std::string get_name(void) const { return name; }
  // true if the gate needs an additional parameter
  bool is_parametrised(void) const { return parametrised; }
  // true if associated to an arity-1 fixed arity operation
  bool is_single(void) const { return (optype() == OPTYPE_SINGLE); }
  // true if associated to an arity-2 fixed arity operation
  bool is_double(void) const { return (optype() == OPTYPE_DOUBLE); }
  // true if associated to a macro operation
  bool is_macro(void) const  { return (optype() == OPTYPE_MACRO); }
  // the operation type associated to the operation code
  opcode_type optype(void) const { return GET_OPTYPE(opcode); }
private:
  // the index of this base gate in the gate database
  size_type index;
  // the operation code for the associated time slice
  opcode_type opcode;
  // human-readable text describing the gate
  std::string name;
  // true if the gate needs an additional parameter
  bool parametrised;
};

#endif  // __TIME_SLICE_BASEGATES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
