/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_REGISTER_QUBIT_LIST_TYPE_
#define __QUANTUM_REGISTER_QUBIT_LIST_TYPE_

#include <iostream>                 // C++ standard I/O
#include <list>                     // for STL doubly linked lists
#include <vector>                   // for STL vectors
#include <qexception.h>             // for "quantum" exception
#include <qsegment.h>               // for the basic address segment

/* this is a container for a collection of qubit lists. */
class Qubit_list;
typedef std::vector<Qubit_list> Qubit_list_collection;

/* *********************************************************************
   | This class is a container for Qubit_segments, and is used both    |
   | by quantum operators and quantum registers. While objects of the  |
   | Qubit_segment class hold contiguous regions of addresses, the     |
   | Qubit_list is more general and represents a generic set of ad-    |
   | dresses, though it is not very efficient when the set is highly   |
   | fragmented. The order of segments in the list and their orienta-  |
   | tion define a "natural" ordering of qubits inside the objects:    |
   | the "first" qubit is the most significant, the "last" qubit is    |
   | the least significant.                                            |
   | ----------------------------------------------------------------- |
   | (16 Dec 2001) The class now inherits privately from std::list.    |
   |  Some of the base class methods had to be exported and a few rou- |
   |  tines slightly modified, but it's worth the pain, since now no   |
   |  one is able to create (or modify to) a non-overlapfree list.     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   | Stefano Bettelli, added second ordering, 09 Jul 2001              |
   | Stefano Bettelli, split the class into two, 02 Aug 2001           |
   | Stefano Bettelli, return to a single class + recalc, 29 Nov 2001  |
   | Stefano Bettelli, private inheritance from std::list, 16 Dec 2001 |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse, 13 May 2002              |
   | Stefano Bettelli, added a run_remap() method, 13 May 2002         |
   | Stefano Bettelli, added a ctor from a vector, 14 May 2002         |
   ********************************************************************* */
#define _QUBIT_LIST_BASE_CLASS std::list<Qubit_segment>
class Qubit_list : private _QUBIT_LIST_BASE_CLASS {
public:
  // the type of the immediate base class
  typedef _QUBIT_LIST_BASE_CLASS base_class;
  // size_type is the same type as for qubit segments
  typedef value_type::size_type size_type;
  // address is the same type as for qubit segments
  typedef value_type::address address;
  // the constant iterator type exported
  typedef base_class::const_iterator const_iterator;
public:
  // a class for problems during the construction of a list
  _Qexception(invalid_construction);
  // a class for problems during list splicing
  _Qexception(invalid_splice);
  // a class for invalid address translations
  _Qexception(invalid_translation);
  // a class for problems during list joins
  _Qexception(invalid_join);
  // a class for problems during creation of an index list
  _Qexception(invalid_permutation);
  // a class for problems during the index->address translation
  _Qexception(out_of_range);
  // a class for problems during the invert operation
  _Qexception(invert_failed);
  // a class for errors during the split operation
  _Qexception(split_error);
  // a class for highest() or lowest() method called on empty list
  _Qexception(invalid_limit);
  // a class for problems in the break_overlap method
  _Qexception(invalid_break_overlap);
public:
  // default constructor (creates an empty list)
  Qubit_list() { }
  // copy constructor
  Qubit_list(const Qubit_list &a_list) : base_class(a_list) { }
  // yet another copy ctor which incorporates (by swapping) the supplied list
  Qubit_list(Qubit_list &a_list, bool);
  // ctor with a single segment (first address and size != 0)
  Qubit_list(address the_first, size_type the_size);
  // ctor with a single segment (first and last address + disambiguation)
  Qubit_list(address the_first, address the_last, bool);
  // ctor with a single segment (const reference to a segment)
  Qubit_list(const Qubit_segment &a_segment);
  // ctor reading from a vector of qubit addresses
  Qubit_list(const std::vector<address> &a_list);
  // this is the assignement operator
  Qubit_list &operator=(const Qubit_list &a_list);
  // trivial virtual destructor
  virtual ~Qubit_list() { }
private:
  // the range constructor (exports a constructor from STL lists)
  Qubit_list(const_iterator segment_begin, const_iterator segment_end);
public:
  // a constant iterator to the first segment
  const_iterator begin(void) const { return base_class::begin(); }
  // a constant iterator to the past-the-end segment
  const_iterator end(void) const { return base_class::end(); }
  // a constant reference to the first segment
  const Qubit_segment &front(void) const { return base_class::front(); }
  // a constant reference to the last segment
  const Qubit_segment &back(void) const { return base_class::back(); }
  // this returns the number of qubits in the list
  size_type size(void) const;
  // this returns the number of segments in the list
  size_type segments(void) const { return base_class::size(); }
  // this returns the highest address in the segments of the list
  address highest(void) const;
  // this returns the lowest address in the segments of the list
  address lowest(void) const;
  // this reverses the order of qubits inside the list.
  Qubit_list &reverse(void);
  // an interface to the base class swap (exchanges two lists)
  void swap(Qubit_list &a_list) { base_class::swap(a_list); }
public:
  // this checks whether two lists are overlapping
  bool is_overlapping(const Qubit_list &second_list) const;
  // this checks whether the current list overlaps the passed segment
  bool is_overlapping(const Qubit_segment &a_segment) const;
  // this checks whether the list has no segments
  bool is_empty(void) const { return base_class::empty(); }
  // this checks whether the list has only one segment
  bool is_single(void) const;
  // this checks whether the list contains more than one segment
  bool is_multiple(void) const;
  // this checks whether any join can be done or not
  bool is_minimal(void) const;
  // similar to is_minimal, but segments are actually joined.
  void make_minimal(void);
  // this method is the comparison operator for qubit lists
  bool operator==(const Qubit_list &a_list) const;
  // this method is the opposite of the comparison operator for qubit lists
  bool operator!=(const Qubit_list &a_list) const;
public:
  // put all addresses in the list in increasing value order
  void reorder(void);
  // reorder two lists and eliminate the common addresses
  void reorder_and_annihilate(Qubit_list &second_list);
  // this method is an equivalence test for qubit lists
  bool is_equivalent(const Qubit_list &a_list) const;
  // alias for get_subrange() without the additional arguments
  Qubit_list operator()(address an_index, size_type a_size = 0) const;
  // this returns a subrange of the current list with size = 1
  Qubit_list get_subrange(address an_index, address &selected_address,
			  const_iterator &selected_segment) const;
  //this returns a subrange of the current list with non trivial size
  Qubit_list get_subrange(address an_index, size_type a_size,
			  address &begin_address, address &end_address,
			  const_iterator &begin_segment,
			  const_iterator &end_segment) const;
  // this method performs the index to address translation
  Qubit_list translate(const Qubit_list &address_list) const;
public:
  // this static method performs the mixing of a list collection
  static Qubit_list mix_lists(Qubit_list_collection &a_collection);
  // this static method calculates a permutation list
  static Qubit_list create_permutation(Qubit_list &first_list,
				       Qubit_list &second_list,
				       size_type total_size);
  // this erases some qubits from the first segment of the list
  void pop_front(size_type the_size = 0); 
  // this splices the two lists and tries to join the segments on the border
  bool join(Qubit_list &second_list);
  // an utility for the split operation
  void run_split(size_type head, size_type jump);
  // an utility for the invert operation
  void run_invert(size_type head, size_type size);
  // an utility for the remap operation
  void run_remap(const Qubit_list &permutation_list);
private:
  // interface to the non-const begin iterator
  iterator begin(void) { return base_class::begin(); }
  // interface to the non-const end iterator
  iterator end(void) { return base_class::end(); }
  // this splits an entry of the list in two parts (given new size)
  void split(iterator first_element, size_type new_size_first);
  // this splits an entry of the list in two parts (given break point)
  void split_at(iterator first_element, address break_point);
  // this tries to join the segment with the next one + cleanup
  bool try_join_and_cleanup_forward(iterator a_segment);
  // this tries to join the segment with the previous one + cleanup
  bool try_join_and_cleanup_backward(iterator a_segment);
  // this tries to join three segments and cleans up on success
  bool try_join_and_cleanup_both_sides(iterator a_segment);
  // this method performs the index to address translation (with maps)
  Qubit_list  translate_map(const Qubit_list &address_list) const;
  // this method performs the index to address translation (without maps)
  Qubit_list  translate_nomap(const Qubit_list &address_list) const;
  // this is code is common to translate_map and translate_nomap
  Qubit_list *translate_segment(const Qubit_list &address_list,
				const const_iterator &index_segment,
				bool reset_search) const;
  // this can be used as deep engine for formal-to-real address translation
  Qubit_list *extract(address low_address, address high_address,
		      const_iterator low_address_segment,
		      const_iterator high_address_segment, bool reverse) const;
  // deep engine for the index to address translation (higher level)
  address find_element(address an_index, const_iterator &a_segment) const;
  // deep engine for the index to address translation (lower level)
  size_type find_segment(address an_index, const_iterator &a_segment) const;
  // modify segments as a preparation to "reorder_and_annihilate"
  void break_overlap_with(Qubit_list &second_list,
			  iterator first_segment, iterator second_segment);
};

// this output function shows the content of a list (for debugging)
std::ostream &operator<<(std::ostream &os, const Qubit_list &a_list);

/* *********************************************************************
   | This is a sort of "copy constructor" which incorporates the con-  |
   | tent of the given list with a list swap. Hence, after the call    |
   | the list supplied as an argument is empty. The boolean argument   |
   | is needed only for disambiguation.                                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 Nov 2001         |
   ********************************************************************* */
inline Qubit_list::Qubit_list(Qubit_list &a_list, bool) {
  /* the method of the base class can be used. */
  swap(a_list);
}

/* *********************************************************************
   | This is the range constructor for a list which accepts two itera- |
   | tors to another list (as customary for STL lists).                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 07 Feb 2001         |
   ********************************************************************* */
inline Qubit_list::Qubit_list(const_iterator segment_begin,
			      const_iterator segment_end) :
  /* call the STL copy range constructor */
  base_class(segment_begin, segment_end) { }

/* *********************************************************************
   | The basic constructor of a single segment list requires the first |
   | address and the size of the region of addresses. It will create a |
   | list with a single segment with these informations. This approach |
   | is used when creating a register from scratch. Note that the      |
   | segment is never reversed in this case.                           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 28 Aug 2001         |
   ********************************************************************* */
inline Qubit_list::Qubit_list(address the_first,
			      size_type the_size) :
  /* create the list with a single appropriately sized segment. */
  base_class(1, Qubit_segment(the_first, the_size)) { }

/* *********************************************************************
   | A variation of the previous constructor, which accepts the abso-  |
   | lute addresses of the first and last qubit (this allows creation  |
   | of a single segment list with a reversed segment). The boolean    |
   | argument is needed only to disambiguate the ctor resolution       |
   | (otherwise it would have the same prototype as the other one,     |
   | because address can be the same type as size_type).               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 28 Aug 2001         |
   ********************************************************************* */
inline Qubit_list::Qubit_list(address the_first,
			      address the_last, bool) :
  /* create the list with the appropriate segment (with strange ctor). */
  base_class(1, Qubit_segment(the_first, the_last, true)) { }

/* *********************************************************************
   | Another constructor of a list which requires only a reference to  |
   | an already existing segment: this is used by the address manager. |
   | Of course, it copies the passed segments at the beginning of the  |
   | (empty) list.                                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 28 Aug 2001         |
   ********************************************************************* */
inline Qubit_list::Qubit_list(const Qubit_segment &a_segment) :
  /* create the list with a copy of the segment. */
  base_class(1, a_segment) { }

/* *********************************************************************
   | This is the assignement operator. Currently, it only calls the    |
   | assignement between STL lists, but one never knows.               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 Nov 2001         |
   ********************************************************************* */
inline Qubit_list &
Qubit_list::operator=(const Qubit_list &a_list) {
  /* run assignement between the lists. */
  base_class::operator=(a_list);
  /* return the current object */
  return *this;
}

/* *********************************************************************
   | The opposite of the comparison operator, it returns true if the   |
   | two lists are not identical. It is the negation of what is retur- |
   | ned by the comparison operator.                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 05 Dec 2001         |
   ********************************************************************* */
inline bool Qubit_list::operator!=(const Qubit_list &a_list) const {
  return ! (*this == a_list);
}

/* *********************************************************************
   | This method returns "true" if the list contains only one segment, |
   | "false" otherwise. This method is useful for optimising the algo- |
   | rithm for some other method.                                      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      26 Oct 2002         |
   ********************************************************************* */
inline bool Qubit_list::is_single(void) const {
  return (!is_empty() && ++begin() == end());
}

/* *********************************************************************
   | This method returns "true" if list contains two or more segments. |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      26 Oct 2002         |
   ********************************************************************* */
inline bool Qubit_list::is_multiple(void) const {
  return (!is_single() && !is_empty());
}

/* *********************************************************************
   | This method erases "the_size" qubits from the first segment of    |
   | the current list. If the supplied size is zero, the whole first   |
   | segment is erased. In the other case, the segment is split in two |
   | parts and the first is eliminated. Remember that if "the_size"    |
   | exceedes the size of the first segment, the split method will     |
   | throw an exception.                                               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Dec 2001         |
   ********************************************************************* */
inline void Qubit_list::pop_front(size_type the_size) {
  /* an empty list cannot be popped! */
  if (is_empty()) throw split_error();
  /* if the_size is not zero (which means: erase the whole first segment) 
     split the first segment in two parts. If the size is too large, the
     split method will throw an exception. */
  if (the_size != 0) split(begin(), the_size);
  /* now erase the first segment, whichever it is. */
  base_class::pop_front();
}

/* *********************************************************************
   | This method is a variation of split() which accepts the address   |
   | at which the segment must be broken ("break_point" is the first   |
   | address of the second segment if the segment is_direct() or the   |
   | last address of the first segment if the segment is_reversed()).  |
   | After this call, one part contains only addresses which are equal |
   | to or greater than "break_point", the other one only smaller      |
   | addresses. first() address of the first segment will be the same  |
   | as first() address of the whole segment (thus relative natural    |
   | ordering is preserved).                                           |
   | ----------------------------------------------------------------- |
   | Dangerous! no test is run in order to be sure that "break_point"  |
   | is really inside the original segment! This is only a wrapper for |
   | a call to split() after the new size of the first segment has     |
   | been calculated. The size of the new "first" part depends on      |
   | segment->is_direct() or equivalently on segment->first() being    |
   | smaller than "break_point" or not (smaller, NOT smaller OR equal) |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 08 Jun 2001         |
   ********************************************************************* */
inline void Qubit_list::split_at(iterator first_element, address break_point) {
  split(first_element, (break_point > first_element->first() ?
 			break_point - first_element->first() :
 			(first_element->first() - break_point) + 1));
}

/* *********************************************************************
   | This method is a small inline function which chooses one of two   |
   | implementations for the index to address translation (see method  |
   | "translate_map" for further details). A simplified routine is     |
   | selected when there are not "too much" segments in the current    |
   | and address list: it has not been studied which is the best tre-  |
   | shold yet! Currently, the numbers of segments are multiplied and  |
   | if the result is less than five the routine without maps is called|
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      22 Jul 2002         |
   ********************************************************************* */
inline Qubit_list Qubit_list::translate(const Qubit_list &address_list) const {
  /* When the number of segments in the lists is not "too much" we should
     better call the simpler routine which does not use maps. */
  if ((this->segments() * address_list.segments()) <= 5)
    return translate_nomap(address_list);
  /* In the general case, call the expensive routine with maps. This rou-
     tine guarantees that its complexity does not explode with the product
     of the number of segments of the two lists, but with their sum. */
  else
    return translate_map(address_list);
}

#endif  // __QUANTUM_REGISTER_QUBIT_LIST_TYPE_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
