/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <cmath>                       // for some exotic math functions
#include <qop_phase_like.h>            // the definition of the class

/* *********************************************************************
   | This method encodes a pair (k,x) into a mantissa. The pair corre- |
   | sponds to the angle (2 PI x / 2^k). The fraction x / 2^k is then  |
   | written in the form A / 2^n, where n is the number of bits in the |
   | representation of A, i.e. 0 <= A < 2^n. Of course, this is done   |
   | modulo 1, and the final result can be an approximation of the     |
   | initial fraction, since angles are "quantised" (the "quantum" of  |
   | angle will be 2 PI / 2^n). The mantissa A is the returned.        |
   | ----------------------------------------------------------------- |
   | If x/2^k is an integer number, this method generates an exception |
   | of type invalid_angle instead of creating the identity slice.     |
   | Actually, it generates an exception also when the_power is so     |
   | large that the mantissa becomes zero (i.e. when the angle cannot  |
   | be represented in this fixed point representation.)               |
   | ----------------------------------------------------------------- |
   | (27 Jan 2003) S.Bettelli. Use a different exception when the      |
   |   angle becomes impossible to represent (angle_underflow).        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         | 
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Jan 2003         | 
   ********************************************************************* */
Qop_phase_like::mantissa_type Qop_phase_like::encode_mantissa
(power_type the_power, floating_type the_fraction) {
  /* If the fraction is negative, transform it into a positive number
     and remember that it was negative. */
  bool fraction_is_negative = (the_fraction < 0.0);
  if (fraction_is_negative) the_fraction = - the_fraction;
  /* Reduce the two inputs to a standard form, where either the_power
     is zero or the_fraction is strictly smaller than one. */
  while (the_fraction >= 1.0 && the_power > 0) 
    { the_fraction /= 2.0; --the_power; }
  /* If the_fraction is still greater= than one, then the_power must
     be zero, so that x / 2^k is simply x. In this case the integer
     part of the_fraction is redundant, and the_fraction must be put
     in the range [0,1[. Use the modf POSIX function for this. */
  double dummy;
  if (the_fraction >= 1.0) the_fraction = modf(the_fraction, &dummy);
  /* If the_fraction is now zero, then the initial x/2^k was an integer
     number, which corresponds to the identity. Treat it the same way
     as x=0, i.e. throw an exception. */
  if (the_fraction == 0.0) throw invalid_angle();
  /* Now, transform the fraction into a long unsigned integer number.
     First, multiply it by 2^n, then cast to an unsigned integer.
     Use the ldexp POSIX function for this. Add an explicit cast in
     order to avoid warnings from the compiler. */
  mantissa_type the_mantissa = (mantissa_type)
    ldexp(the_fraction, QUANTUM_MESSAGE_TYPE_SIZE);
  /* If k is still not zero, the mantissa must be shifted to the right.
     If the_power is big enough, the mantissa can become zero. I don't
     know which is the most intelligent thing to do in this case;
     currently I throw an exception of type "angle_underflow". */
  the_mantissa >>= the_power;
  if (the_mantissa == 0) throw angle_underflow();
  /* If the_fraction was initially negative, we have to change mantissa
     into (2^n - mantissa). This means taking the 2's complement.*/
  if (fraction_is_negative) the_mantissa = ((~ the_mantissa) + 1);
  /* Return the calculated mantissa */
  return the_mantissa;
}

/* *********************************************************************
   | This method decodes the current mantissa into a pair (k,x). It is |
   | the "inverse" of the encode_mantissa method (where one should look|
   | for more details on the encoding-decoding process). The values of |
   | the pair are returned through the arguments passed by reference.  |
   | ----------------------------------------------------------------- |
   | Note: this must be non-static! There is a call to get_mantissa()  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         | 
   ********************************************************************* */
void Qop_phase_like::decode_mantissa
(power_type &the_power, floating_type &the_fraction) const {
  /* calculate the angle corresponding to the mantissa, undoing a part
     of the calculations of the ctor. Use the POSIX ldexp function.
     Then shift the ]+0.5,+1.0[ range into ]-0.5,-0.0[ (subtracting 1) */
  floating_type angle = ldexp(get_mantissa(), - QUANTUM_MESSAGE_TYPE_SIZE);
  if (angle > 0.5) angle -= 1.0;
  /* split the angle into a fractional part whose absolute value is in
     the range [0.5,1[ and an exponent (the base is two). Use the frexp
     POSIX function for this (the second argument must be a pointer to
     an integer, sorry). */
  int the_power_as_int;
  the_fraction = frexp(angle, &the_power_as_int);
  /* if the fraction is exactly +0.5 or -0.5 now, it is better to
     increase it to +1 or -1 and to subtract one from the_power.
     If I got it correctly, the case -0.5 should never take place. */
  if (the_fraction == +0.5) { the_fraction = +1.0; --the_power_as_int; }
  if (the_fraction == -0.5) { the_fraction = -1.0; --the_power_as_int; }
  /* Change sign to the_power (we use it in the denominator) and return. */
  the_power = - the_power_as_int;
  return;
}

/* *********************************************************************
   | This method outputs a formatted string which specifies the phase  |
   | shift. Indeed it prints "k=" followed by the k parameter, and     |
   | also "x=" followed by x if it is not 1. The corresponding phase   |
   | is e^( 2 PI i x / 2^k ). The sign of x is written before k.       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      11 Nov 2002         |
   ********************************************************************* */
void Qop_phase_like::get_details(std::ostream &a_stream) const {
  /* decode the mantissa into a pair (x,k). The following line is indeed
     a macro, which declares and fills the_power and the_fraction. */
  get_power_fraction_macro;
  /* take out the sign of the_fraction */
  bool fraction_is_negative = (the_fraction < 0.0);
  if (fraction_is_negative) the_fraction = (- the_fraction);
  /* write the value of the_power into the stream, and attach the
     sign of the_fraction to it (this is my choice). */
  a_stream << " k=" << (fraction_is_negative ? "-" : "") << the_power;
  /* write the absolute value of the fraction only if it is not
     exactly one or minus one (this is my choice). */
  if (the_fraction != 1.0)
    a_stream << ",x=" << the_fraction;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
