/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <qop_summary.h>                        // the summary class
#include <qop_slice_basegate.h>                 // a class for basegates      

/* *********************************************************************
   | This method registers some information about a base gate type in  |
   | the internal database. First, it checks the operation code: if    |
   | it is already known, the method throws a meaningful exception.    |
   | Otherwise the pointer to the base gate is saved in the internal   |
   | database for later reference.                                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 Nov 2002         |
   ********************************************************************* */
Qop_summary::size_type Qop_summary::register_basegate
(const Qop_slice_basegate *a_basegate) {
  /* a mutable reference to the database in the summary. */
  database_type &database = get_Summary().database;
  /* Check that the operation code for "a_basegate" has not made its
     way into the database yet. If it is already there, throw an
     exception in order to signal this. */
  for (database_type::iterator an_entry = database.begin();
       an_entry != database.end(); ++an_entry)
    if ((*an_entry)->get_opcode() == a_basegate->get_opcode())
      throw gate_already_registered();
  /* push the entry into the internal database. */
  database.push_back(a_basegate);
  /* the index of the base gate is now the size of the database minus
     one. Return this value so that the base gate can know about it. */
  return (database.size() - 1);
}

/* *********************************************************************
   | This output function prints out the information content of the    |
   | passed base gate summary. Since this function is designed for an  |
   | easy debugging, print out as much information as you can.         |
   | ----------------------------------------------------------------- |
   | 1) the index into the list of base gates                          |
   | 2) the operation code and the human-readable name                 |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 Nov 2002         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qop_summary &summary) {
  const char *line = "---------------------------";
  /* output a title and the pointer to the summary. */
  os << "Summary (" << &summary << ") of available time slices:\n"
     << '\t' << line << "\n\tIndex\tOpcode\tName\n\t" << line << '\n';
  /* output the index to opcode list. */
  for (Qop_summary::size_type index = 0; 
       index < Qop_summary::get_gates(); ++index)
    os << "\t(" << index << ")"
       << "\t" << Qop_summary::get_basegate(index)->get_opcode()
       << "\t" << Qop_summary::get_basegate(index)->get_name() << '\n';
  /* return the output stream. */
  return os;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
