/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
/* *********************************************************************
   |                                                                   |
   |         #########        C R E D I T S        #########           |
   |                                                                   |
   | The following code was inspired by a quantum simulator written by |
   | Alexei Chepelianskii; Alexei's simulator has been used to produce |
   | the data shown in quant-ph/0202113 (available on the arXiv), now  |
   | published as: "Simulation of chaos-assisted tunneling in a semi-  |
   | classical regime on existing quantum computers", Phys. Rev. A     |
   | v.66 (2002) p.054301.                                             | 
   ********************************************************************* */
#ifndef __BASIC_QUANTUM_SIMULATOR_
#define __BASIC_QUANTUM_SIMULATOR_

#include <complex>                          // C++ complex numbers

/* *********************************************************************
   | The Qsimulator class is a proof-of-concept quantum simulator for  |
   | the Q programming language. It was written in order to show an    |
   | actual implementation of a consumer task which is fed by the      |
   | bytecode produced by the quantum library. However, it should be   |
   | stressed that in NO WAY this is the preferential simulator to be  |
   | attached to the language: better, you should use it to get inspi- |
   | ration about writing your own simulator.                          |
   | ----------------------------------------------------------------- |
   | The internals of the simulator are very simple: n qubits are ma-  |
   | naged as an array of 2^n complex numbers. The simulator interface |
   | allows for the following operations to be performed: initialisa-  |
   | tion of a qubit, measurement of a qubit (with and without wave    |
   | function collapse), Hadamard gates, phase gates and conditional   |
   | phase gates. Moreover, it is possible to ask for the probability  |
   | that a qubit is in |0> or in |1> (partial trace) and to access as |
   | constant objects the elements of the array.                       |
   | ----------------------------------------------------------------- |
   | In the array, the element indexed by the bit string "x" corres-   |
   | ponds to the projection of the quantum state on the computational |
   | basis element |x> (therefore x as an unsigned integer number must |
   | be in the range [0, 2^n - 1].                                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
class Qsimulator {
public:
  // the numeric type for a real number
  typedef double rnumber;
  // the numeric type for a complex number
  typedef std::complex<rnumber> cnumber;
  // an integer type for specifying addresses
  typedef unsigned long address;
  // an integer type for specifying sizes
  typedef unsigned long size_type;
public :
  // default constructor
  Qsimulator() { set_numqubits(0); set_noise(0.0); }
  // destructor (deallocating classical memory)
  ~Qsimulator() { if (size()) delete [] data; }
  // index-like access to the classical memory elements
  const cnumber &operator[](address index) const { return *(data + index); }
  // this returns the number of elements in the simulator
  size_type size(void) const { return memsize; }
  // this returns the number of simulated qubits
  size_type qubits(void) const;
  // this returns the probability that the target-th qubit is |0>
  rnumber is_unset(address target) const { return 1.0 - is_set(target); }
  // this returns the probability that the target-th qubit is |1>
  rnumber is_set(address target) const;
  // this returns the amplitude of the index-th basis element
  cnumber amplitude(size_type index) const;
  // squared modulus of the amplitude of the index-th basis element
  rnumber probability(size_type index) const;
public:
  // set the gate noise level for the simulation
  void set_noise(rnumber amplitude) { noise = amplitude; }
  // set the number of simulated qubits
  void set_numqubits(size_type num_qubits);
  // (re)initialisation of the random number generator
  unsigned long init_random_seed(unsigned long seed=0);
public:
  // this initialises the target-th qubit to "value"
  void init(address target, bool value);
  // this measures the target-th qubit and returns the result
  bool measure(address target) { return do_measurement(target, true); }
  // just like measure, but the quantum state does not collapse
  bool fake_measure(address target) { return do_measurement(target, false); }
  // this applies a Hadamard gate on the target-th qubit
  void hadamard(address target);
  // this applies a phase gate on the target-th qubit
  void phase(address target, rnumber fraction);
  // this applies a conditional phase gate
  void condphase(address control, address target, rnumber fraction);
private:
  // coefficients for the Hadamard gate
  void prepare_hadamard(cnumber &m00, cnumber &m01);
  // coefficient for the phase or conditional phase gate
  cnumber prepare_phase(rnumber fraction);
  // reimplementation of norm() of complex.h for performance
  static rnumber norm2(const cnumber &z);
  // this implements a measurement (with or without collapse)
  bool do_measurement(address target, bool collapse);
  // this collapses the quantum state of the target-th qubit to value
  void do_collapse(address target, bool value, rnumber probability);
private:
  // a parameter which governs the intensity of errors
  rnumber noise;
  // this is the amount of classical memory we use
  size_type memsize;
  // this is a pointer to the memory block
  cnumber *data;
};

/* *********************************************************************
   | For the time being, the simulator is a global object. It is de-   |
   | clared as an external object here, so that each file including    |
   | the class declaration will have access to the simulator too.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
extern Qsimulator simulator;

#endif  // __BASIC_QUANTUM_SIMULATOR_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
