/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iostream>
#include <quantum_computing.h>
#include <quantum_library.h>
#include <qsimulator.h>

int main(void) {
  try {
    /* This example reproduces the left-right oscillations (without noise)
       shown in quant-ph/0202113 (available on the arXiv), now published as:
       "Simulation of chaos-assisted tunneling in a semiclassical regime
       on existing quantum computers", Phys. Rev. A v.66 (2002) p.054301. */
    int    q     = 5;                // resources (# of qubits)
    int    c     = 2;                // number of simulated momentum cells
    int    niter = 500;              // number of map iterations
    double noise = 0.00;             // noise intensity
    double K     = 0.04;             // K = k * T defines the classical map
    double a     = 1.6;              // "centre" of the potential (2*PI*a/L)
    int N = (1 << q);                // number of quantum levels
    int memsize = q+1;               // simulated memory
    /* initialisation of the simulator. */
    simulator.set_numqubits(memsize);
    simulator.set_noise(noise);
    /* The following two operators commute from position to momentum
       representation and back. Since these two representations are
       conjugate, these Qops are simply Fourier transforms. */
    Qop mom_to_pos = QFourier(q);
    Qop pos_to_mom = !mom_to_pos;    // this is an anti-transform
    /* Coefficient and operator for the free evolution of any quantum map.
       In a more standard notation, this is the kinetic term exp(-iTn^2/2),
       therefore we get T = 2*PI*c / N . If Tsym is true, add a few gates
       in order to have a perfect p symmetry (i.e.  i --> i+1/2). */
    Qop rotator = expower(q, 2, - c/(2.0*N));
    /* This is the construction of the "kick" for the double well map
       algorithm. Remember that ((i+DD)^2 - AA^2)^2 = sum_j f_j i^j.
       The choice DD = 0.5*(1-N) guarantees that we take a coordinate
       range which is symmetric around zero. */
    double lb = 2*M_PI;              // adimensional cell size
    double AA = a * (N/lb);          // scaled "centre" for the potential
    double DD = 0.5 * (1-N);         // x_i = (L/N)(i + DD)
    double f  = - (lb*lb * K)/(c * N*N*N);
    double f1 = f * 4*DD * ( DD*DD - AA*AA );
    double f2 = f * ( 6*DD*DD - 2*AA*AA );
    double f3 = f * 4*DD;
    double f4 = f;
    Qop kick = (expower(q, 1, f1) & expower(q, 2, f2) &
		expower(q, 3, f3) & expower(q, 4, f4));
    /* The following operator is a complete map step. */
    Qop mapstep = kick & pos_to_mom & rotator & mom_to_pos;
    /* This operator will "prepare" the initial state. */
    Qop preparation = QNot(1) & QHadamard(q-1)>>1;
    /* statistics for one map step. */
    std::cerr << "\nQubit utilisation for one map step:\n";
    mapstep.print_statistics(std::cerr);
    /* initialise the random seed. */
    simulator.init_random_seed();
    /* reset the register */
    Qreg simul(q);
    /* prepare the initial state */
    preparation(simul);
    /* now iterate the map "niter" times. */
    for (int count = 0; count < niter; ++count) { 
      /* get the probability of being on the left or on the right (i.e.
	 the probability that the most significant qubit is |0> or |1>). */
      unsigned long msb = (simul.locate())[0];
      /* read the exact probability by accessing the simulator. */
      double prob_on_the_left = simulator.is_set(msb);
      /* run a step of the map */
      mapstep(simul);
      /* print out the current result. */
      std::cout << (count + 1) << '\t' << prob_on_the_left << std::endl;
    }
    /* skip the next example. */
    goto end;
    
    /* This example reproduces the address manager status which
       is shown in figure 2, pag.5 of the reference article. */
    Qreg A(36);                        // A is simple
    Qreg B(30);                        // B is simple
    Qreg C = B(0,6).swap() &           // a complicated expr. for C
      (Qreg(A) -= 12).swap();          // unnecessary use of -=
    Qreg D = B(12,18) & Qreg(18);      // create D more simply
    Qreg E = C(0, 24) & D & Qreg(6);   // E is disconnected
    /* Show the address manager status and the register content. */
    std::cout << "============================\n"
	      << get_quantum_interface()
	      << "\nRegister A: " << A
	      << "\nRegister B: " << B
	      << "\nRegister C: " << C
	      << "\nRegister D: " << D
	      << "\nRegister E: " << E
	      << "\n============================\n";
    /* This example reproduces the quantum circuit for summing
       three numbers, sec.4.1, pag.12 in the reference article. */
    int size = 4;
    /* prepare the sequence of conditional rotations */
    Qop rot;
    for (int i=0; i<size; ++i)
      rot << QCondPhase(size-i, i+1).offset(i);
    /* prepare a translated Fourier transform */
    Qop tras = (QFourier(size) & QSwap(size)).offset(size);
    /* compose the two argument adder */
    Qop add = tras & rot & ! tras;
    /* offset the 2-adder down, split another 2-adder,
       compose the two 2-adders into one 3-adder */
    Qop add3 = (add >> size) & add(size, size, Qop::SPLIT);
    /* Show the quantum operator structure. */
    std::cout << "============================\n"
	      << add3 << std::endl
	      << "============================\n";
    /* "Run" the computation on first 12 qubits of the E register. */
    add3(E(18,12));
  } catch_qexception;
 end:
  return 0;
}
