/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */

#include <cstdlib>                          // C standard library
#include <unistd.h>                         // POSIX symbolic constants
#include <fcntl.h>                          // File control operations
#include <string.h>                         // C handling of memory areas
#include <limits.h>                         // for CHAR_BIT
#include <iostream>                         // C++ I/O handling
#include <qsimulator.h>                     // access to the simulator

/* *********************************************************************
   | For the time being, the simulator is a global object. It is in-   |
   | stantiated in this file and declared as an external object in the |
   | simulator header file, so that each file including the class de-  |
   | claration will have access to the actual simulator too.           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
Qsimulator simulator;

/* *********************************************************************
   | This private inline static method returns the squared norm of the |
   | complex argument (i.e. the square of the real part plus the square|
   | of the imaginary one). The C++ header <complex> (at the time of   |
   | writing) supplies a norm() function which, despite the name, cal- |
   | culates the same quantity; this implementation is however subopti-|
   | mal, since it involves the calculation of the norm first (which   |
   | needs a sqrt) so I preferred to write my own (squared) norm.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
inline Qsimulator::rnumber Qsimulator::norm2(const cnumber &z) {
  rnumber R = z.real();
  rnumber I = z.imag();
  return (R*R + I*I);
}

/* *********************************************************************
   | The following inline functions return a new random number from    |
   | the random stream. myrandom returns a value homogeneously distri- |
   | buted in [0, 1[ while myrandom2 casts it in [-0.5, 0.5[. Both     |
   | functions currently use the C library rand(), and they must be    |
   | the only way to request a random number in this simulator.        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
inline Qsimulator::rnumber myrandom(void)  { return (rand()/(RAND_MAX + 1.)); }
inline Qsimulator::rnumber myrandom2(void) { return (myrandom() - 0.5); }

/* *********************************************************************
   | This method initialises the random number generator for the       |
   | simulator; currently, random numbers are used for simulating the  |
   | results of measurements, initialisations and the effects of noise.|
   | There is only one generator though, so that the random stream is  |
   | shared. This method reads a new random seed from the /dev/urandom |
   | device; I don't know how much this is portable.                   |
   | ----------------------------------------------------------------- |
   | (09Jan03) S.Bettelli, add the possibility of setting the random   |
   |   seed to a given value. The chosen seed is also returned as the  |
   |   output of the function. If the input value is zero (which is    |
   |   the default) the behaviour is the same as before.               |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
unsigned long Qsimulator::init_random_seed(unsigned long seed) {
  /* open the urandom device and read the correct amount of bytes, which
     are then composed and interpreted as a seed. Then, close the device.
     Don't do this if "seed" is not zero! */
  if (seed == 0) {
    int fd = open("/dev/urandom", O_RDONLY);
    read(fd, &seed, sizeof(unsigned long));
    close(fd);
  }
  /* set the seed for a new sequence of pseudo-random integers to be
     returned by rand(), which is a global random number generator from
     the C library. We could think about making it a private object. */
  srand(seed);
  /* return the value of the chosen seed. */
  return seed;
}

/* *********************************************************************
   | This method sets the number of qubits managed by the simulator.   |
   | All the pre-existing data structures are cleared and new memory   |
   | is allocated. The managed classical memory is then initialised so |
   | that it corresponds to the quantum state |0> (the memory element  |
   | which corresponds to |0> is the first (since it is indexed by the |
   | bit string 000..0)                                                |
   | ----------------------------------------------------------------- |
   | (07/Dec/2002) S.Bettelli. Since the amount of allocated space can |
   | easily be enormous, it is mandatory to check wheather the alloca- |
   | tion was successful or not. Also, we must check for an overflow   |
   | in memsize (i.e. that it can hold the number 2^num_qubits).       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      08 Dec 2002         |
   ********************************************************************* */
void Qsimulator::set_numqubits(size_type num_qubits) {
  /* the number of bits in the type for memsize. */
  const unsigned long maxsize = sizeof(memsize)*CHAR_BIT;
  /* check that the amount of requested classical memory can be at least
     written in memsize. num_qubits must be strictly less than memsize. */
  if (num_qubits >= maxsize) {
    std::cerr << "The requested number of qubits (" << num_qubits
	      << ") cannot be managed by this simulator\nsince "
	      << "the integer type we are using contains only "
	      << maxsize << " bits. The maximum number of qubits\n"
	      << "is therefore " << (maxsize-1) << ". Aborting ...\n";
    abort(); }
  /* release any previously allocated memory area */
  if (size()) delete [] data;
  /* calculate the amount of memory we need (2^num_qubits). */
  memsize = (0x1 << num_qubits);
  /* allocate a new big array for storing all the 2^num_qubits amplitudes.
     Since this number can be enormous, be sure that the allocation was
     succesful, and print a meaningful error message if not. */
  try { data = new cnumber[memsize]; } catch(...) {
    std::cerr << "The allocation of " << (memsize*sizeof(cnumber))
	      << " memory elements was not succesful\nProbably, there"
	      << " is not enough available memory, aborting ...\n";
    abort(); }
  /* reset all amplitudes to zero, exception made for the first, which
     is set to 1. This corresponds to the initialisation to |00..00> */
  for (cnumber *i = data; i != data + size(); ++i) *i = 0.0;
  *data = 1.0;
}

/* *********************************************************************
   | This method returns the number of qubits managed by the simulator.|
   | It is actually the base-2 logarithm of the number of coefficients |
   | stored in the classical memory. Since by construction this number |
   | is a power of two, we can get it with shifts and additions.       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
Qsimulator::size_type Qsimulator::qubits(void) const {
  size_type power = 0;
  size_type sizeval = size();
  while (sizeval /= 2) ++power;
  return power;
}

/* *********************************************************************
   | This method returns the probability that the target-th qubit is   |
   | set (i.e. it is in the state |1>). This probability is the sum    |
   | of the probabilities corresponding to those basis states where    |
   | the target-th bit is 1. It is of course assumed that the repre-   |
   | sentation of the quantum state is normalised.                     |
   | ----------------------------------------------------------------- |
   | The algorithm for finding the relevant elements in the classical  |
   | memory is a bit cryptic. It however uses only assignements, in-   |
   | crements and dereferentations, so it should be very fast.         |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
Qsimulator::rnumber Qsimulator::is_set(address target) const {
  /* the probability to be returned at the end. */
  rnumber prob_is_set = 0.0;
  /* "block" is the number of consecutive integers, starting from zero,
     which do not have the target-th bit set. We are interested in those
     memory elements indexed by integers falling in odd blocks. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (const cnumber *i = data; i != data + size(); i+= block)
    /* "j" loops over all the integers in a relevant block. */
    for (const cnumber *j = i += block; j != i + block; ++j)
      /* get the contribution from this memory element. */
      prob_is_set += norm2(*j);
  /* return the calculated probability of being set. */
  return prob_is_set;
}

/* *********************************************************************
   | This method returns the amplitude of the index-th basis element   |
   | in the quantum memory. Beware that the quantum memory does not    |
   | know about any register, so "index" is an absolute index into the |
   | quantum memory as a whole.                                        |
   | ----------------------------------------------------------------- |
   | This method is not trivial since the bit order is reversed in the |
   | simulator (i.e. the most significant bit is on the right). For    |
   | instance, index=1 corresponds to the (2^(num_qubits-1))-th memory |
   | element. Therefore, a reversal of the bit order is necessary.     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      09 Jan 2003         |
   ********************************************************************* */
Qsimulator::cnumber Qsimulator::amplitude(size_type index) const {
  /* "reversed_index" will be the obviuous thing. "mem_mask"
     will act as a counter which gets spilled bit by bit. */
  size_type reversed_index = 0, mem_mask = (memsize >> 0x1);
  /* continue till the counter is non-zero. */
  while (mem_mask) {
    /* capture the rightmost bit of "index" */
    size_type test_last = index & 0x1;
    /* shift "index" right, thus loosing the rightmost bit. */
    index >>= 0x1;
    /* push the saved bit into "reversed_index", from the right. */
    (reversed_index <<= 0x1) ^= test_last;
    /* decrease the counter by shifting it one bit on the right. */
    mem_mask >>= 0x1; }
  /* return the memory element corresponding to "reversed_index". */
  return (*this)[reversed_index];
}

/* *********************************************************************
   | This method returns the squared amplitude of the index-th basis   |
   | element in the quantum memory (i.e. the probability to be measu-  |
   | red during a computational basis measurement). Have a look at the |
   | "amplitude" method for other details.                             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 Jan 2003         |
   ********************************************************************* */
Qsimulator::rnumber Qsimulator::probability(size_type index) const {
  return norm2(amplitude(index)); // should it be inline?
}

/* *********************************************************************
   | This method implements the collapse of the quantum state of the   |
   | simulated system to the state |value> for the target-th qubit.    |
   | The action of the projector |value><value| for the target-th      |
   | qubit tensor the identity for the rest of the system consists in  |
   | setting to zero all the memory elements which correspond to the   |
   | "wrong" projection and in renormalising the others. The last ar-  |
   | gument is the "probability" that the collapse takes place, which  |
   | must be known in advance.                                         |
   | ----------------------------------------------------------------- |
   | Be aware that if the current quantum state lives in the zero      |
   | eigenvalue eigenspace of the projector, the state vector after    |
   | the projection is the null vector, and the renormalisation is     |
   | not possible! This method should therefore remain private and be  |
   | called only by carefully crafted routines. The renormalisation    |
   | coefficient is calculated from "probability".                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Nov 2002         |
   ********************************************************************* */
void Qsimulator::do_collapse(address target, bool value, rnumber probability) {
  /* the unnormalised state after the collapse has a norm equal to
     the square root of the probability of the given outcome. Prepare
     a renormalisation coefficient. */
  rnumber renormalisation = 1.0 / sqrt(probability);
  /* prepare two factors for later use. renorm_zero is the appropriate
     factor for those memory indexes with the target-th bit set to zero:
     the corresponding elements must be zeroed or renormalised depending
     on "value" being true or false. renorm_one is the "reverse" ... */
  rnumber renorm_zero = (value ? 0.0 : renormalisation);
  rnumber renorm_one  = (value ? renormalisation : 0.0);
  /* "block" is the size of a block of consecutive integers
     which have the same value for the target-th bit. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (cnumber *i = data; i != data + size(); i += block) {
    /* loop over all indexes with the target-th bit set to zero. */
    for (cnumber *j = i         ; j != i + block; ++j) *j *= renorm_zero;
    /* loop over all indexes with the target-th bit set to one. */
    for (cnumber *j = i += block; j != i + block; ++j) *j *= renorm_one;
  }
}

/* *********************************************************************
   | This method implements the measurement of a qubit. In practice,   |
   | it calculates the probability that the target-th qubit is set or  |
   | unset, and then generates a random result according to this       |
   | probability. If "collapse" is set, the memory elements are then   |
   | modified and renormalised just as if a collapse of the state of   |
   | the quantum system took place. The return value is the outcome    |
   | of the fake measurement.                                          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
bool Qsimulator::do_measurement(address target, bool collapse) {
  /* the result to be returned at the end. */
  bool result;
  /* calculate the probability that the target-th qubit is set. */
  rnumber prob_is_set = is_set(target);
  /* get a random number in [0,1[. If this number is smaller than the 
     probability, assign "true" to the result, otherwise assign "false". */
  if (myrandom() < prob_is_set) result = true; else result = false;
  /* if "collapse" is true, project according to the measurement result.
     The probability for the projection is the probability that the qubit
     is set or unset depending on the result. */
  if (collapse)
    do_collapse(target, result, (result ? prob_is_set : (1.0 - prob_is_set)));
  /* return the measured value */
  return result;
}

/* *********************************************************************
   | This method implements the initialisation of the target-th qubit  |
   | to "value" (which can be |0> or |1>). This must correspond to the |
   | partial measurement (target-th qubit) of the whole quantum system |
   | possibly followed by a correction (like a "not" gate) of that     |
   | qubit if the outcome is "wrong". The state remains normalised.    |
   | ----------------------------------------------------------------- |
   | (08/Dec/2002) S.Bettelli. Don't allocate the temporary buffer on  |
   | the stack (this may cause segmentation fault errors difficult to  |
   | track down). Use dynamical allocation instead, and check explici- |
   | tely that it was succesful.                                       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      08 Dec 2002         |
   ********************************************************************* */
void Qsimulator::init(address target, bool value) {
  /* measure the target-th qubit and save the result */
  bool result = do_measurement(target, true);
  /* if the result is not what we want, we must add a correction. This
     consists in interchanging the coefficient of |x0y> and |x1y>, where
     the 0 and the 1 are in the target-th position. */
  if (result != value) {
    /* "block" is the size of a block of consecutive integers
       which have the same value for the target-th bit. */
    const size_type block = 0x1 << target;
    /* calculate the block size (in bytes) and allocate a temporary
       work space able to contain this amount of memory. Since this
       number can be enormous, be sure that the allocation was
       succesful, and print a meaningful error message if not. */
    size_type blocksize = sizeof(cnumber)*block;
    cnumber *temp = NULL;
    try { temp = new cnumber[blocksize]; } catch(...) {
      std::cerr << "The allocation of " << (blocksize*sizeof(cnumber))
		<< " memory elements was not succesful\nProbably, there"
		<< " is not enough available memory, aborting ...\n";
      abort(); }
    /* "i" loops over all beginnings of double blocks. */
    for (cnumber *i = data; i != data + size(); i += 2*block) {
      /* swap [i,i+block[ and [i+block,i+2*block[. This corresponds to
	 interchanging the coefficients of |x0y> and |x1y> with fixed x.
	 Use the C library memcpy which should be the fastest available. */
      memcpy(temp      , i         , blocksize);
      memcpy(i         , i + block , blocksize);
      memcpy(i + block , temp      , blocksize);
    }
    /* release the temporary workspace. */
    delete [] temp;
  }
}

/* *********************************************************************
   | This routine prepares the single-qubit matrix for implementing a  |
   | Hadamard rotation. The perfect Hadamard matrix is {{1,1},{1,-1}}  |
   | times 1/sqrt(2), that is a rotation of an angle PI around the     |
   | axis {1,0,1} times 1/sqrt(2) on the Bloch sphere. If the noise    |
   | level is not zero, the axis is perturbed by a kick uniformly      |
   | distributed in a solid angle with theta = [-noise/2, noise/2[.    |
   | ----------------------------------------------------------------- |
   | It turns out that only two matrix elements are independent in the |
   | perturbed matrix {{m00,m01},{m10,m11}}, that is m00 and m01,      |
   | since m11 = - m00 and m10 = conj(m01). For sake of efficiency,    |
   | this routine only calculates and returns m00 and m01.             |
   | ----------------------------------------------------------------- |
   | Reference: Toulouse's group error model ---> see ???              |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Nov 2002         |
   ********************************************************************* */
inline void Qsimulator::prepare_hadamard(cnumber &m00, cnumber &m01) {
  /* in presence of noise, we adopt an error model which perturbs the
     rotation axis, not the rotation angle. First, the axis is rotated
     in the X-Z plane of an angle A in [-noise/2, noise/2[; then, the
     resulting axis is rotated around the original axis of a random
     angle in [0,2*PI[. The following formulas corresponds to the 
     overall transformation, although they are a bit cryptic in order
     to gain efficiency. */
  rnumber A = noise * myrandom2();       // random in [-noise/2, noise/2[
  rnumber B = 2.0 * M_PI * myrandom();   // random in [0, 2*PI[
  rnumber sina = sin(A), cosa = cos(A);
  rnumber sinacosb = sina * cos(B);
  m00 = cnumber(M_SQRT1_2 * ( cosa - sinacosb ), 0.0        );
  m01 = cnumber(M_SQRT1_2 * ( cosa + sinacosb ), sina * sin(B));
}

/* *********************************************************************
   | This method modifies the internal data structures according to    |
   | a Hadamard gate executed on the target-th qubit. If noise is      |
   | non-zero, the gate is perturbed.                                  |
   | ----------------------------------------------------------------- |
   | The details of the gate perturbation are managed by the method    |
   | "prepare_hadamard". The iteration through the memory elements is  |
   | similar to that for is_set().                                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
void Qsimulator::hadamard(address target) {
  /* some variables for later use. m00 and m01 are the independent
     matrix elements of the perturbed Hadamard. */
  cnumber m00, m01, mim00, f, fc;
  /* this boolean variable is true if the gate is to be perturbed. */
  const bool is_noisy = (noise != 0.0);
  /* If the noise level is not zero, we must calculate some
     quantities before entering the main loop. */
  if (is_noisy) {
    prepare_hadamard(m00, m01);  // get the independent matrix elements
    f = m01 / m00;               // f is the ratio of these two elements
    fc = conj(f);                // fc is the conjugate of f
    mim00 = - 1. / m00; }        // m(inus) (the) i(nverse) (of) m00
  /* "block" is the size of a block of consecutive integers
     which have the same value for the target-th bit. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (cnumber *i = data; i != data + size(); i += 2*block)
    /* loop over all indexes with the target-th bit set to zero. */
    for (cnumber *j = i; j != i + block; ++j) { 
      /* the Hadamard matrix is going to mix the coefficients of |x0y>
	 and |x1y>, where the 0 and the 1 are in the target-th position.
	 In the following, the coefficient "A" of |x0y> is *j and the
	 coefficient "B" of |x1y> is *k. */
      cnumber *k = j;
      k += block;
      /* General (noisy) case: mixing of "A" and "B" using the matrix
	 {{m00,m01},{m01^*,-m00}}. The following procedure is the fastest
	 I was able to devise (it uses the minimum number of temporaries,
	 and it does not involve subractions or division). */
      if (is_noisy) {
	*j += *k * f;      // *j == A + B*m01/m00     *k == B
	*j *= m00;         // *j == A*m00 + B*m01     *k == B
	*k *= mim00;       // *j == A*m00 + B*m01     *k == - B/m00
	*k += *j * fc;     // *j == A*m00 + B*m01     *k == A*m01^* - B*m00
      } 
      /* In the particular case of a perfect Hadamard matrix the procedure
	 is even simpler, since all the multiplicative coefficients are
	 constants (and sometimes they are zero). We can use the constant
	 M_SQRT1_2, defined in the C math library, which is 1/sqrt(2), as
	 well as M_SQRT2 which is sqrt(2). */
      else {
	*j += *k;          // *j == A + B             *k == B
	*j *= + M_SQRT1_2; // *j == (A + B)/sqrt(2)   *k == B
	*k *= - M_SQRT2;   // *j == (A + B)/sqrt(2)   *k == - B*sqrt(2)
	*k += *j;          // *j == (A + B)/sqrt(2)   *k == (A - B)/sqrt(2)
      }
    }
}

/* *********************************************************************
   | This routine accepts a real number "fraction" specifying a frac-  |
   | tion of an angle in [0, 2*PI], and prepares the phase for a phase |
   | (or conditional phase) rotation. This phase is set to the value   |
   | of exp(2*PI*i*fraction). If the noise level isn't zero, the angle |
   | is perturbed by a kick uniformly distrib. in [-noise/2, noise/2[. |
   | ----------------------------------------------------------------- |
   | Reference: Toulouse's group error model ---> see ???              |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
inline Qsimulator::cnumber Qsimulator::prepare_phase(rnumber fraction) {
  /* convert the "fraction" into an angle in [0,2PI]. M_PI is
     the greek pi constant defined in the C math library. */
  rnumber angle = 2.0 * M_PI * fraction;
  /* if the noise level is not zero, add a perturbation to the
     angle, homogeneously distributed in [-noise/2, noise/2[. */
  if (noise != 0.0) angle += noise * myrandom2();
  /* the phase to return is exp(imaginary unit * angle). */
  return cnumber(cos(angle), sin(angle));
}

/* *********************************************************************
   | This method modifies the internal data structures according to a  |
   | phase gate executed on the target-th qubit with param. "fraction".|
   | If noise is non-zero, the gate is perturbed.                      |
   | ----------------------------------------------------------------- |
   | A perturbed phase gate is a phase gate with a perturbed phase     |
   | angle. The details of this perturbation are managed by the method |
   | "prepare_phase". The iteration through the memory elements is the |
   | same as for is_set(), since the concerned elements are the same.  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
void Qsimulator::phase(address target, rnumber fraction) {
  /* get the phase parameter + noise */
  cnumber phase = prepare_phase(fraction);
  /* "block" is the size of a block of consecutive integers
     which have the same value for the target-th bit. */
  const size_type block = 0x1 << target;
  /* "i" loops over all beginnings of double blocks. */
  for (cnumber *i = data; i != data + size(); i += block)
    /* "j" loops over all the integers with the target-th bit set. */
    for (cnumber *j = i += block; j != i + block; ++j)
      /* apply the phase shift. */
      *j *= phase;
}

/* *********************************************************************
   | This method modifies the internal data structures according to a  |
   | phase gate executed on the target-th qubit conditioned by the     |
   | state of the control-th qubit, with parameter "fraction".         |
   | If noise is non-zero, the gate is perturbed.                      |
   | ----------------------------------------------------------------- |
   | A perturbed conditional phase gate is a conditional phase gate    |
   | with a perturbed phase angle. The details of this perturbation    |
   | are managed by the method "prepare_phase". The iteration through  |
   | the memory elements is a variation of the algorithm used for the  |
   | phase() method (indeed, a double loop).                           |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
void Qsimulator::condphase(address control, address target, rnumber fraction) {
  /* get the phase parameter + noise */
  cnumber phase = prepare_phase(fraction);
  /* "block_t" and "block_c" are the sizes of blocks of consecutive
     integers which have the same value for the target-th or the 
     control-th bit. We reason as if control was less than target
     (and take the appropriate countermeasure if this is not true). */
  bool ok = control < target;
  const size_type block_t = (ok ? (0x1 << target ) : (0x1 << control));
  const size_type block_c = (ok ? (0x1 << control) : (0x1 << target ));
  /* "i" loops over all beginnings of double block_t blocks. */
  for (cnumber *i = data; i != data + size(); i += block_t)
    /* "j" loops over all beginnings of double block_c blocks. */
    for (cnumber *j = i += block_t; j != i + block_t; j += block_c)
      /* "k" loops over all the integers with the target-th and
	 the control-th bit set. */
      for (cnumber *k = j += block_c; k != j + block_c; ++k)
	/* apply the phase shift. */
	*k *= phase;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
