/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_EXCEPTIONS_HANDLING
#define __QUANTUM_EXCEPTIONS_HANDLING

#include <string>                          // for C++ strings
#include <exception>                       // include standard exception

/* *********************************************************************
   | This class will be used for all exceptions thrown by the quantum  |
   | library, so that catching this base class will catch a generic    |
   | problem in the run-time quantum code generation. It inherits from |
   | the standard exception base class and for the C++ string class,   |
   | for displaying a more informative message (for debug).            |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 09 Feb 2001         |
   ********************************************************************* */
class Qexception : public std::exception, public std::string {
public:
  /* constructor (stores the string internally) */
  Qexception(const char *new_message) : std::string(new_message) {}
  /* virtual destructor for the exception class */
  virtual ~Qexception() throw () { }
  /* this method shows an informative message about the exception */
  const char* what(void) const throw () { return std::string::c_str(); }
};

/* *********************************************************************
   | This define is helpful in generating exceptions for the quantum   |
   | classes. The input string will be both the name of the class and  |
   | the message contained in the base class quantum_exception.        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 09 Feb 2001         |
   ********************************************************************* */
#define _Qexception(name) class name : public Qexception { \
                          public:                          \
                            name() : Qexception(#name) {}  \
			    virtual ~name() throw () { }   \
                          }

/* *********************************************************************
   | Another macro to speed up the debug cycle. It is the catch block  |
   | for catching a generic quantum exception.                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 10 Feb 2001         |
   ********************************************************************* */
#define catch_qexception catch (Qexception ex) \
  { std::cerr << "Caught quantum exception <"  \
              << ex.what() << ">" << std::endl; }

#endif  // __QUANTUM_EXCEPTIONS_HANDLING
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
