/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __COMMUNICATION_WITH_QUANTUM_DEVICE_DRIVER_
#define __COMMUNICATION_WITH_QUANTUM_DEVICE_DRIVER_

#include <queue>                                // for STL queues
#include "qinterface_base.h"                    // base class definition
#ifdef QINTERFACE_USES_PTHREADS
#include <pthread.h>                            // for POSIX threads
#endif

/* *********************************************************************
   | This class specialises the properties for a Qinterface_base, that |
   | is an interface between the language classes and the quantum      |
   | device driver. See the base class header for further comments.    |
   | ----------------------------------------------------------------- |
   | Address manager:                                                  |
   |   objects from this class initialise the Qaddress_manager_base    |
   |   pointer with a dinamically allocated address manager of some    |
   |   particular type (the details are hidden in the implementation   |
   |   file) which is accessed only through its interface.             |
   | Output of bytecode:                                               |
   |   objects from this class have a FIFO queue which holds the       |
   |   bytecode toward the quantum device driver.                      |
   | Feedback from the quantum device:                                 |
   |   objects from this class have a FIFO queue which holds the       |
   |   results of measurements and other feedbacks when they are sent  |
   |   back by the quantum device.                                     | 
   | ----------------------------------------------------------------- |
   | Threaded simulation: (15 Jul 2002, S.Bettelli)                    |
   |   if the QINTERFACE_USES_PTHREADS symbol is defined at compile    |
   |   time, a POSIX thread is started which consumes the bytecode     |
   |   FIFO (just to be a bit realistic); otherwise the routine which  |
   |   decodes and dispatches the bytecode is called by the channel_*  |
   |   methods each time (serialisation). If threads are on, the       |
   |   FIFO queues are protected during reads and writes (mutex).      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 19 Jul 2001         |
   ********************************************************************* */
class Qinterface : public Qinterface_base {
public:
  // constructor for a Qinterface
  Qinterface();
  // destructor (virtual)
  virtual ~Qinterface();
private:
  // no copy constructor, please!
  Qinterface(const Qinterface &an_interface);
  // no assignement operator, please!
  Qinterface &operator=(const Qinterface &an_interface);
private:
  // how to send bytecode to the quantum device
  void write_bytecode(const message_type &message);
  // how to write data to the feedback buffer
  void write_feedback(const message_type &message);
  // this method blocks till there is data in the feedback channel
  void poll_feedback(void);
private:
  // this method is an example of quantum "simulator"
  void run_evolution(void);
  // this method returns the value of the last measurement (blocking)
  Qbitset    get_measurement_result(Qbitset::size_type bits);
  // this method returns the result of the last locate command (blocking)
  Qubit_list get_locate_result(Qubit_list::size_type size);
#ifdef QINTERFACE_USES_PTHREADS
  // this is the thread object for the consumer thread
  pthread_t consumer_thread;
  // this is are mutual exclusion devices for our thread
  pthread_mutex_t command_mutex, feedback_mutex;
  // if this variable is false, the consumer thread should stop
  volatile bool run_consumer;
  // this is the start routine for the consumer thread
  static void *consumer(void *interface);
  // this sets the command channel to the busy state
  void channel_busy(void) { pthread_mutex_lock(&command_mutex); }
  // this sets the command channel to the free state
  void channel_free(void) { pthread_mutex_unlock(&command_mutex); }
  // this sets the feedback channel to the busy state
  void channel_feed_busy(void) { pthread_mutex_lock(&feedback_mutex); }
  // this sets the feedback channel to the free state
  void channel_feed_free(void) { pthread_mutex_unlock(&feedback_mutex); }
#else
  // this consumes the bytecode for the serialised simulator
  void channel_filled(void) { run_evolution(); }
  // this consumes the bytecode for the serialised simulator (blocking)
  void channel_blocking(void) { run_evolution(); }
#endif
private:
  // typedef the queue type for holding buffers
  typedef std::deque<message_type> command_queue_type;
  // the bytecode buffer for this interface
  command_queue_type command_queue;
  // the feedback buffer for this interface
  command_queue_type feedback_queue;
  // how to read data from a generic queue (popping it)
  static message_type read_queue(command_queue_type &a_queue);
};

/* *********************************************************************
   | Sending a command to the quantum device in this class is very     |
   | simple. We simply have to write the value at the end of the queue.|
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      18 Jul 2002         |
   ********************************************************************* */
inline void Qinterface::write_bytecode(const message_type &message) {
  command_queue.push_back(message);
}

/* *********************************************************************
   | Sending data to the feedback buffer in this class is very simple. |
   | We only need to write the value at the end of the queue.          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   ********************************************************************* */
inline void Qinterface::write_feedback(const message_type &message) {
  feedback_queue.push_back(message);
}

/* *********************************************************************
   | Commands and data are read out from the beginning of queues. They |
   | are subsequently popped out. This static function is fit for a    |
   | routine which consumes the bytecode, like a simulator, or decodes |
   | a measurement result, and is here just as a facility for testing  |
   | new implementations.                                              |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      19 Jul 2002         |
   ********************************************************************* */
inline Qinterface::message_type 
Qinterface::read_queue(command_queue_type &a_queue) {
  message_type command = a_queue.front();
  a_queue.pop_front();
  return command;
}

#endif  // __COMMUNICATION_WITH_QUANTUM_DEVICE_DRIVER_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
