/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <cmath>                            // standard C math library
#include "quantum_library.h"                // function prototypes
#include "qntlib_expower.h"                 // auxiliary function protos
#include "qprimitives.h"                    // high level quantum primitives

/* *********************************************************************
   | This function returns a circuit to be run on a quantum register   |
   | with q="qubits" qubits. When acting on a basis element |n>, where |
   | n is a binary string with q bits, i.e. an integer number in the   |
   | range [0, 2^q[, it transforms |n> into exp(2 PI i f n^p) |n>,     |
   | where f="factor" is a floating point number and p="power" is an   |
   | integer number; it is thus a diagonal operator. If "qubits" is    |
   | zero or "power" is zero or "factor" is 0.0, this function returns |
   | the identity operator. The circuit is built in such a way that    |
   | the first bit in each register passed to the circuit will be con- |
   | sidered, as usual, the most significant bit (MSB).                | 
   | ----------------------------------------------------------------- |
   | This function is indeed only an interface. All the calculations   |
   | and numerical checks are performed by an auxiliary class, which   |
   | is not available to the language user.                            |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      04 May 2002         |
   ********************************************************************* */
Qop expower(::qubit_address_type qubits, unsigned int power, double factor) {
  return gate_calculator(qubits, power, factor).get_Qop();
}

/* *********************************************************************
   | This is the constructor of the auxiliary class which fills in the |
   | data structures which describe the quantum gates to be fed into   |
   | the Qop returned by the "expower" function. The circuit is built  |
   | in such a way that the first bit in each register passed to the   |
   | circuit will be considered the most significant bit (MSB).        |
   | ----------------------------------------------------------------- |
   | First, some tests are run to check that the inputs are meaningful |
   | and that they are not too big. Then the factorials of all the     |
   | integer numbers up to "the_power" are calculated and stored in an |
   | array. Then a loop is executed on the arity of multicontrolled    |
   | gates, which performs the real calculations (the bulk of these    |
   | calculations is encapsulated in two other private methods.)       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Mar 2003         |
   ********************************************************************* */
gate_calculator::gate_calculator(address_type the_qubits,
				 power_type   the_power,
				 factor_type  the_factor) : 
  /* save the inputs and put 0! in the array for factorials. */
  qubits(the_qubits), power(the_power),
  factor(the_factor), factorials(1, 1) {
  /* initialize the data structure which will contain the description of
     the gates that are to be built. Well, in this case it's simply empty. */
  gates.clear();
  /* get rid of unmeaningful inputs. If "qubits" is zero, we don't have
     a register for the Qop, so it can as well be the identity. If "factor"
     or "power" are zero, we are multiplying each basis vector by a const
     phase, which is irrelevant in the context of quantum computing, and
     the identity Qop is perfect too. In all cases, the internal data
     structures are not filled, which will lead to the identity Qop. */
  if (qubits == 0 || power == 0 || factor == 0.0) return;
  /* calculate the maximum number of different indexes. It cannot be larger
     than the number of indexes (i.e. power), nor, of course, than the number
     of possible values for the indexes (i.e. the number of qubits). */
  number max_different_indexes = (qubits < power) ? qubits : power;
  /* precalculate the factorials of all the integers up to the power we are
     dealing with. None of these calculations is redundant, so we don't waste
     any CPU cycle. The bignumber data type is now a multiprecision integer,
     i.e. we don't have to worry about how large the integer becomes, because
     it is always representable (unless we fill up all the RAM ...). */
  for (number i = 1; i <= power; ++i)
    factorials.push_back(factorials.back() * i);
  /* now run over all possible values for the number of different indexes;
     the minimum value is 1, the maximum value was calculated before. */
  for (number i = 1; i <= max_different_indexes; ++i) {
    /* clear this data structure which will contain the possible index
       partitions and their multiplicity for "i" different indexes. */
    partitions.clear();
    /* load the "index_partitions" data structure, then use it in order
       to calculate the gate indexes and their phase. */
    calculate_partitions(i, power);
    calculate_gates(i, qubits);
  }
}

/* *********************************************************************
   | This method calculates the number of partions of "p" indexes into |
   | "k" groups (i.e. all possible ways to write p = p_1 + ... + p_k   |
   | with p_i positive integer numbers) and the number of their permu- |
   | tations "n" (which is p!/(p_1!p_2!...p_k!) ). For each partition, |
   | a record containg the partition and its multiplicity (the number  |
   | of its permutations) is saved in the "index_partitions" structure.|
   | ----------------------------------------------------------------- |
   | This function works recursively by choosing, one at a time, all   |
   | the possible values v for a group and calling itself with one     |
   | group less and v indexes less. The "multiplicity" variable is     |
   | updated during the computation in order to hold the number of     |
   | permutations for a partition.                                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2002         |
   ********************************************************************* */
void gate_calculator::calculate_partitions(number k, power_type p) {
  /* This vector of integers will be used by all the recursive calls
     for keeping the status of the current partition. It is initially
     set to the empty vector, and it must be left in this state by a
     complete tree of calls (i.e. when the root call returns). */
  static partition_type current_partition;
  /* The integer "multiplicity" will be used as well for keeping the
     number of permutations of the current partition. Since it is updated
     by multiplication or division, initialise it to one. It can grow
     to a very large value, so declare it as "bignumber". */
  static bignumber multiplicity = 1;
  /* If this is a leaf call (i.e. it does not call the method itself again
     since the current partition is complete), save the description of the
     current partition (the vector (p_1, ..., p_k) and its multiplicity. */
  if (k == 0)
    partitions.push_back(make_pair(current_partition, multiplicity));
  else
    /* in the general case, loop over all possible values for a partition
       element. If there is only one group left, it must take all the
       indexes (i.e. v = p); otherwise, v can grow from 1 up to p-k+1
       (the maximum value is fixed by the condition that every other
       group must contain at least one index). */
    for (power_type v = (k==1 ? p : 1); v <= (p-k+1); ++v) {
      /* precalculate the value of the binomial coefficient (p v) */
      bignumber binomial = ((factorials[p]/factorials[v])/factorials[p-v]);
      /* for each value v, push it onto the current partition, then update
	 the value of multiplicity with the binomial coefficient (p v);
	 since p is updated too in the recursion tree, this will leave
	 multiplicity = p! (p_1!...p_k!) at the end. */
      current_partition.push_back(v);
      multiplicity *= binomial;
      /* descend the recursion tree */
      calculate_partitions(k-1, p-v);
      /* pop v from the current partition and restore the multiplicity. */
      current_partition.pop_back();
      multiplicity /= binomial;
    }
}

/* *********************************************************************
   | This method calculates all the ways to choose k qubits out of a   |
   | total of q qubits without taking their order into account; their  |
   | number is obviously the binomial coefficient (q k). Then, each    |
   | combination of selected qubit positions is combined with all the  |
   | partitions in the index_partitions data structure to calculate    |
   | a phase exponent, and the pair is saved into "gates".             |
   | The phase exponent is saved only if it is non-trivial (i.e. 0.0)  |
   | ----------------------------------------------------------------- |
   | The qubit selection will determine which qubits the multicontrol- |
   | led phase gate will act on; the phase exponent is 2 to the sum of |
   | the qubit indexes (each one with its multiplicity) for all the    |
   | partitions of k qubits and is calculated by an auxiliary method.  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2002         |
   ********************************************************************* */
void gate_calculator::calculate_gates(number k, address_type q) {
  /* This vector of integers will be used by all the recursive calls
     for keeping the status of the current selection. It is initially
     set to the empty vector, and it must be left in this state by a
     complete tree of calls (i.e. when the root call returns). */
  static selection_type current_selection;
  /* If this is a leaf call (i.e. it does not call the method itself again
     since the current selection is complete), save the description of the
     current selection and the phase exponent (which is calculated by an
     auxiliary method). */
  if (k == 0) {
    factor_type gate_factor = calculate_factor(current_selection);
    /* save the factor only if it is non-trivial! */
    if (gate_factor != 0.0)
      gates.push_back(make_pair(current_selection, gate_factor)); }
  /* in the general case, branch the call tree into two branches, which
     correspond to selecting or not the current qubit index: this is
     calculated as (qubits - q), so that it starts from zero (during
     the root call when q is "qubits") and it increases up to qubits-1.
     This is only a cosmetic trick if you want to print the results for
     debugging, it doesn't change anything in the circuit at the end. */
  else {
    /* The first branch adds the current index to the selection. This is
       always an option when k != 0. Then go on with one qubit less to
       choose (k --> k-1) and one qubit less in the group among which
       one can choose (q --> q-1). Remember to pop the index when this
       branch is complete. */
    current_selection.push_back(qubits - q);
    calculate_gates(k-1, q-1);
    current_selection.pop_back();
    /* The second branch does not put the current index in the selection.
       This is however a choice only if enough qubits (i.e. k) remain in
       the group of available qubits (whose size will decrease to q-1). */
    if (q > k) calculate_gates(k, q-1);
  }
}

/* *********************************************************************
   | This method takes as input a selection of i indexes, and matches  |
   | them against all partitions of the "power" indexes into i groups. |
   | This allows to determine the sum of the "power" indexes for all   |
   | the index combinations which concern the qubits in "a_selection". |
   | For each sum "s", the quantity 2^s is calculated and accumulated  |
   | into a counter which will be associated to the current selection. |
   | ----------------------------------------------------------------- |
   | In practice, not all partitions are necessary, since the index    |
   | sum is invariant under a permutation. Therefore each 2^s is mul-  |
   | tiplied by the multiplicity factor which is the second element of |
   | each record in the index_partitions data structure.               |
   | ----------------------------------------------------------------- |
   | We now use a multiprecision data type for bigintegers, so that we |
   | don't have to worry anymore about integer overflow, during the    |
   | calculation of 2^s and during the accumulation of the 2^s times   |
   | their multiplicity (28 mar 2003). The routine remains however     |
   | numerically critical at the stage of the multiplication of the    |
   | big integer with the floating point coefficient, but this is      |
   | dealt with in the biginteger class itself (this replaces a pre-   |
   | vious tentative to estimate the significancy of the returned      |
   | mantissa in this method, 29 Mar 2003).                            |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Mar 2003         |
   ********************************************************************* */
gate_calculator::factor_type 
gate_calculator::calculate_factor(selection_type a_selection) {
  /* prepare an accumulator for index_sum * multiplicity terms. This 
     will be our most critically big integer number. */
  bignumber accumulator = 0;
  /* loop over all the previously calculated index partitions. */
  for (partition_list::const_iterator a_partition = partitions.begin();
       a_partition != partitions.end(); ++a_partition) {
    /* calculate a sum of qubit indexes by matching the passed index
       selection and the current index partition. This is nothing more
       than an inner product. Note that we want that the first qubit
       of each register passed to this circuit will be considered as
       the most significant qubit (MSB). For this reason the power
       associated to each index j is (qubits-1-j) and not simply j. */
    number index_sum = 0;
    partition_type::const_iterator a_count = a_partition->first.begin();
    for (selection_type::const_iterator an_index = a_selection.begin();
    	 an_index != a_selection.end(); ++an_index, ++a_count)
      index_sum += (*a_count) * (qubits - 1 - *an_index);
    /* get the multiplicity of the current index sum (a simple permuta-
       tion of the partition does not change the index sum). This can
       be a very large number, so use the "bignumber" type. */
    bignumber multiplicity = a_partition->second;  
    /* multiply the partition multiplicity times 2^(index_sum). */
    bignumber partition_weight = multiplicity << index_sum;
    /* accumulate the product of index_sum and multiplicity. */
    accumulator += partition_weight;
  }
  /* now use a method of the bignumber class to return the phase
     gate exponent, which is obviously a floating point in [0, 1[.
     All the numerical problems in dealing with this multiplication
     are hidden by the bignumber method. */
  return accumulator.mantissa_of_multiplication_by(factor);
}  

/* *********************************************************************
   | This method reads the internal data structures and returns the    |
   | corresponding quantum circuit. All one-, two- and three- qubit    |
   | gates are translated with the corresponding high level quantum    |
   | primitives. More complicated gates are built with the constructor |
   | for controlled gates (and will need ancilla qubits).              |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      13 May 2002         |
   ********************************************************************* */
Qop gate_calculator::get_Qop(void) {
  /* prepare an empty quantum operator to be returned at the end. */
  Qop circuit;
  /* this is the main loop on the "gates" data structure. Each record in
     the structure contains a vector of integers selecting the qubits on
     which the gate is to act, and a floating point number which is the
     phase parameter for the (multicontrolled) phase gate. Translate 
     each gate separately. */
  for (gate_list::const_iterator a_gate = gates.begin();
       a_gate != gates.end(); ++a_gate) {
    /* the first part of the record is the vector of qubit addresses.
       Get a reference to it. Also get a reference to the phase factor. */
    const selection_type &a_selection = a_gate->first;
    const factor_type    &a_factor    = a_gate->second;
    /* get the number of addresses inside it: this will be the arity
       of the multicontrolled phase gate. */
    selection_type::size_type arity = a_selection.size();
    /*  prepare a permutation list for remapping the current gate after
	its "base" circuit is ready. This simplifies all the index
	stuff a lot. There is a special list constructor for this. */
    Qubit_list permutation(a_selection);
    /* set up a try-catch pair in order to catch the underflowed angles. */
    try {
      /* For arity equal to one, we have a normal phase gate. For arity
	 equal to two we have a controlled phase gate. In the general
	 case we build a controlled QCondPhase (the number of controls
	 is obviously arity-2). */
      switch (arity) {
      case  1: circuit << QPhase(1,0,a_factor).remap(permutation);     break;
      case  2: circuit << QCondPhase(1,0,a_factor).remap(permutation); break;
      default: circuit << Qop(QCondPhase(1,0,a_factor),
			      arity-2).remap(permutation);             break;
      }
    }
    /* It is difficult to decide what to do in these cases. For
       the time being we will simply discard underflowed gates and
       print a warning message. */
    catch (Qexception ex) { 
      std::cerr << "Routine expower: discarding a phase gate due to "
		<< "exception " << ex << " (a_factor = " << a_factor << ")\n";
    }
  } 
  /* return the operator which was prepared in this method. */
  return circuit;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
