/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_PHASE_LIKE_PRIMITIVES_
#define __QUANTUM_PHASE_LIKE_PRIMITIVES_

#include "qexception.h"               // for "quantum" exceptions
#include "qop_slice.h"                // for basic time slices
#include "qop_codes.h"                // codes for quantum operations
#include "qop_basegates.h"            // base gates for phase-like ops.

/* *********************************************************************
   | This is a utility class for a phase-like operator, that is an     |
   | operator which transforms all the computational basis states into |
   | themselves, multiplying the "last" (that encoded by the string    |
   | 11....1) by e^(2 PI i x/2^k), where x is a floating point value   |
   | and k is an unsigned integer number. Obviously, changing x into   |
   | -x is equivalent to adjoining the operator.                       |
   | ----------------------------------------------------------------- |
   | A single ::quantum_message_type number A is used to remember the  |
   | multiples of an elementary angle; if the number of bits in this   |
   | integer type is n, the correspondence is x/2^k ~ A/2^n mod 2 PI,  |
   | and an approximation can occur since A is integer while x is a    |
   | floating point number. This means that A is essentially a mantis- |
   | sa. The constructors accept the two parameters k and x and calcu- |
   | late the corresponding A, saving it internally.                   |
   | If x/2^k is an integer number, these ctors generate an exception  |
   | of type invalid_angle instead of creating the identity slice.     |
   | Actually, and exception is generated also when the_power is so    |
   | large that the mantissa becomes zero (i.e. when the angle cannot  |
   | be represented in this fixed point representation.)               |
   | ----------------------------------------------------------------- |
   | (26 Sep 2001) S.Bettelli. Continuous version has been taken away. |
   |    Moreover, we explicitely provide support for arity=1,2 opera-  |
   |    tors only, generic arity is gone.                              |
   | (28 Nov 2001) S.Bettelli. k=0 is now an invalid value (it would   |
   |    specify the identity operator, hence it is not useful, and     |
   |    could be source of confusion). Also remember that k=1 and k=-1 |
   |    are equivalent.                                                |
   | (05 Dec 2001) S.Bettelli, the default has_adjoint_basegate() and  |
   |     adjoin() methods are to be overridden, due to the angle parm. |
   | (30 Apr 2002) S.Bettelli. Reintroduce the "continuous phase gate" |
   |    through the additional parameter x, whose sign also encodes    |
   |    the adjointness of the operator. Change the internal represen- |
   |    tation to "quantised angles".                                  |
   | (10 Nov 2002) S.Bettelli. Adapted for the changes in Qop_slice,   |
   |    namely the base gate pointer instead of the opcode in the ctor |
   |    and output_details --> get_details.                            |
   | ----------------------------------------------------------------- |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 12 Jun 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulose,       30 Apr 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulose,       10 Nov 2001         |
   ********************************************************************* */
class Qop_phase_like : public Qop_slice {
public:
  // an unsigned integer type for specifying the power k
  typedef ::quantum_phase_power_type    power_type;
  // a floating point type for specifying the fraction x
  typedef ::quantum_phase_floating_type floating_type;
  // a class for an invalid angle specification (multiple of 2PI)
  _Qexception(invalid_angle);
  // a class for an invalid angle specification (too small)
  _Qexception(angle_underflow);
protected:
  // an unsigned integer type for specifing the angle mantissa
  typedef ::quantum_message_type mantissa_type;
  // Constructor with one preformed list, arity=1, k and optional x
  Qop_phase_like(const Qubit_list &the_list,
		 power_type the_power, floating_type the_fraction);
  // Constructor with two preformed lists, arity=2, k and optional x
  Qop_phase_like(const Qubit_list &the_list, 
		 const Qubit_list &the_targets,
		 power_type the_power, floating_type the_fraction);
  // virtual destructor
  virtual ~Qop_phase_like() { }
public:
  // this method adjoins the time slice
  void adjoin(void);
  // true if the gates associated to the two slices are adjoint
  bool has_adjoint_basegate(const Qop_slice &a_slice) const;
  // this outputs a formatted string describing the phase.
  virtual void get_details(std::ostream &a_stream) const;
  // a method for optional parameter passing (returns the mantissa)
  opcode_type get_parameter(void) const {
    return static_cast<opcode_type>(get_mantissa()); }
  // this method decodes the current mantissa into a pair (k,x)
  void decode_mantissa(power_type &the_power,
		       floating_type &the_fraction) const;
private:
  // the angle mantissa A --> angle = ( 2 PI A / 2^n )
  mantissa_type mantissa;
  // this static method encodes a pair (k,x) into a mantissa A
  static mantissa_type encode_mantissa(power_type the_power,
				       floating_type the_fraction);
  // a method for returning the mantissa
  mantissa_type get_mantissa(void) const { return mantissa; }
};

/* *********************************************************************
   | A constructor for a phase like operator with arity equal to one.  |
   | It accepts a qubit list and the pair (k,x) defining the phase an- |
   | gle. The encoding of (k,x) is delegated to a subroutine.          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         | 
   ********************************************************************* */
inline Qop_phase_like::Qop_phase_like(const Qubit_list &the_list,
				      power_type the_power,
				      floating_type the_fraction) :
  Qop_slice(&BASEGATE_PHASE, the_list),
  mantissa(encode_mantissa(the_power, the_fraction)) { }

/* *********************************************************************
   | A constructor for a phase like operator with arity equal to two.  |
   | It accepts two qubit lists and the pair (k,x) defining the phase  |
   | angle. The encoding of (k,x) is delegated to a subroutine.        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         | 
   ********************************************************************* */
inline Qop_phase_like::Qop_phase_like(const Qubit_list &the_list, 
				      const Qubit_list &the_targets,
				      power_type the_power,
				      floating_type the_fraction) :
  Qop_slice(&BASEGATE_COND_PHASE, the_list, the_targets),
  mantissa(encode_mantissa(the_power, the_fraction)) { }

/* *********************************************************************
   | This method adjoins the current time slice. Since the only inter- |
   | nal field is a mantissa representing the angle, and adjoining the |
   | time slice means changing sign to the angle, this method simply   |
   | complements the mantissa to 2^n (takes the 2's complement).       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         | 
   ********************************************************************* */
inline void Qop_phase_like::adjoin(void) {
  mantissa = ((~ get_mantissa()) + 1);
}

/* *********************************************************************
   | This method returns true if the passed slice and the current      |
   | slice are associated to two basic quantum gates which are one     |
   | the adjoint of the other.                                         |
   | ----------------------------------------------------------------- |
   | Phase like gates are diagonal gates, with only the last element   |
   | being nontrivial. Of course, the underlying gates are adjoint if  |
   | the sum of their mantissas is an integer number. The mantissas    |
   | are accessed through the get_parameters method.                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 05 Dec 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         |
   ********************************************************************* */
inline bool
Qop_phase_like::has_adjoint_basegate(const Qop_slice &a_slice) const {
  /* return immediately if the other slice is not built on top of a
     phase gate of the same type. This can be tested with opcodes. */
  if (get_opcode() != a_slice.get_opcode()) return false;
  /* return true if the two mantissas sum to zero, false otherwise.
     Note that during an integer sum the carry is automatically 
     discarded, so both 0 and 2^n end up in zero. */
  return ((get_parameter() + a_slice.get_parameter()) == 0);
}

/* *********************************************************************
   | This macro is a shortcut for accessing the "power" and "fraction" |
   | derived fields. Indeed it implies some work for decoding the man- |
   | tissa. It can be called only in a derived class.                  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         | 
   ********************************************************************* */
#define get_power_fraction_macro            \
  power_type the_power;                     \
  floating_type the_fraction;               \
  decode_mantissa(the_power, the_fraction)

#endif  // __QUANTUM_PHASE_LIKE_PRIMITIVES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
