/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_COMPUTATIONAL_PRIMITIVE_SLICES_
#define __QUANTUM_COMPUTATIONAL_PRIMITIVE_SLICES_

#include "qop_slice.h"           // for basic time slices
#include "qop_phase_like.h"      // for phase-like operations
#include "qop_codes.h"           // codes for quantum operations

class Qop_slicelist;             // forward declaration for slice lists

/* the following is a macro for common methods for primitives. */
#define standard_methods(class)                                         \
/* virtual destructor */                                                \
virtual ~Qop_slice_ ## class() { }                                      \
/* a method for cloning this time slice */                              \
Qop_slice *clone(void) const { return new Qop_slice_ ## class(*this); } \
/* a method for generating the controlled version of this time slice */ \
void control(Qop_slicelist *the_operations, size_type cfirst) const;

/* *********************************************************************
   | This primitive slice is a single qubit operation. Its action is   |
   | the application of a Hadamard transformation on some qubits in    |
   | its formal register (hence it depends on no further parameter).   |
   | The Hadamard transform brings |0> to (|0> + |1>)/sqrt(2) and |1>  |
   | to (|0> - |1>)/sqrt(2), i.e. it transforms the computational      |
   | basis into the Bell's basis. Hadamard is selfadjoint, so that the |
   | default methods adjoin() and has_adjoint_basegate() are OK.       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 25 May 2001         |
   ********************************************************************* */
class Qop_slice_Hadamard : public Qop_slice {
public:
  // Constructor with one preformed list (may throw exception)
  Qop_slice_Hadamard(const Qubit_list &the_list)
    : Qop_slice(&BASEGATE_HADAMARD, the_list) { }
  // clone(), get_class_name(), control() and the virtual destructor
  standard_methods(Hadamard);
};

/* *********************************************************************
   | This primitive slice is a single qubit operation. Its action is   |
   | the application of |0> --> |0> and |1> --> e^(2 PI i x / 2^k)|1>  |
   | on some qubits in its formal register. k is an unsigned integer   |
   | number and x is a floating point number. Some rounding can take   |
   | place (see the Qop_phase_like class for this). The x parameter    |
   | has a default value, so that only specifying the power works.     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 25 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         |
   ********************************************************************* */
class Qop_slice_Phase : public Qop_phase_like {
public:
  // Constructor with a preformed list (may throw various exceptions).
  Qop_slice_Phase(const Qubit_list &the_list,
		  power_type the_power,
		  floating_type the_fraction = 1.0)
    : Qop_phase_like(the_list, the_power, the_fraction) { }
  // clone(), get_class_name(), control() and the virtual destructor
  standard_methods(Phase);
};

/* *********************************************************************
   | This primitive slice is a two qubit operation. Its action is the  |
   | application of |0> --> |0> and |1> --> e^(2 PI i x / 2^k)|1> on   |
   | some qubits in its formal register for targets conditioned by the |
   | corresponding control being found in |1> (it is the identity      |
   | otherwise). k is an unsigned integer number and x is a floating   |
   | point number. Some rounding can take place (see the Qop_phase_like|
   | class for this). The x parameter has a default value, so that     |
   | only specifying the power works.                                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Jun 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulousem      30 Apr 2002         |
   ********************************************************************* */
class Qop_slice_CondPhase : public Qop_phase_like {
public:
  // Constructor with preformed lists (may throw various exceptions).
  Qop_slice_CondPhase(const Qubit_list &the_controls,
		      const Qubit_list &the_targets,
		      power_type the_power,
		      floating_type the_fraction = 1.0)
    : Qop_phase_like(the_controls, the_targets, the_power, the_fraction) { }
  // clone(), get_class_name(), control() and the virtual destructor
  standard_methods(CondPhase);
};

/* *********************************************************************
   | This primitive slice is a two qubit operation. Its action is the  |
   | application of quantum state swaps to some qubits in its formal   |
   | register: corresponding elements of the two lists are exchanged.  |
   | Swap is selfadjoint, so that the default methods adjoin() and     |
   | has_adjoint_basegate() are already OK.                            |
   | ----------------------------------------------------------------- |
   | This "primitive" is now implemented completely in software. For   |
   | this to work we need to ovveride the execution method operator(). |
   | We must revert to the quantum operations for the controlled ver-  |
   | sion however. (03 Dec 2001, S.Bettelli)                           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
class Qop_slice_Swap : public Qop_slice {
public:
  // Constructor with two preformed lists (may throw exception)
  Qop_slice_Swap(const Qubit_list &list_1, const Qubit_list &list_2)
    : Qop_slice(&BASEGATE_SWAP, list_1, list_2) { }
  // clone(), get_class_name(), control() and the virtual destructor
  standard_methods(Swap);
  // very important! override the execution method
  void operator()(const Qreg &a_register) const
  { classical_permutation(a_register); }
};

#endif  // __QUANTUM_COMPUTATIONAL_PRIMITIVE_SLICES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
