/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __ABSTRACT_QUANTUM_SLICE_LISTS_
#define __ABSTRACT_QUANTUM_SLICE_LISTS_

#include <iostream>                   // C++ I/O streams
#include <list>                       // STL doubly linked lists
#include "qexception.h"               // for "quantum" exceptions
#include "qop_slice.h"                // for Qop_slice type

/* *********************************************************************
   | This class embeds a linked list of time slices. Indeed, since a   |
   | time slice is not constructible, the list keeps the pointers, not |
   | the objects. Real objects will be created with specific primitive |
   | constructors through virtual constructors (i.e. calls to clone()).|
   | Methods are provided which runs Qop_slice abstract methods on all |
   | the time slices in the embedded list.                             |
   | ----------------------------------------------------------------- |
   | Since leaving pointers around is dangerous (one could erase the   |
   | pointer without deallocating the object), the actual STL list is  |
   | kept private and only a few methods are exported. No other class  |
   | should care about cloning Qop_slices for a Qop_slicelist: all the |
   | work is hidden in the operator= method.                           |
   | ----------------------------------------------------------------- |
   | (30 Apr 2002) S.Bettelli. [Moved here from qop_builders.h]        |
   | operator~ returns the current object as a reference. Why is it    |
   | important? [assume T = Qop_slicelist in the following] If we have |
   | a function f(T &list), the compiler will never allow for an ex-   |
   | pression like f(T(...)), since T(...) is a temporary and cannot   |
   | be bound to a non-const reference (like the argument of f()).     |
   | But it allows for non-const methods to be called for the constru- |
   | cted object, i.e. T(...).operator~() or equivalently ~T(...) is a |
   | valid statement and it returns the constructed object by referen- |
   | ce, which can be then passed to f(). This is very handy if we     |
   | want to use these classes for generating template circuits and    |
   | then move all the slices into another contatiner without copies.  |
   | ----------------------------------------------------------------- |
   | (30 Sep 2002) S.Bettelli; the field "ancillae" (which was once a  |
   | property of individual slices) has been added. It declares the    |
   | addresses in [0, "ancillae"-1] to concern ancillae. This makes a  |
   | difference in various places, for instance in the split and invert|
   | (and offset) calls, which ignore these qubits and start counting  |
   | after them. This field is necessary to support (multi-)controlled |
   | operations, but can come useful for library routines too.         |
   | ----------------------------------------------------------------- |
   | (01 Feb 2002) S.Bettelli; reorganised the controlled method. It   |
   | is still unsatisfactory thoug and it should be revised. See the   |
   | header of the method for further details.                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 24 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Sep 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      01 Feb 2003         |
   ********************************************************************* */
class Qop_slicelist {
  // an output function for debugging of a list of time slices
  friend std::ostream &operator<<(std::ostream&os, const Qop_slicelist&a_list);
protected:
  // a type for the internal list of pointers to slices
  typedef std::list<Qop_slice *> slice_list_type;
  // the following type is inherited from Qop_slice
  typedef Qop_slice::size_type size_type;
  // for problems in Qop_slicelist::controlled
  _Qexception(invalid_control);
public:
  // DEBUG!!! don't use this
  Qop_slicelist transform(Qop_slice::address i,
			  const Qop_slicelist &e,
			  const Qubit_list &EL,
			  const Qop_slicelist &enc) const;
  // the default constructor creates an empty list with 0 ancillae
  Qop_slicelist() : slice_list(0), ancillae(0) { }
  // this is the non trivial copy constructor
  Qop_slicelist(const Qop_slicelist &a_list) { *this = a_list; }
  // this is the non trivial assignement operator
  Qop_slicelist &operator=(const Qop_slicelist &a_list);
  // the virtual destructor (non trivial)
  virtual ~Qop_slicelist() { clear(); }
  // this operator is just for tricks with non-constant references
  Qop_slicelist &operator~(void) { return *this; }
public:
  // this method returns the number of slices in the internal list
  size_type get_size(void) const { return slices().size(); }
  // this method returns the size of the longest address list in a slice
  size_type get_parallelisation(void) const;
  // this returns the highest slice occupation (skipping ancillae)
  size_type get_occupation(void) const;
  // this returns the number of qubits declared as ancillae
  size_type get_ancillae(void) const { return ancillae; }
  // this method returns true if the list is empty (no slices)
  bool empty(void) const { return slices().empty(); }
  // print list statistics to an output stream
  void print_statistics(std::ostream &os) const;
public:
  // this calls the reverse() method for the internal list
  void reverse(void) { slices().reverse(); }
  // this method clears the slice list and deallocates memory
  void clear(void);
  // this method clones and append the passed list + automatic simplification
  void append(const Qop_slicelist &a_list) { splice(~Qop_slicelist(a_list)); }
  // cannibalises the content of the passed list + automatic simplification
  void splice(Qop_slicelist &a_list);
  // this method saves the pointer of the supplied time slice at the beginning
  void push_front(Qop_slice *a_slice);
  // this method saves the pointer of the supplied time slice at the end
  void push_back(Qop_slice *a_slice);
  // this method runs "shift" on each slice in the list
  void split(size_type head, size_type jump);
  // this method runs "invert" on each slice in the list
  void invert(size_type head, size_type size);
  // this method runs "offset" on each slice in the list
  void offset(size_type jump) { split(0, jump); }
  // this method runs "remap" on each slice in the list
  void remap(const Qubit_list &permutation_list);
  // this method runs "adjoin" on each slice in the list + list reversal
  void adjoin(void);
  // this method runs "operator()" on each slice in the list
  void operator()(const Qreg &a_register) const;
  // this method prepares a list for a controlled operator (to be revised)
  Qop_slicelist controlled(const Qop_slicelist &control_circuit,
			   size_type additional_lines,
			   size_type additional_ancillae) const;
private:
  // an iterator inside a slice container
  typedef slice_list_type::iterator iterator;
  // a constant iterator inside a slice container
  typedef slice_list_type::const_iterator const_iterator;
  // this returns the actual slice container
  slice_list_type &slices(void)             { return slice_list;       }
  // this returns the actual slice container as a constant object
  const slice_list_type &slices(void) const { return slice_list;       }
  // an iterator to the first element
  iterator begin(void)                      { return slices().begin(); }  
  // an iterator to the past-the-end element
  iterator end(void)                        { return slices().end();   }  
  // a constant iterator to the first element
  const_iterator begin(void) const          { return slices().begin(); }  
  // a constant iterator to the past-the-end element
  const_iterator end(void)  const           { return slices().end();   }  
  // a wrapper around erase() deallocating the slice
  iterator erase(iterator a_slice);
  // this sets the number of qubits declared as ancillae
  void set_ancillae(size_type the_ancillae) { ancillae = the_ancillae; }
private:
  // this subroutine runs an algebraic simplification of the operations
  bool simplify(const iterator &a_junction);
  // simplification of adjoint slices (one step)
  bool simplify_adjoint_slices(iterator &a_slice);
  // simplification of slices with adjoint base gates (one step)
  bool simplify_adjoint_basegates(iterator &a_slice);
private:
  // the actual list of pointers to Qop_slice-derived objects
  slice_list_type slice_list;
  // the number of formal addresses declared as ancillae
  size_type ancillae;
};

/* input all the inline functions */
#include "qop_slicelist_inl.h"

#endif  // __ABSTRACT_QUANTUM_SLICE_LISTS_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
