/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __LIST_OF_TIME_SLICES_INLINES_
#define __LIST_OF_TIME_SLICES_INLINES_

#include "qop_slicelist.h"                  // declaration for the class

/* *********************************************************************
   | This method "shifts a part of the circuit forward" for a slice    |
   | list. This means that each formal address whose value is less     |
   | than "head" is left unaltered, while the other addresses are      |
   | shifted ahead "jump" positions. Note that "head" is internally    |
   | increased using "get_ancillae()". In this way split() only works  |
   | on the non-ancilla part of the list and the user doesn't have to  |
   | care about the number of ancillae (he shouldn't know about).      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      31 Oct 2002         |
   ********************************************************************* */
inline void Qop_slicelist::split(size_type head, size_type jump) {
  /* increase "head" with the number of ancillae. */
  head += get_ancillae();
  /* split each time slice individually. */
  for (iterator a_slice = begin(); a_slice != end(); ++a_slice)
    (*a_slice)->split(head, jump);
}

/* *********************************************************************
   | This method "inverts the order of a circuit part" for a slice li- |
   | st. This means that each formal address whose value is less than  |
   | "head" or greater or equal to "head + size" is left unaltered,    |
   | while the other addresses undergo a mirror simmetry in the central|
   | region of addresses [head, size[. Note that "head" is internally  |
   | increased using "get_ancillae()". In this way invert() only works |
   | on the non-ancilla part of the list and the user doesn't have to  |
   | care about the number of ancillae (he shouldn't know about).      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      31 Oct 2002         |
   ********************************************************************* */
inline void Qop_slicelist::invert(size_type head, size_type size) {
  /* increase "head" with the number of ancillae. */
  head += get_ancillae();
  /* invert each time slice individually. */
  for (iterator a_slice = begin(); a_slice != end(); ++a_slice)
    (*a_slice)->invert(head, size);
}

/* *********************************************************************
   | This method transforms the slice list into its adjoint list. This |
   | effect is simply achieved by adjoining each time slice indivi-    |
   | dually and reversing the order of slices in the list.             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      31 Oct 2002         |
   ********************************************************************* */
inline void Qop_slicelist::adjoin(void) {
  /* adjoin each time slice individually. */
  for (iterator a_slice = begin(); a_slice != end(); ++a_slice)
    (*a_slice)->adjoin();
  /* reverse the order of slices in the list. */
  reverse();
}

/* *********************************************************************
   | This method is a wrapper around the erase() method of the under-  |
   | lying list. In addition, it deallocates the actual slice (this    |
   | prevents a memory leak!). The returned value is an iterator which |
   | points to the element immediately following the erased one. The   |
   | most derived nature is read by the virtual table (Qop_slices can  |
   | not be instantiated).                                             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Dec 2001         |
   ********************************************************************* */
inline Qop_slicelist::iterator Qop_slicelist::erase(iterator a_slice) {
  /* deallocate the pointed slice. */
  delete *a_slice;
  /* erase the element and return its result. */
  return slices().erase(a_slice);
}

/* *********************************************************************
   | This method clears the whole slice list in one go: we need to re- |
   | move all the elements (pointers) from the list, as well as to     |
   | deallocate the pointed time slices. The most derived nature for   |
   | each slice is read by the virtual table (Qop_slices can not be    |
   | instantiated).                                                    |
   | ----------------------------------------------------------------- |
   | This routine could be implemented by repeatedly calling erase(),  |
   | but it is faster to deallocate all the slices first and then to   |
   | clear the list with STL clear (we don't need to deal with itera-  |
   | tors which get invalidated).                                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Dec 2001         |
   ********************************************************************* */
inline void Qop_slicelist::clear(void) {
  /* deallocate all the pointed slices. */
  for (iterator a_slice = begin(); a_slice != end(); ++a_slice)
    delete *a_slice;
  /* remove the pointers from the list. */
  slices().clear();
}

/* *********************************************************************
   | This method saves the supplied pointer to time slice at the be-   |
   | ginning of the current list. It is assumed that this time slice   |
   | is, from now on, property of this list (there is no copy here,    |
   | the pointer is simply saved, and the pointed object will be deal- |
   | located inside the destructor). If the current list was not pre-  |
   | viously empty, the fast simplify() routine is called at the end.  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      01 Nov 2002         |
   ********************************************************************* */
inline void Qop_slicelist::push_front(Qop_slice *a_slice) {
  /* test the size of the list. */
  bool was_empty = empty();
  /* push the time slice at the beginning of the list. */
  slices().push_front(a_slice);
  /* call simplify if the list was not previously empty. */
  if (! was_empty) simplify(begin());
}

/* *********************************************************************
   | This method saves the supplied pointer to time slice at the end   |
   | of the current list. It is assumed that this time slice is, from  |
   | now on, property of this list (there is no copy here, the pointer |
   | is simply saved, and the pointed object will be deallocated in-   |
   | side the destructor). If the current list was not previously      |
   | empty, the fast simplify() routine is called at the end.          |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      01 Nov 2002         |
   ********************************************************************* */
inline void Qop_slicelist::push_back(Qop_slice *a_slice) {
  /* test the size of the list. */
  bool was_empty = empty();
  /* push the time slice at the end of the list. */
  slices().push_back(a_slice);
  /* call simplify if the list was not previously empty. */
  if (! was_empty) simplify(----end());
}

#endif  // __LIST_OF_TIME_SLICES_INLINES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
