/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __BASE_GATE_SUMMARY_CLASS_
#define __BASE_GATE_SUMMARY_CLASS_

#include <iostream>                // C++ I/O streams
#include <vector>                  // STL vectors
#include "qexception.h"            // for "quantum" exceptions

class Qop_slice_basegate;          // forward declaration for base gates

/* *********************************************************************
   | This class is a container for a set of properties which can be    |
   | predicated about a base gate for a time slice. The idea is that   |
   | one can use this class to become aware of the set of basic (algo- |
   | rithmically raw) quantum gates; it should also possible to store  |
   | here some properties which concern pairs of gates (like commuta-  |
   | tion relation or controlled translation or else). [not yet]       |
   | ----------------------------------------------------------------- |
   | This class is used in a strange way. There is only one instance   |
   | of it which is interesting (almost a singleton!), and it is owned |
   | statically by a static method of the class which can return a     |
   | reference to it. There is therefore no danger if ctors are public.|
   | ----------------------------------------------------------------- |
   | (07 Nov 2002) S.Bettelli. Begin with listing all the registered   |
   | gates; the idea is that each base gate registers itself here when |
   | it becomes available: as a first try, register a pointer to them. |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 Nov 2002         |
   ********************************************************************* */
class Qop_summary {
public:
  // an integer type for indexes inside the base gate list
  typedef unsigned long size_type;
  // an error for two gates trying to register with the same opcode
  _Qexception(gate_already_registered);
public:
  // this returns a constant reference to the summary
  static const Qop_summary &get_summary(void) { return get_Summary(); }
  // a method for registering a new base gate
  static size_type register_basegate(const Qop_slice_basegate *a_basegate);
  // this method returns the number of registered base gates
  static size_type get_gates(void) { return get_summary().database.size(); }
  // this method returns a const pointer to the index-th base gate type
  static const Qop_slice_basegate *get_basegate(size_type index);
private:
  // a typedef for the container of base gate types
  typedef std::vector<const Qop_slice_basegate *> database_type;
  // this method returns a mutable reference to the singleton
  static Qop_summary &get_Summary(void);
  // the internal database of base gates
  database_type database;
};

/* *********************************************************************
   | This method returns a pointer to the index-th base gate type from |
   | the internal database. The pointer links to a constant object, so |
   | that its properties can be inquired, but it cannot be modified.   |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 Nov 2002         |
   ********************************************************************* */
inline const Qop_slice_basegate *Qop_summary::get_basegate(size_type index) {
  return get_summary().database[index];
}

/* *********************************************************************
   | This method returns a reference to the unique meaningful instance |
   | of this class, which is instantiated like a static object in this |
   | routine; it is therefore initialised only once, the first time    |
   | this method is called, and it is subsequently always available.   |
   | Since the allocation is automatic, we don't have to care about    |
   | its destruction. This method is private, there should be another  |
   | public method exporting the summary to the rest of the world as   |
   | a constant reference.                                             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 Nov 2002         |
   ********************************************************************* */
inline Qop_summary &Qop_summary::get_Summary(void) {
  static Qop_summary singleton;
  return singleton;
}

/* *********************************************************************
   | This is an output function for debugging the time slice summary.  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 Nov 2002         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qop_summary &a_summary);

#endif  // __BASE_GATE_SUMMARY_CLASS_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
