/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include "qoperator.h"              // base class declaration

/* *********************************************************************
   | This is the concatenation operation for quantum operators. It     |
   | takes two constant references to Qop objects and builds a Qop,    |
   | returned by value, which is the composition of the two. The first |
   | primitives to be executed will be those from the first operator   |
   | in the argument list. Concatenation is not a member function of   |
   | the Qop class. The list of operations undergoes the simplifica-   |
   | tion routine before being inserted into the Qop.                  |
   | (27 Sep 2001) S.Bettelli. New convention on direction of concat.  |
   | (02 May 2002) S.Bettelli. Simplify the routine and make it more   |
   |   robust by using operator<< between Qops                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 14 Jun 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulousem      02 May 2002         |
   ********************************************************************* */
Qop operator&(const Qop &operator_1, const Qop &operator_2) {
  /* This is the operator which will be returned at the end. */
  Qop joint_operator;
  /* create a copy of both the arguments. */
  Qop copy_operator_1(operator_1);
  Qop copy_operator_2(operator_2);
  /* fetch the operation(s) from the two copies with splicing. This
     will trigger the simplification routine in the Qop_slicelist
     class. No unnecessary copies take place at this point. */
  joint_operator << copy_operator_1 << copy_operator_2;
  /* return (by value) the joint operator (hope that the compiler is
     optimised and that it does not make a copy at this point!). */
  return joint_operator;
}

/* *********************************************************************
   | This output function prints an operator identifier (the pointer   |
   | actually) and then its operatin list. For debugging.              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 24 May 2001         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qop &an_operator) {
  os << "Qop " << static_cast<const void *>(&an_operator)// operator identifier
     << " and its time slices:\n"                
     << an_operator.get_operations();                    // operation list
  return os;                                             // return stream
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
