/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_COMPUTATIONAL_PRIMITIVES_
#define __QUANTUM_COMPUTATIONAL_PRIMITIVES_

#include "qoperator.h"               // for generic quantum operations

/* *********************************************************************
   | This quantum operation applies a Hadamard transformation on some  |
   | qubits in its formal register. The Hadamard transform brings |0>  |
   | to (|0> + |1>)/sqrt(2) and |1> to (|0> - |1>)/sqrt(2), i.e. it    |
   | transforms the computational basis into the Bell's basis.         |
   | 1) Ctor with 1 list: apply a Hadamard gate on all the elements.   |
   | 2) Ctor with size  : like 1) with list = [0, size[                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
class QHadamard : public Qop {
public:
  // Default constructor ("the_size" Hadamard gates at the beginning)
  QHadamard(size_type the_size = 1);
  // Constructor through preformed list (may throw exception)
  QHadamard(const Qubit_list &the_list);
  // virtual destructor
  virtual ~QHadamard() { }
};

/* *********************************************************************
   | This quantum operation applies a relative phase shift on some qu- |
   | bits in its formal register. A phase shift brings |0> into |0>    |
   | and |1> into e^(2 PI i x/ 2^k)|1>, where k is an unsigned integer |
   | number and x is a floating point number. The default value of x   |
   | is 1, which is helpful in many situations.                        |
   | ----------------------------------------------------------------- |
   | The first ctor applies a QPhase gate on all the elements. The     |
   | second one is similar and creates a list = [0, size[              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         |
   ********************************************************************* */
class QPhase : public Qop {
public:
  // Default constructor (k=1 i.e. the Z Pauli matrix)
  QPhase(size_type the_size = 1,
	 ::quantum_phase_power_type the_power = 1,
	 ::quantum_phase_floating_type the_fraction = 1.0);
  // Constructor with a preformed list (may throw exception).
  QPhase(const Qubit_list &the_list,
	 ::quantum_phase_power_type the_power = 1,
	 ::quantum_phase_floating_type the_fraction = 1.0);
  // virtual destructor
  virtual ~QPhase() { }
};

/* *********************************************************************
   | This quantum operation applies a "NOT" transformation (the Pauli  |
   | X matrix) on some qubits in its formal register. A "NOT" gate is  |
   | the obvious thing, i.e. it brings |0> to |1> and |1> to |0>.      |
   | 1) Ctor with 1 list: apply a NOT gate on all the elements.        |
   | 2) Ctor with size  : like 1) with list = [0, size[                |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 Jun 2002         |
   ********************************************************************* */
class QNot : public Qop {
public:
  // Default constructor ("the_size" NOT gates at the beginning)
  QNot(size_type the_size = 1);
  // Constructor through preformed list (may throw exception)
  QNot(const Qubit_list &the_list);
  // virtual destructor
  virtual ~QNot() { }
};

/* *********************************************************************
   | This quantum operation is analogous to QPhase, but the transfor-  |
   | mation is conditioned by the corresponding control qubit being    |
   | found in |1> (the identity otherwise)                             |
   | ----------------------------------------------------------------- |
   | The first ctor applies a conditional QPhase gate on all the cor-  |
   | responding elements of the two lists (think of the first as the   |
   | "control" one, if you need to fix ideas). The second ctor is si-  |
   | milar and creates the two lists [0, size[ and [size, 2*size[      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         |
   ********************************************************************* */
class QCondPhase : public Qop {
public:
  // Default constructor (k=1, i.e. a controlled Z)
  QCondPhase(size_type the_size = 1,
	     ::quantum_phase_power_type the_power = 1,
	     ::quantum_phase_floating_type the_fraction = 1.0);
  // Constructor with preformed lists (may throw exception).
  QCondPhase(const Qubit_list &the_controls,
  	     const Qubit_list &the_targets,
	     ::quantum_phase_power_type the_power = 1,
	     ::quantum_phase_floating_type the_fraction = 1.0);
  // virtual destructor
  virtual ~QCondPhase() { }
};

/* *********************************************************************
   | This quantum operation applies a controlled not (CNOT) on some    |
   | qubits in its formal register. A controlled not brings |0> --> |1>|
   | and |1> --> |0> in a target qubit conditioned by the correspon-   |
   | ding control qubit being found in |1> (the identity otherwise).   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
class QCnot : public Qop {
public:
  // Default constructor
  QCnot(size_type the_size = 1);
  // Constructor, through preformed lists (may throw exception)
  QCnot(const Qubit_list &the_controls, const Qubit_list &the_targets);
  // virtual destructor
  virtual ~QCnot() { }
};

/* *********************************************************************
   | This quantum operation is analogous to QPhase, but the transfor-  |
   | mation is conditioned by the two corresponding control qubits     |
   | being found in |1> (the identity otherwise)                       |
   | ----------------------------------------------------------------- |
   | The first ctor applies a conditional QPhase gate on all the cor-  |
   | responding elements of the three lists (think of the first and    |
   | the second as the "control" ones, if you need to fix ideas). The  |
   | second ctor is similar and creates the three lists [0, size[ ,    |
   | [size, 2*size[ and [2*size, 3*size[                               |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      02 May 2002         |
   ********************************************************************* */
class QCondCondPhase : public Qop {
public:
  // Default constructor (k=1, i.e. a controlled Z)
  QCondCondPhase(size_type the_size = 1,
		 ::quantum_phase_power_type the_power = 1,
		 ::quantum_phase_floating_type the_fraction = 1.0);
  // Constructor with preformed lists (may throw exception).
  QCondCondPhase(const Qubit_list &the_controls_1,
		 const Qubit_list &the_controls_2,
		 const Qubit_list &the_targets,
		 ::quantum_phase_power_type the_power = 1,
		 ::quantum_phase_floating_type the_fraction = 1.0);
  // virtual destructor
  virtual ~QCondCondPhase() { }
};

/* *********************************************************************
   | This quantum operation applies a Toffoli gate to all the qubits   |
   | in its formal registers. A Toffoli gate brings the |11x> states   |
   | to |11y>, where y is the negation of x (and acts as the identity  |
   | on all the other cases).                                          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 14 Sep 2001         |
   ********************************************************************* */
class QToffoli : public Qop {
public:
  // Default constructor
  QToffoli(size_type the_size = 1);
  // Constructor, through preformed lists (may throw exception)
  QToffoli(const Qubit_list &the_controls_1,
	   const Qubit_list &the_controls_2, const Qubit_list &the_targets);
  // virtual destructor
  virtual ~QToffoli() { }
};

/* *********************************************************************
   | This quantum operation applies a swap gate to some qubits in its  |
   | formal register. A swap gate interchanges the quantum state of    |
   | two qubits, and is a two qubit quantum gate. The constructor syn- |
   | tax is a bit unusual, due to the nature of swap:                  |
   | 1) ctor with two lists: (standard) interchanges the corresponding |
   |    elements of the two lists.                                     |
   | 2) ctor with one list: (additional) interchanges the first loca-  |
   |    tion with the last one, the second one with the second last    |
   |    one and so on ... If the list has an odd number of locations,  |
   |    the "central" qubit is left untouched.                         |
   | 3) ctor with size: (unusual) this is like calling the second ctor |
   |    with one list = [0, size[, differently from other operations!  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
class QSwap : public Qop {
public:
  // Default constructor
  QSwap(size_type the_size = 1);
  // Constructor, through one preformed list (may throw exception)
  QSwap(const Qubit_list &the_list);
  // Constructor, through two preformed lists (may throw exception)
  QSwap(const Qubit_list &list_1, const Qubit_list &list_2);
  // virtual destructor
  virtual ~QSwap() { }
};

/* *********************************************************************
   | This quantum operation applies a quantum Fourier transformation   |
   | on some qubits in its formal register; it depends on no further   |
   | parameter. This transformation includes the reversal of qubit     |
   | order at the end of the operation.                                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
class QFourier : public Qop {
public:
  // Default constructor
  QFourier(size_type the_size = 1);
  // Constructor through preformed list (may throw exception)
  QFourier(const Qubit_list &the_list);
  // virtual destructor
  virtual ~QFourier() { }
};

#endif  // __QUANTUM_COMPUTATIONAL_PRIMITIVES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***

