/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_REGISTERS_
#define __QUANTUM_REGISTERS_

#include <iostream>                     // C++ standard I/O
#include "qexception.h"                 // for "quantum" exceptions
#include "qlist.h"                      // for the qubit list class
#include "qbitset.h"                    // for collection of booleans

/* *********************************************************************
   | This class implements the quantum register abstraction, which is  |
   | an interface to a portion of a quantum device. It doesn't know    |
   | about the register "value" (otherwise we would be simulating the  |
   | quantum machine!) unless you decide to measure it; with this in   |
   | mind it is easy to understand why e.g. measurement operation is   |
   | a constant method. Conventionally, the first qubit referenced by  |
   | the register (i.e. that returned by operator[](0)) corresponds to |
   | the most significant bit (MSB), the last one corresponds to the   |
   | least significant bit (LSB).                                      |
   | ----------------------------------------------------------------- |
   |                                                                   |
   | Requirements on quantum registers                                 |
   | =================================                                 |
   |                                                                   |
   | 1)  Qregs can be allocated, with an explicit size, and released   |
   |     by interacting with an address manager.                       |
   | 2)  A Qreg can be created which corresponds to a part of another  |
   |     quantum register (both single qubit and range addressing).    |
   | 3)  Qregs can be concatenated, provided that their linked qubits  |
   |     sets are disjoint (explicit check at run-time).               |
   | 4)  The ordering of the qubits inside a Qreg can be swapped with- |
   |     out interacting with the quantum device.                      |
   | 5)  A pair of Qregs can have their addresses swapped.             |
   | 6)  Qregs can be initialised and assigned (integers and bitmasks) |
   | 7)  Qregs can be measured (blocking, return integers or bitmasks) |
   | 8)  Registers can be extended/reduced by adding/dropping qubits   |
   |     at their beginning.                                           |
   | 9)  The register's size can be inquired.                          |
   |                                                                   |
   | Not really required, but useful                                   |
   | ===============================                                   |
   |                                                                   |
   | 10) The register's address set can be inspected.                  |
   |                                                                   |
   | ----------------------------------------------------------------- |
   | (04 Sep 2001, S.Bettelli) Added a custom container,Qbitset, which |
   | holds an arbitrary number of boolean values, for input-output     |
   | with registers (initialisations and measurements).                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   ********************************************************************* */
class Qreg {
  // an output function for debugging
  friend std::ostream &operator<<(std::ostream &os, const Qreg &a_register);
public:
  // a type for values which registers are to be initialised to
  typedef ::quantum_message_type value_type;
  // this is an alias for the qubit address type
  typedef Qubit_segment::address address;
  // this is an alias for the qubit segment size type
  typedef Qubit_segment::size_type size_type;
  // this is an alias for a constant iterator inside the qubit list
  typedef Qubit_list::const_iterator qubits_iterator;
public:
  // a class for register initialisation errors
  _Qexception(init_error);
  // a class for assignement overflows
  _Qexception(assignment_error);
  // a class for problem during operator-=
  _Qexception(invalid_discard);
public:
  // default constructor (may specify explicit size and value)
  Qreg(size_type the_size = 1, value_type the_value = 0);
  // constructor which accepts a collection of boolean values
  Qreg(const Qbitset &the_bits);
  // copy constructor (this avoids a default one)
  Qreg(const Qreg &a_register);
  // destructor (this will release allocated resources)
  virtual ~Qreg();
private:
  // custom constructor (incorporating a qubit list)
  Qreg(Qubit_list &a_list);
  // inhibit the assignement operator through registers (const fools g++ 3.0)
  Qreg &operator=(const Qreg &a_register) const;
public:
  // how to inizialise a register to a given sequence of bits
  const Qreg &operator=(const Qbitset &the_bits) const;
  // initialisation using an integer number instead of a bitmask
  const Qreg &operator=(value_type a_value) const;
  // this method returns the register size
  size_type size(void) const { return get_qubit_list().size(); }
  // this method swaps the order of qubits in a register
  Qreg &swap(void);
  // this method swaps the qubit locations of two registers
  Qreg &swap(Qreg &a_register);
  // this operator implements the single qubit addressing
  Qreg operator[](address an_index) const;
  // this operator implements the qubit range addressing
  Qreg operator()(address an_index, size_type a_size = 0) const;
  // this operator implements register concatenation
  Qreg operator&(const Qreg &second_register) const;
  // this operator implements register augmentation
  Qreg &operator&=(const Qreg &second_register);
  // this operator prepends some fresh qubits to the register
  Qreg &operator+=(size_type the_size);
  // this operator discards some qubits from the register beginning
  Qreg &operator-=(size_type the_size);
  // this method acts a register measurement (blocking)
  Qbitset measure(void) const;
  // this method locates the qubits in the quantum device (blocking)
  Qubit_list locate(void) const;
public:
  // this method exports the qubit list through a const reference
  const Qubit_list &get_qubit_list(void) const { return qubits; }
private:
  // the list of managed qubits for this register
  Qubit_list qubits;
  // a method which performs the core of construction
  void run_construction(const Qbitset &the_bits);
  // a method which performs the core of register initialisation
  void run_initialisation(const Qbitset &the_bits,
			  const Qubit_list &a_list) const;
  // an encapsulation of join() from the list class
  void run_join(Qubit_list &a_list);
  //a method which performs a segment fragmentation
  void run_fragmentation(const Qubit_segment &the_segment,
			 address target, bool reverse = false) const;
  // this method incorporates (through swap) the supplied list internally
  void incorporate_qubits(Qubit_list &a_list, bool run_checkin = true);
  // a quick link to the address manager for this register (private!)
  class Qaddress_manager_base *get_manager(void) const;
};

/* *********************************************************************
   | Constructor for a quantum register, which accepts a size and a    |
   | value (the class declaration should provide default values for    |
   | both, one qubit in the register initialised to zero).             |
   | ----------------------------------------------------------------- |
   | 1) Check that size is positive and value is suitable (this means  |
   |    it is in the range [0,2^n[ of integers), otherwise throw       |
   |    exception of type init_error().                                |
   | 2) Transform the value into a collection of booleans, then call   |
   |    the regular construction routine.                              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
inline Qreg::Qreg(size_type the_size, value_type the_value) {
  /* First check that the size is greater than zero (a zero qubit
     register is a nonsense up to now). */
  if (the_size < 1) throw init_error();
  /* Then build a collection of booleans of size "the_size" with the
     supplied value. The routine checks that the_size is indeed big
     enough and throws an exception in case it is not (this is the reason
     why we didn't check it before). Then run a specific routine which
     takes care of the construction and initialisation, including some
     sanity checks. */
  run_construction(Qbitset(the_size, the_value));
}

/* *********************************************************************
   | General constructor for a quantum register, accepting a sequence  |
   | of booleans. The size of the vector is the size of the register.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
inline Qreg::Qreg(const Qbitset &the_bits) {
  /* run a specific routine which takes care of the construction and
     the initialisation, included some sanity checks. */
  run_construction(the_bits);
}

/* *********************************************************************
   | This is a custom constructor for a quantum register which uses    |
   | the supplied list as its internal list. This is to say that       |
   | "a_list" will be empty after this call. The incorporate_qubits()  |
   | method takes care to deal with the address manager.               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 26 Mar 2001         |
   ********************************************************************* */
inline Qreg::Qreg(Qubit_list &a_list) {
  incorporate_qubits(a_list);
}

/* *********************************************************************
   | This member function implements the reversal of qubits inside a   |
   | register (their order, I mean). Since the association between     |
   | logical qubits and hardware qubits is managed by software, it is  |
   | faster to update our data structures than physically move the     |
   | qubit systems. This function is linear time and returns a refe-   |
   | rence to the current register.                                    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Feb 2001         |
   ********************************************************************* */
inline Qreg &Qreg::swap(void) {
  /* call the appropriate method from the list class. */
  qubits.reverse();
  /* return the current register */
  return *this;
}

/* *********************************************************************
   | This member function implements the swapping of qubits of this    |
   | register with those of another one. This is even simpler than     |
   | reversing the order of qubits inside the register since the list  |
   | of qubits have a special constant time method for this.           |
   | The return value is a reference to the current register.          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Feb 2001         |
   ********************************************************************* */
inline Qreg &Qreg::swap(Qreg &a_register) {
  /* use constant time swap() method of lists. */
  qubits.swap(a_register.qubits);
  /* return the current register */
  return *this;
}

/* *********************************************************************
   | This method initialises a register to a given value. It is only   |
   | an interface to the more general assignement operator which can   |
   | accept collections of booleans.                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
inline const Qreg &Qreg::operator=(value_type a_value) const {
  /* build a boolean vector from the supplied value. The size of the
     vector must be the same as the size of the register. Remeber that
     this conversion could throw an exception. Then run the real initia-
     lisation by calling the other version of the assignement operator. */
  return operator=(Qbitset(size(), a_value));
}

/* *********************************************************************
   | This method initialises a register to a given bit string. Indeed  |
   | all the hard work is done by run_initialisation. This method only |
   | adds the qubit list to the arguments and return a reference to    |
   | the current object as usual for assignement operators.            |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
inline const Qreg &Qreg::operator=(const Qbitset &the_bits) const {
  /* push the qubit list into the arguments and initialise */
  run_initialisation(the_bits, get_qubit_list());
  /* return the current object, as usual for assignement operators. */
  return *this;
}

#endif  // __QUANTUM_REGISTERS_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
