/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iomanip>                       // C++ I/O manipulators
#include "qsegment.h"                    // class declaration

/* *********************************************************************
   | The (non-trivial) constructor of a memory segment needs to know   |
   | the beginning of the segment and its size. Fresh segments are     |
   | never reversed. "the_size" must be greater than zero, otherwise   |
   | an exception of class invalid_size() will be thrown.              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 17 Jan 2001         |
   | Stefano Bettelli, INFN and Trento University, 07 Dec 2001         |
   ********************************************************************* */
Qubit_segment::Qubit_segment(address the_first, size_type the_size) :
  first_address(the_first),
  last_address(the_first + the_size - 1) {
  /* throw an exception if the size is zero; we can't represent zero
     sized segments with our current approach! They are unmeaningful
     to quantum computing anyway. */
  if (the_size == 0) throw invalid_size();
}

/* *********************************************************************
   | A variation of the previous constructor, which accepts the        |
   | absolute addresses of the first and last qubits. Note that this   |
   | segment can be created reversed. The boolean argument is needed   |
   | only to disambiguate the ctor resolution (otherwise it would have |
   | the same prototype as the default one, because the address type   |
   | and the size_type type are the same).                             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 09 Feb 2001         |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
Qubit_segment::Qubit_segment(address the_first, address the_last, bool) :
  first_address(the_first),
  last_address(the_last)
{ }

/* *********************************************************************
   | This method checks whether the current and the supplied register  |
   | have any common address, i.e. if their ranges overlap.            |
   | ----------------------------------------------------------------- |
   | Since segments are collections of contiguous addresses, there is  |
   | segment overlap if and only if the lowest or highest address in   |
   | one of them falls inside the valid range of the other.            |
   | (08 Dec 2001) S.Bettelli, since now the internal representation   |
   |   keeps first() and last(), use lowest() and highest() of the     |
   |   current segment to define the "valid range", but test the range |
   |   against first() and last() of the second segment.               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 Mar 2001         |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
bool Qubit_segment::is_overlapping(const Qubit_segment &a_segment) const {
  /* test that the first address of the supplied segment falls outside
     of the range of the current register. Return true if the condition
     is not met. This means that the registers are overlapping. */
  if (a_segment.first() >= lowest() &&
      a_segment.first() <= highest()) return true;
  /* run the same check with the last address of supplied segment. */
  if (a_segment.last() >= lowest() &&
      a_segment.last() <= highest()) return true;
  /* if we didn't return yet it means that the registers are disjoint. */
  return false;
}

/* *********************************************************************
   | This method modifies the segment size leaving the first address   |
   | and the orientation unmodified. If supplied size is zero, an      |
   | exception of class invalid_size() is thrown.                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 22 Jan 2001         |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
void Qubit_segment::resize(size_type new_size) {
  /* don't let anyone set size equal to zero! */
  if (new_size == 0) throw invalid_size();
  /* if segment is "direct", the last address is the first plus something. */
  if (is_direct()) last_address = first_address + new_size - 1;
  /* if it is reversed, do the opposite ... */
  else last_address = first_address - new_size + 1;
}

/* *********************************************************************
   | This method modifies the segment by increasing all the contained  |
   | addresses by "a_shift". If the shift would cause the indexes to   |
   | overflow, throw exception invalid_shift().                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
void Qubit_segment::shift_forward(size_type a_shift) {
  /* if any address is overflowing, throw an exception. We only need to 
     test the highest absolute address (it is the first which overflows!) */
  if ((highest() + a_shift) < highest()) throw invalid_shift();
  /* increase the first and the last addresses. */
  first_address += a_shift;
  last_address += a_shift;
}

/* *********************************************************************
   | This method modifies the segment by decreasing all the contained  |
   | addresses by "a_shift". If the shift would cause the indexes to   |
   | overflow, throw exception invalid_shift().                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
void Qubit_segment::shift_backward(size_type a_shift) {
  /* if any address is overflowing, throw an exception. We only need to 
     test the lowest absolute address (it is the first which overflows!) */
  if ((lowest() - a_shift) > lowest()) throw invalid_shift();
  /* decrease the first and last addresses. */
  first_address -= a_shift;
  last_address -= a_shift;
}

/* *********************************************************************
   | This method modifies the lowest (and the highest) address of the  |
   | current segment setting the lowest address to "new_lowest"; the   |
   | segment size and orientation is left unaltered. If the shift would|
   | cause the indexes to overflow, throw exception invalid_shift().   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   ********************************************************************* */
void Qubit_segment::shift_lowest(address new_lowest) {
  /* if new lowest address is smaller, shift back */
  if (new_lowest < lowest()) shift_backward(lowest() - new_lowest);
  /* otherwise, shift forward. */
  else shift_forward(new_lowest - lowest());
}

/* *********************************************************************
   | This method checks the possibility of joining *this with the      |
   | supplied memory segment. This routine preserves the order of      |
   | qubits in the two segments, i.e. it can attach the second AFTER   |
   | the first, but NOT before! The return value is true if and only   |
   | if the joining is possible. If a pointer to a segment is supplied |
   | (different from NULL) the joined segment is saved for later use.  |
   | Since each segment must contain at least one address, a joined    |
   | segment must contain at least two addresses, hence the orienta-   |
   | tion is always well defined.                                      |
   | ----------------------------------------------------------------- |
   | We will adopt the following criterion for understanding whether   |
   | the join is possible or not. Say F the first address of the first |
   | segment and L the last address of the second segment (remember    |
   | that first() and last() depend on the orientation). Then the pair |
   | (F,L) defines a new segment: its size must be equal to the sum of |
   | the sizes of the two registers separately.                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 17 Jan 2001         |
   | (27 Apr 2001) S.Bettelli - preserve order !                       |
   | (07 May 2001) S.Bettelli - complete redesign after having changed |
   |                the internal representation of Qubit_segment's.    |
   | (10 May 2001) S.Bettelli - found a bug, condition not sufficient. |
   | (04 Jun 2001) S.Bettelli - Check only, don't actually join.       |
   |                optionally save the temporary segment.             |
   ********************************************************************* */
bool Qubit_segment::can_join(const Qubit_segment &a_segment,
			     Qubit_segment *save_me) const {
  /* get the first address of the first segment and the last address
     of the second segment (which define the candidate joined segment),
     then create a temporary segment with these borders. Use the second
     constructor (that which is selected by the boolean argument). */
  Qubit_segment joined(this->first(), a_segment.last(), true);
  /* compare the size of the temporary segment with the sum of the sizes
     of the first and second segment. If they are different, the join is
     not possible: return with the appropriate termination status. */
  if (joined.size() != (this->size() + a_segment.size())) return false;
  /* the previous condition is not sufficient! (10 May 2001) Check that
     the last address in the first segment and the first address in the
     second segment have difference one. */
  Qubit_segment paranoia(this->last(), a_segment.first(), true);
  if (paranoia.size() != 2) return false;
  /* if we are still here, we can join the two segments. Return success
     and save the intermediate result if a slot is available. */
  if (save_me) *save_me = joined;
  return true;
}

/* *********************************************************************
   | This function shows the content of a segment (for debugging). For |
   | each segment it prints the segment first and last valid addresses |
   | and the segment size. If the segment is reversed a little label   |
   | is added.                                                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 17 Jan 2001         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qubit_segment &a_segment) {
  os << std::setfill('.')                             // fill with dots
     << "begin " << std::setw(5) << a_segment.first() // segment begin
     << " last " << std::setw(5) << a_segment.last()  // segment last valid
     << " size " << std::setw(5) << a_segment.size()  // segment size
     << (a_segment.is_reversed() ? " (rev.)" : "")    // reversed ?
     << std::setfill(' ');                            // reset normal filling
  return os;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
