/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iostream>
#include "quantum_computing.h"
#include "quantum_library.h"
#include "qsimulator.h"

int main(void) {
  try {
#if 1 // set this to "true" for the first example
    /* This example reproduces the left-right oscillations (without noise)
       shown in quant-ph/0202113 (available on the arXiv), now published as:
       "Simulation of chaos-assisted tunneling in a semiclassical regime on
       existing quantum computers", Phys. Rev. A v.66 (2002) p.054301. */
    int    q     = 5;                // resources (# of qubits)
    int    c     = 2;                // number of simulated momentum cells
    int    niter = 500;              // number of map iterations
    double K     = 0.04;             // K = k * T defines the classical map
    double a     = 1.6;              // "centre" of the potential (2*PI*a/L)
    int N = (1 << q);                // number of quantum levels
    int memsize = q+1;               // simulated memory
    /* initialisation of the simulator. */
    simulator.set_numqubits(memsize);
    simulator.set_noise(0.0, 0.0, 0.0);
    /* The following two operators commute from position to momentum
       representation and back. Since these two representations are
       conjugate, these Qops are simply Fourier transforms. */
    Qop mom_to_pos = QFourier(q);
    Qop pos_to_mom = !mom_to_pos;    // this is an anti-transform
    /* Coefficient and operator for the free evolution of any quantum map.
       In a more standard notation, this is the kinetic term exp(-iTn^2/2),
       therefore we get T = 2*PI*c / N . If Tsym is true, add a few gates
       in order to have a perfect p symmetry (i.e.  i --> i+1/2). */
    Qop rotator = expower(q, 2, - c/(2.0*N));
    /* This is the construction of the "kick" for the double well map
       algorithm. Remember that ((i+DD)^2 - AA^2)^2 = sum_j f_j i^j.
       The choice DD = 0.5*(1-N) guarantees that we take a coordinate
       range which is symmetric around zero. */
    double math_TWOPI = 4.0*asin(static_cast<double>(1.0));
    double lb = math_TWOPI;          // adimensional cell size
    double AA = a * (N/lb);          // scaled "centre" for the potential
    double DD = 0.5 * (1-N);         // x_i = (L/N)(i + DD)
    double f  = - (lb*lb * K)/(c * N*N*N);
    double f1 = f * 4*DD * ( DD*DD - AA*AA );
    double f2 = f * ( 6*DD*DD - 2*AA*AA );
    double f3 = f * 4*DD;
    double f4 = f;
    Qop kick = (expower(q, 1, f1) & expower(q, 2, f2) &
		expower(q, 3, f3) & expower(q, 4, f4));
    /* The following operator is a complete map step. */
    Qop mapstep = kick & pos_to_mom & rotator & mom_to_pos;
    /* This operator will "prepare" the initial state. */
    Qop preparation = QNot(1) & QHadamard(q-1)>>1;
    /* statistics for one map step. */
    std::cerr << "\nQubit utilisation for one map step:\n";
    mapstep.print_statistics(std::cerr);
    /* initialise the random seed. */
    simulator.init_random_seed();
    /* reset the register */
    Qreg simul(q);
    /* prepare the initial state */
    preparation(simul);
    /* now iterate the map "niter" times. */
    for (int count = 0; count < niter; ++count) { 
      /* get the probability of being on the left or on the right (i.e.
	 the probability that the most significant qubit is |0> or |1>). */
      unsigned long msb = (simul.locate())[0];
      /* read the exact probability by accessing the simulator. */
      double prob_on_the_left = simulator.is_set(msb);
      /* run a step of the map */
      mapstep(simul);
      /* print out the current result. */
      std::cout << (count + 1) << '\t' << prob_on_the_left << std::endl;
    }
#else // skip the next example. 
    /* This example reproduces the quantum circuit for summing three
       numbers, section 4.1, page 12 in the reference article. It uses
       3 registers of 4 qubits each, so 12 qubits in the qmem.
       The result of the "computation" should be 3+7+10 = 4 mod 16. */
    int size = 4;
    simulator.set_numqubits(3*size);
    simulator.init_random_seed();
    Qreg x(size,3), y(size,7), z(size,10), all_registers(x & y & z);
    /* prepare the sequence of conditional rotations */
    Qop rot;
    for (int i=0; i<size; ++i)
      rot << QCondPhase(size-i, i+1).offset(i);
    /* prepare a translated Fourier transform */
    Qop tras = (QFourier(size) & QSwap(size)).offset(size);
    /* compose the two argument adder */
    Qop add = tras & rot & ! tras;
    /* offset the 2-adder down, split another 2-adder,
       compose the two 2-adders into one 3-adder */
    Qop add3 = (add >> size) & add(size, size, Qop::SPLIT);
    /* Show the quantum operator structure. */
    std::cout << "============================\n"
	      << add3 << std::endl
	      << "============================\n";
    /* Show the content of the registers (well, measure them ...) */
    std::cout << "===== Before the addition ====="
	      << "\nRegister x: " << x.measure()
	      << "\nRegister y: " << y.measure()
	      << "\nRegister z: " << z.measure() << std::endl;
    /* "Run" the computation on a 12 qubit register. */
    add3(all_registers);
    /* Measure the registers after the computation */
    std::cout << "===== After  the addition ====="
	      << "\nRegister x: " << x.measure()
	      << "\nRegister y: " << y.measure()
	      << "\nRegister z: " << z.measure() << std::endl;
#endif
  } catch_qexception;
  return 0;
}
