/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include "qbitset.h"                              // class declaration

/* *********************************************************************
   | This private method duplicates a vector of storage_type elements  |
   | into the private data area. The "num_elements" field must be      |
   | already set, and the passed vector must be of the same size (this |
   | is not explicitely checked, but the method is private ...).       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   ********************************************************************* */
void Qbitset::duplicate_data(const storage_type *data_source) {
  /* if there was something allocated, deallocate it. */
  if (data) delete [] data;
  /* allocate a new data area with the correct size */
  data = new storage_type[num_elements];
  /* copy all the elements. */
  for (size_type i = 0; i < num_elements; ++i)
    data[i] = data_source[i];
}

/* *********************************************************************
   | This method calculates how many bits are required to store some   |
   | integer number. It is useful during initialisations. It turns out |
   | to be the floor() of the base-2 logarithm of 'a_value' plus one   |
   | (so that 15 requires 4 qubits while 16 gets 5 qubits).            |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 16 Jan 2001         |
   ********************************************************************* */
Qbitset::size_type Qbitset::bits_for(storage_type a_value) {
  /* this variable will hold the calculated size. */
  size_type the_calculated_size = 0;
  /* Using right shift operations turns out to be simpler */
  while (a_value) { ++the_calculated_size; a_value >>= 1; }
  /* return the calculated size. */
  return the_calculated_size;
}

/* *********************************************************************
   | This constructor builds a bit collection of size "a_size" and     |
   | writes into it the binary representation of the storage_type      |
   | object "a_value" (which defaults to zero). Note that if a_size is |
   | less than the number of bits required for storing a_value, an     |
   | exception qbitset_overflow() will be thrown. The binary represen- |
   | tation of "a_value" is written into the container so that its     |
   | least significant bit (LSB) is matched against the LSB of the     |
   | container (that with index (size-1)).                             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Sep 2001         |
   ********************************************************************* */
Qbitset::Qbitset(storage_type a_size, size_type a_value) :
  /* initialise the number of actually used bits. */
  real_size(a_size),
  /* initialise the number of storage elements in the container. */
  num_elements(calculate_container_size(real_size)),
  /* allocate a C++ vector for holding the storage elements */
  data(new storage_type[num_elements]) {
  /* be sure all the elements are zeroed. */
  for (size_type index = 0; index < elements(); ++index)
    set_element(index, 0);
  /* if "a_value" is zero, we are ready. */
  if (a_value == 0) return;
  /* check that a_size is big enough. If it is not, throw exception. */
  if (bits_for(a_value) > real_size) throw qbitset_overflow();
  /* write the non trivial value in the last element of the container.
     Am I assuming something on the representation of an integer
     number here? I want the LSB to be the last element. */
  data[elements() - 1] = a_value;
  /* calculate if all digits in the last element are significant. */
  size_type mismatch = real_size % storage_type_size;
  /* if the last element is not complete (i.e. not all its digits are
     significant), we must carry some bits. First, save the carry. */
  storage_type carry = data[elements() - 1] >> mismatch;
  /* update the content of the last element, so that all
     non-significant bits are zero. */
  data[elements() - 1] <<= (storage_type_size - mismatch);
  /* if the carry is not trivial, write it into the previous to last
     element (which is always present in this case, due to the bits_for
     check which has already been run. */
  if (carry != 0) data[elements() - 2] = carry;
}

/* *********************************************************************
   | This is the implementation of the assignement operator, which sim-|
   | ply copies all the private counters and duplicate the data area.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Jan 2002         |
   ********************************************************************* */
Qbitset &Qbitset::operator=(const Qbitset &a_bitset) {
  /* copy the number of storage elements and significant bits */
  real_size = a_bitset.real_size;
  num_elements = a_bitset.num_elements;
  /* duplicate the data area. */
  duplicate_data(a_bitset.data);
  /* return the current object */
  return *this;
}

/* *********************************************************************
   | This static function returns the bits inside the bitset as an     |
   | object of type storage_type. Note that if the bit representation  |
   | does not fit in the limited data type, i.e. the conversion is not |
   | possible, an exception of type qbitset_overflow() will be thrown. |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Sep 2001         |
   ********************************************************************* */
const Qbitset::storage_type Qbitset::to_integer_value(void) const {
  /* check if stored data is too much. In this case the conversion
     is not possible and an exception must be thrown. */
  if (real_size > storage_type_size) throw qbitset_overflow();
  /* calculate if all digits in the last element are significant. */
  size_type mismatch = real_size % storage_type_size;
  /* if they are all significant, simply return the last element. */
  if (mismatch == 0) return data[elements() - 1];
  /* if the last element is not complete (i.e. not all its digits are
     significant), we must shift some bits. Note that if we managed
     to reach this point there is surely only one element, hence
     don't look for a carry. */
  return (data[elements() - 1] >> (storage_type_size - mismatch));
}

/* *********************************************************************
   | This output function shows the content of a bitset for debugging  |
   | purposes. It simply pushes on the output stream the values of the |
   | bits, from the MSB to the LSB (showing the boundary of a storage  |
   | element with a space). If the container size is small enough, the |
   | corresponding integer value is shown too.                         |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      09 Jul 2002         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qbitset &a_bitset) {
  /* for each valid index in the container, read the value and print
     either '0' or '1'. Also print ' ' when we cross the boundary of
     a storage element. */
  for (Qbitset::size_type i = 0; i < a_bitset.size(); ++i) {
    if (((i % Qbitset::storage_type_size) == 0) && (i != 0)) os << ' ';
    os << ((a_bitset.read(i)) ? '1' : '0');
  }
  /* Print the integer representation if possible.
     Use exceptions to handle this task. */
  try { os << "  (" << static_cast< ::quantum_message_type>(a_bitset) << ')'; }
  catch (Qbitset::qbitset_overflow) { }
  /* return the modified stream */
  return os;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***

