/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_INTERFACE_BASE_CLASS_
#define __QUANTUM_INTERFACE_BASE_CLASS_

#include <iostream>                    // C++ I/O streams
#include "qexception.h"                // for "quantum" exceptions
#include "qinterface_types.h"          // address and size_type typedefs
#include "qlist.h"                     // for the qubit list class
#include "qbitset.h"                   // for collection of qubits
#include "qmanager_base.h"             // interface for address managers
#include "qop_codes.h"                 // codes for the bytecode protocol

/* *********************************************************************
   | This is an abstract class which defines the properties for an     |
   | interface between the language classes and the quantum device     |
   | driver. It acts as a double channel: the first channel delivers   |
   | bytecode for unitary operations, measurements, initialisations    |
   | and other control instructions to the driver; the second channel  |
   | returns the feedback from the quantum driver to the user code     |
   | for further processing. The output of bytecode must be mediated   |
   | by the submit_* methods, which take care to forward data to the   |
   | quantum device.                                                   |
   | ----------------------------------------------------------------- |
   | The main goal of this class is to define the bytecode protocol,   |
   | i.e. how command codes, command parameters, boolean lists and     |
   | qubit lists are encoded before being sent to a quantum device.    |
   | It does not specify the low level details of communication with   |
   | a quantum device (they must be provided by some derived class,    |
   | this encapsulation will allow multiple quantum devices ....)      |
   | The class also provides functions for inquiring the quantum       |
   | device capabilities and stores some private objects:              |
   | - the quantum address manager (see qmanager_base.h)               |
   | - the permutation list. This is a list which is used for permu-   |
   |   ting qubit addresses before they are sent to the quantum device;|
   |   the list is needed in order to implement a software QSwap.      |
   | ----------------------------------------------------------------- |
   | This class uses some virtual methods which can trigger different  |
   | behaviours in the derived classes; these methods have a dummy     |
   | default implementation, so that those derived classes which do    |
   | not need to use them can simply ignore them: (15 Jul 2002)        |
   | - channel_busy()     before a write to the command channel        |
   | - channel_free()     when the write is done                       |
   | - channel_filled()   when there is data in the command channel    |
   | - channel_blocking() when a blocking instruction is present       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      18 Jul 2002         |
   ********************************************************************* */
class Qinterface_base {
  // allow an output function to inspect the internal status
  friend std::ostream &operator<<(std::ostream &os,
				  const Qinterface_base &an_interface);
public:
  // typedef again the data type for a quantum message/command/...
  typedef ::quantum_message_type message_type;
  // a class for problems during modifications of the permutation list
  _Qexception(invalid_swapgate);
  // a class for problems while reading a measurement result
  _Qexception(measurement_mismatch);
  // a class for problems while reading the result of a location request
  _Qexception(location_mismatch);
public:
  // inhibit the standard constructor
  Qinterface_base();
  // inhibit the copy constructor
  Qinterface_base(const Qinterface_base &an_interface);
  // constructor for the quantum interface (with parameters)
  Qinterface_base(Qaddress_manager_base *the_address_manager);
  // inhibit the assignement operator
  Qinterface_base &operator=(const Qinterface_base &an_interface);
  // non-trivial virtual destructor
  virtual ~Qinterface_base();
public:
  // use this for a generic unitary quantum operation
  void submit_timeslice(message_type a_code, message_type a_parameter,
			const Qubit_list_collection &the_lists);
  // use this for a request to locate the physical addresses of some qubits
  void submit_locate_request(const Qubit_list &a_list);
  // use this for a request to measure some qubits
  void submit_measurement(const Qubit_list &a_list);
  // use this for a request to initialise some qubits
  void submit_initialisation(const Qbitset &a_bitset,
			     const Qubit_list &a_list);
  // use this in order to modify the current permutation
  void submit_permutation(const Qubit_list &swap_list_1,
			  const Qubit_list &swap_list_2);
public:
  // use this to obtain the value of the last measurement (blocking)
  virtual Qbitset    get_measurement_result(Qbitset::size_type bits) = 0;
  // use this to obtain the result of a "locate" request (blocking)
  virtual Qubit_list get_locate_result(Qubit_list::size_type size) = 0;
  // this returns a pointer to ethe address manager
  Qaddress_manager_base *get_manager(void) const { return address_manager; }
protected:
  // this actually writes row data to the device
  virtual void write_bytecode(const message_type &message) = 0;
  // this sets the command channel to the busy state
  virtual void channel_busy(void) { }
  // this sets the command channel to the free state
  virtual void channel_free(void) { }
  // this signals the presence of instructions on the queue
  virtual void channel_filled(void) { }
  // this signals the presence of a blocking instruction on the queue
  virtual void channel_blocking(void) { }
  // this sets the feedback channel to the busy state
  virtual void channel_feed_busy(void) { }
  // this sets the feedback channel to the free state
  virtual void channel_feed_free(void) { }
private:
  // use this for a generic blocking or non-blocking request
  void submit_request(message_type a_code,
		      message_type a_parameter, bool blocking,
		      const Qubit_list_collection *lists = NULL,
		      const Qubit_list *a_list = NULL,
		      const Qbitset *a_bitset = NULL);
  // this encodes a command and sends it to the command channel
  void submit_operation(message_type a_code);
  // this encodes a command parameter and sends it to the command channel
  void submit_parameter(message_type a_parameter);
  // this encodes one register and sends it to the command channel
  void submit_segments(const Qubit_list &segment_list);
  // this encodes some registers and sends them to the command channel
  void submit_segments(const Qubit_list_collection &segment_lists);
  // this encodes a qubit list and sends it to the command channel
  void submit_segment_spec(const Qubit_list &a_list);
  // this encodes a bit sequence and sends it to the command channel
  void submit_booleans(const Qbitset &a_bitset);
  // this returns a pointer to the permutation list
  Qubit_list *get_permutation(void) const { return permutation_list; }
private:
  // a pointer to the address manager for this interface
  Qaddress_manager_base *address_manager;
  // a pointer to the permutation list for this interface
  Qubit_list *permutation_list;
  // this variable is true if the permutation list is trivial
  bool permutation_is_trivial;
};

/* *********************************************************************
   | This global function will be used in order to retrive the single  |
   | instance of a quantum interface (until we will need more than 1)  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 21 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      06 Nov 2002         |
   ********************************************************************* */
Qinterface_base &get_quantum_interface(void);

/* *********************************************************************
   | This method forwards a command to the quantum device. No enco-    |
   | ding is necessary, the value is simply copied.                    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
inline void Qinterface_base::submit_operation(message_type an_operation) {
  write_bytecode(an_operation);
}

/* *********************************************************************
   | This method forwards an optional operation parameter to the device|
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Aug 2001         |
   ********************************************************************* */
inline void Qinterface_base::submit_parameter(message_type a_parameter) {
  write_bytecode(a_parameter);
}

/* *********************************************************************
   | This method is an optimisation of submit_segments which accepts a |
   | single list (obviously, the mixing routine is not run). See the   |
   | more general routine for more details. We can further optimise    |
   | by checking whether the permutation list is trivial or not (no    |
   | need for "translate" in this case).                               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
inline void Qinterface_base::submit_segments(const Qubit_list &segment_list) {
  /* translate the passed list with the permutation list, then send it
     to the quantum device without further encoding. If the permutation
     is trivial, simply submit the passed list. */
  submit_segment_spec(permutation_is_trivial ? segment_list :
		      segment_list.translate(*get_permutation()));
}

/* *********************************************************************
   | This method encodes a request for a generic unitary operation     |
   | (a time slice): it sends the operation code, the optional addi-   |
   | tional parameter and the lists for the qubit addresses. During    |
   | the execution of this method, the command channel is considered   |
   | busy. At the end, the "filled" condition is signalled.            |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      15 Jul 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      16 Feb 2003         |
   ********************************************************************* */
inline void
Qinterface_base::submit_timeslice(message_type a_code,
				  message_type a_parameter, 
				  const Qubit_list_collection &the_lists) {
  submit_request(a_code, a_parameter, false, &the_lists);
}

/* *********************************************************************
   | This method encodes the initialisation of a qubit list to a given |
   | bit string: it sends the operation code, the bit string with the  |
   | value to be written and the qubit list. During the execution of   |
   | this method, the command channel is considered busy. At the end,  |
   | the "filled" condition is signalled.                              |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      15 Jul 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      16 Feb 2003         |
   ********************************************************************* */
inline void Qinterface_base::submit_initialisation(const Qbitset &a_bitset,
						   const Qubit_list &a_list) {
  submit_request(OPCODE_INITIALISATION, 0, false, NULL, &a_list, &a_bitset);
}

/* *********************************************************************
   | This method encodes a request for a measurement. We need to send  |
   | the "operation code" for "measure" and the encoding of the regi-  |
   | ster. During the execution of this method, the command channel is |
   | considered busy. The fact that there is a blocking instruction on |
   | the command queue is signalled at the end.                        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      15 Jul 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      16 Feb 2003         |
   ********************************************************************* */
inline void Qinterface_base::submit_measurement(const Qubit_list &a_list) {
  submit_request(OPCODE_MEASURE, 0, true, NULL, &a_list);
}

/* *********************************************************************
   | This method encodes a request to locate some physical qubits on   |
   | the quantum device (this can sometimes be useful for simulators,  |
   | where we have access to the wavefunction, but we need additional  |
   | information to interpret it). We need to send the "operation code"|
   | for "locate" and the encoding of the register. During the execu-  |
   | tion of this method, the command channel is considered busy. The  |
   | fact that there is a blocking instruction on the command queue is |
   | signalled at the end.                                             |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      16 Feb 2003         |
   ********************************************************************* */
inline void Qinterface_base::submit_locate_request(const Qubit_list &a_list) {
  submit_request(OPCODE_LOCATE, 0, true, NULL, &a_list);
}

#endif  // __QUANTUM_INTERFACE_BASE_CLASS_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
